/* Copyright 2006 aQute SARL 
 * Licensed under the Apache License, Version 2.0, see http://www.apache.org/licenses/LICENSE-2.0 */
package aQute.lib.osgi;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;
import java.util.regex.*;

import aQute.lib.header.*;
import aQute.lib.reporter.*;

public class Processor implements Reporter {
	
	public String DEFAULT_PLUGINS = ""; // "aQute.lib.spring.SpringComponent";
	
	public final static String	DEFAULT_BND_EXTENSION	= ".bnd";
	public final static String	DEFAULT_JAR_EXTENSION	= ".jar";
	public final static String	DEFAULT_BAR_EXTENSION	= ".bar";

	List						errors					= new ArrayList();
	List						warnings				= new ArrayList();
	boolean						pedantic;
	boolean						exceptions;
	String[]					METAPACKAGES			= { "META-INF",
			"OSGI-INF", "OSGI-OPT"						};

	
	List						analyzers;
	
	public void getInfo(Reporter processor) {
		errors.addAll(processor.getErrors());
		warnings.addAll(processor.getWarnings());
		pedantic = processor.isPedantic();
	}

	public void warning(String string) {
		warnings.add(string);
	}

	public void error(String string) {
		errors.add(string);
	}
	public void error(String string, Throwable t) {
		errors.add(string + " : " + t);
		if ( exceptions )
			t.printStackTrace();
	}

	public List getWarnings() {
		return warnings;
	}

	public List getErrors() {
		return errors;
	}

	public Map parseHeader(String value) {
		return parseHeader(value, this);
	}

	/**
	 * Standard OSGi header parser.
	 * 
	 * @param value
	 * @return
	 */
	static public Map parseHeader(String value, Processor logger) {
		return OSGiHeader.parseHeader(value, logger);
	}

	public Map analyzeBundleClasspath(Jar dot, Map bundleClasspath,
			Map contained, Map referred, Map uses) throws IOException {
		Map classSpace = new HashMap();

		if (bundleClasspath.isEmpty()) {
			analyzeJar(dot, "", classSpace, contained, referred, uses);
		} else {
			for (Iterator j = bundleClasspath.keySet().iterator(); j.hasNext();) {
				String path = (String) j.next();
				if (path.equals(".")) {
					analyzeJar(dot, "", classSpace, contained, referred, uses);
					continue;
				}
				//
				// There are 3 cases:
				// - embedded JAR file
				// - directory
				// - error
				//

				Resource resource = dot.getResource(path);
				if (resource != null) {
					try {
						Jar jar = new Jar(path);
						EmbeddedResource.build(jar, resource);
						analyzeJar(jar, "", classSpace, contained, referred,
								uses);
					} catch (Exception e) {
						warning("Invalid bundle classpath entry: " + path +
								" " + e);
					}
				} else {
					if (dot.getDirectories().containsKey(path)) {
						analyzeJar(dot, path, classSpace, contained, referred,
								uses);
					} else {
						warning("No sub JAR or directory " + path);
					}
				}
			}
		}
		return classSpace;
	}

	/**
	 * We traverse through all the classes that we can find and calculate the
	 * contained and referred set and uses. This method ignores the Bundle
	 * classpath.
	 * 
	 * @param jar
	 * @param contained
	 * @param referred
	 * @param uses
	 * @throws IOException
	 */
	private void analyzeJar(Jar jar, String prefix, Map classSpace,
			Map contained, Map referred, Map uses) throws IOException {
		next: for (Iterator r = jar.getResources().keySet().iterator(); r
				.hasNext();) {
			String path = (String) r.next();
			if (path.startsWith(prefix)) {
				String relativePath = path.substring(prefix.length());
				String pack = getPackage(relativePath);

				if (pack != null && !contained.containsKey(pack)) {
					if (!(pack.equals(".") || isMetaData(relativePath))) {

						Map map = new HashMap();
						contained.put(pack, map);
						Resource pinfo = jar.getResource(prefix +
								pack.replace('.', '/') + "/packageinfo");
						if (pinfo != null) {
							InputStream in = pinfo.openInputStream();
							String version = parsePackageInfo(in);
							in.close();
							if (version != null)
								map.put("version", version);
						}
					}
				}

				if (path.endsWith(".class")) {
					Resource resource = jar.getResource(path);
					Clazz clazz;

					try {
						InputStream in = resource.openInputStream();
						clazz = new Clazz(relativePath, in);
						in.close();
					} catch (Throwable e) {
						errors.add("Invalid class file: " + relativePath + " " +
								e.getMessage());
						e.printStackTrace();
						continue next;
					}
					
					String calculatedPath = clazz.getClassName()+".class";
					if ( !calculatedPath.equals(relativePath))
						error("Class in different directory than declared. Path from class name is " + calculatedPath +" but the path in the jar is " + relativePath + " from " + jar);
					
					classSpace.put(relativePath, clazz);
					referred.putAll(clazz.getReferred());

					// Add all the used packages
					// to this package
					Set t = (Set) uses.get(pack);
					if (t == null)
						uses.put(pack, t = new HashSet());
					t.addAll(clazz.getReferred().keySet());
					t.remove(pack);
				}
			}
		}
		for ( Iterator i = getAnalyzers().iterator(); i.hasNext(); ) {
			AnalyzerPlugin analyzer = (AnalyzerPlugin) i.next();
			try {
				analyzer.analyzeJar(this,jar, prefix, classSpace, contained, referred, uses);
			} catch( Exception e  ) {
				error("Plugin Analyzer "+analyzer+" throws exception " + e);
				e.printStackTrace();
			}
		}
	
	}

	/**
	 * Decide if the package is a metadata package.
	 * 
	 * @param pack
	 * @return
	 */
	boolean isMetaData(String pack) {
		for (int i = 0; i < METAPACKAGES.length; i++) {
			if (pack.startsWith(METAPACKAGES[i]))
				return true;
		}
		return false;
	}

	public String getPackage(String clazz) {
		int n = clazz.lastIndexOf('/');
		if (n < 0)
			return ".";
		return clazz.substring(0, n).replace('/', '.');
	}

	static Pattern	packageinfo	= Pattern.compile("version\\s+([0-9.]+).*");

	static String parsePackageInfo(InputStream jar) throws IOException {
		try {
			byte[] buf = EmbeddedResource.collect(jar);
			String line = new String(buf).trim();
			Matcher m = packageinfo.matcher(line);
			if (m.matches()) {
				return m.group(1);
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	static Map removeKeys(Map source, String prefix) {
		Map temp = new TreeMap(source);
		for (Iterator p = temp.keySet().iterator(); p.hasNext();) {
			String pack = (String) p.next();
			if (pack.startsWith(prefix))
				p.remove();
		}
		return temp;
	}

	public void progress(String s) {
		//System.out.println(s);
	}

	public boolean isPedantic() {
		return pedantic;
	}

	public void setPedantic(boolean pedantic) { // System.out.println("Set
												// pedantic: " + pedantic + " "
												// + this );
		this.pedantic = pedantic;
	}

	public static File getFile(File base, String file) {
		File f = new File(file);
		if (f.isAbsolute())
			return f;
		int n;

		f = base.getAbsoluteFile();
		while ((n = file.indexOf('/')) > 0) {
			String first = file.substring(0, n);
			file = file.substring(n + 1);
			if (first.equals(".."))
				f = f.getParentFile();
			else
				f = new File(f, first);
		}
		return new File(f, file).getAbsoluteFile();
	}

	public void report(PrintWriter pw) {
		pw.println(getClass().getName());
		Class c = getClass();
		while ( c != Object.class ) {
			Field fields[] = c.getDeclaredFields();
			for(int i=0; i<fields.length; i++ ) {
				Field field = fields[i];
				field.setAccessible(true);
				try {
				format(pw, field.getName(), field.get(this));
				} catch( Exception e ) {
					format(pw, field.getName(), e );
				}
			}
		}
	}

	public void format(PrintWriter pw, String header, Object value) {
		pw.print(header);
		for (int i = header.length(); i < 30; i++)
			pw.print(" ");
		pw.println(value);
	}

	public void add( AnalyzerPlugin plugin ) {
		analyzers.add(plugin);
	}
	
	List getAnalyzers() {
		if ( analyzers != null )
			return analyzers;
		
		String plugins = getProperty("-plugin",DEFAULT_PLUGINS).trim();
		String parts[] = plugins.split("\\s*,\\s*");
		analyzers = new ArrayList(parts.length);
		for ( int i =0; i<parts.length; i++ ) {
			if ( parts[i].length() > 0  && !parts[i].equals("none")) {
				try {
					if ( isPedantic() ) {
						warning("Using plugin " + parts[i]);
					}
					Class c = getClass().getClassLoader().loadClass(parts[i]);
					AnalyzerPlugin analyzer = (AnalyzerPlugin) c.newInstance();
					add(analyzer);
				} catch( Exception e ) {
					error("Problem loading the plugin: " + parts[i] + " exception: " + e );
				}
			}
		}
		return analyzers;
	}
	
	protected String getProperty(String key, String deflt )  { return deflt; }
}
