/*
 * Decompiled with CFR 0.152.
 */
package org.hl7.fhir.utilities;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.net.ftp.FTPReply;
import org.hl7.fhir.exceptions.FHIRException;
import org.hl7.fhir.utilities.filesystem.ManagedFileAccess;
import org.hl7.fhir.utilities.settings.FhirSettings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FTPClient {
    private static final Logger logger = LoggerFactory.getLogger(FTPClient.class);
    private final org.apache.commons.net.ftp.FTPClient clientImpl;
    private long createRemotePathIfNotExistsNanos;
    private long storeFileTimeNanos;
    private long deleteFileTimeNanos;
    private final String server;
    private final String path;
    private String resolvedPath = null;
    private final String user;
    private final String password;
    private final int port;
    private final String remoteSeparator;

    public FTPClient(String server, String path, String user, String password) {
        this(server, -1, path, user, password);
    }

    protected FTPClient(String server, int port, String path, String user, String password) {
        this.server = server;
        this.port = port;
        this.remoteSeparator = "/";
        this.path = this.buildPath(path);
        this.user = user;
        this.password = password;
        this.clientImpl = new org.apache.commons.net.ftp.FTPClient();
    }

    private String buildPath(String path) {
        if (path.length() == 0) {
            return "";
        }
        if (path.endsWith(this.remoteSeparator)) {
            return path;
        }
        return path + this.remoteSeparator;
    }

    public void connect() throws IOException {
        if (FhirSettings.isProhibitNetworkAccess()) {
            throw new FHIRException("Network Access is prohibited in this context");
        }
        if (this.port != -1) {
            logger.debug("Connecting to : " + this.server + ":" + this.port);
            this.clientImpl.connect(this.server, this.port);
            logger.debug("Connected");
        } else {
            logger.debug("Connecting to : " + this.server);
            this.clientImpl.connect(this.server);
            logger.debug("Connected");
        }
        this.clientImpl.login(this.user, this.password);
        this.throwExceptionForNegativeCompletion("FTP server could not connect.", true);
        this.resetTimers();
        this.clientImpl.setFileType(2);
        this.clientImpl.enterLocalPassiveMode();
        logger.debug("Setting initial working directory: " + this.clientImpl.printWorkingDirectory());
        this.clientImpl.changeWorkingDirectory(this.path);
        this.throwExceptionForNegativeCompletion("FTP server could not establish default working directory", true);
        logger.debug("Set initial working directory.");
        logger.debug("Resolving remote resolved path.");
        this.resolvedPath = this.clientImpl.printWorkingDirectory();
        logger.debug("Resolved remote resolved path: " + this.resolvedPath);
    }

    public void setBufferSize(int bufferSize) {
        this.clientImpl.setBufferSize(bufferSize);
    }

    public int getBufferSize() {
        return this.clientImpl.getBufferSize();
    }

    private void resetTimers() {
        this.createRemotePathIfNotExistsNanos = 0L;
        this.storeFileTimeNanos = 0L;
        this.deleteFileTimeNanos = 0L;
    }

    public void delete(String path) throws IOException {
        String resolvedPath = this.resolveRemotePath(path);
        logger.debug("Deleting remote file: " + resolvedPath);
        long startTime = System.nanoTime();
        this.clientImpl.deleteFile(resolvedPath);
        this.deleteFileTimeNanos += System.nanoTime() - startTime;
        this.throwExceptionForNegativeCompletion("Error deleting file.", false);
        logger.debug("Deleted remote file: " + resolvedPath);
    }

    protected void createRemotePathIfNotExists(String filePath) throws IOException {
        long startTime = System.nanoTime();
        String[] subPath = filePath.split(this.remoteSeparator);
        try {
            for (int i = 0; i < subPath.length - 1; ++i) {
                boolean exists;
                if (subPath[i].isEmpty() || (exists = this.clientImpl.changeWorkingDirectory(subPath[i]))) continue;
                logger.debug("Creating non-existent directory: " + this.clientImpl.printWorkingDirectory() + this.remoteSeparator + subPath[i] + " Creating");
                this.clientImpl.makeDirectory(subPath[i]);
                this.throwExceptionForNegativeCompletion("Creating directory:", true);
                logger.debug("Created directory: " + subPath[i]);
                logger.debug("Changing to created directory: " + subPath[i]);
                this.clientImpl.changeWorkingDirectory(subPath[i]);
                this.throwExceptionForNegativeCompletion("Changing to directory:", true);
                logger.debug("Changed to directory: " + subPath[i]);
            }
        }
        catch (IOException e) {
            throw new IOException("Error creating remote path: " + filePath, e);
        }
        finally {
            logger.debug("Changing to original directory: " + this.resolvedPath);
            this.clientImpl.changeWorkingDirectory(this.resolvedPath);
            logger.debug("Changed to original directory: " + this.resolvedPath);
        }
        this.createRemotePathIfNotExistsNanos += System.nanoTime() - startTime;
    }

    protected boolean remotePathExists(String path) throws IOException {
        boolean output;
        try {
            output = this.clientImpl.changeWorkingDirectory(path);
        }
        finally {
            this.clientImpl.changeWorkingDirectory(this.resolvedPath);
        }
        return output;
    }

    private String resolveRemotePath(String path) {
        if (path.startsWith(this.remoteSeparator)) {
            throw new IllegalArgumentException("Absolute remote path is not permitted. Path: " + path);
        }
        return String.join((CharSequence)this.remoteSeparator, path.replace(File.separator, this.remoteSeparator));
    }

    public void upload(String source, String path) throws IOException {
        String resolvedPath = this.resolveRemotePath(path);
        logger.debug("Uploading file to remote path: " + resolvedPath);
        this.attemptUpload(source, resolvedPath);
        FTPReplyCodeAndString reply = this.getFTPReplyCodeAndString();
        if (FTPReply.isPositiveCompletion((int)reply.replyCode)) {
            logger.debug("Uploaded file: " + resolvedPath);
            return;
        }
        if (this.possibleDirectoryNotExistsCode(reply)) {
            logger.debug("Uploading failed with reply: " + String.valueOf(reply));
            this.createRemotePathIfNotExists(resolvedPath);
            this.attemptUpload(source, resolvedPath);
            this.throwExceptionForNegativeCompletion("Error uploading file (second attempt).", false);
            logger.debug("Uploaded file after path creation: " + resolvedPath);
            return;
        }
        this.throwExceptionForNegativeCompletion(reply, "Error uploading file.", false);
        logger.debug("Remote file uploaded: " + resolvedPath);
    }

    private boolean possibleDirectoryNotExistsCode(FTPReplyCodeAndString reply) {
        return reply.replyCode == 550 || reply.replyCode == 553;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void attemptUpload(String source, String resolvedPath) throws IOException {
        long startTime = System.nanoTime();
        try (FileInputStream localStream = ManagedFileAccess.inStream(source);){
            this.clientImpl.storeFile(resolvedPath, (InputStream)localStream);
        }
        this.storeFileTimeNanos += System.nanoTime() - startTime;
    }

    private FTPReplyCodeAndString getFTPReplyCodeAndString() {
        int replyCode = this.clientImpl.getReplyCode();
        String replyString = this.clientImpl.getReplyString();
        return new FTPReplyCodeAndString(replyCode, replyString);
    }

    private void throwExceptionForNegativeCompletion(String localErrorMessage, boolean disconnectOnError) throws IOException {
        FTPReplyCodeAndString reply = this.getFTPReplyCodeAndString();
        this.throwExceptionForNegativeCompletion(reply, localErrorMessage, disconnectOnError);
    }

    private void throwExceptionForNegativeCompletion(FTPReplyCodeAndString reply, String localErrorMessage, boolean disconnectOnError) throws IOException {
        if (FTPReply.isPositiveCompletion((int)reply.replyCode)) {
            return;
        }
        if (disconnectOnError) {
            this.clientImpl.disconnect();
        }
        throw new IOException(localErrorMessage + " " + String.valueOf(reply));
    }

    public void disconnect() throws IOException {
        this.clientImpl.disconnect();
    }

    public static void main(String[] args) throws IOException, FHIRException {
        FTPClient ftp = new FTPClient(FTPClient.getNamedParam(args, "-upload-server"), FTPClient.getNamedParam(args, "-upload-path"), FTPClient.getNamedParam(args, "-upload-user"), FTPClient.getNamedParam(args, "-upload-password"));
        ftp.connect();
        ftp.upload("/Users/grahamegrieve/temp/test.xml", "testing/test.xml");
        ftp.delete("testing/test.xml");
        ftp.disconnect();
    }

    private static String getNamedParam(String[] args, String param) {
        boolean found = false;
        for (String a : args) {
            if (found) {
                return a;
            }
            if (!a.equals(param)) continue;
            found = true;
        }
        return null;
    }

    public long getCreateRemotePathIfNotExistsNanos() {
        return this.createRemotePathIfNotExistsNanos;
    }

    public long getStoreFileTimeNanos() {
        return this.storeFileTimeNanos;
    }

    public long getDeleteFileTimeNanos() {
        return this.deleteFileTimeNanos;
    }

    public String getServer() {
        return this.server;
    }

    public String getPath() {
        return this.path;
    }

    public String getUser() {
        return this.user;
    }

    public String getPassword() {
        return this.password;
    }

    public int getPort() {
        return this.port;
    }

    public String getRemoteSeparator() {
        return this.remoteSeparator;
    }

    private class FTPReplyCodeAndString {
        private final int replyCode;
        private final String replyString;

        public String toString() {
            return "Reply code: " + this.replyCode + " Message: " + this.replyString;
        }

        protected FTPReplyCodeAndString(int replyCode, String replyString) {
            this.replyCode = replyCode;
            this.replyString = replyString;
        }

        public int getReplyCode() {
            return this.replyCode;
        }

        public String getReplyString() {
            return this.replyString;
        }
    }
}

