/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util.cache;

import java.lang.ref.WeakReference;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

public class ThreadedLRUCacheStrategy<K, V>
implements Map<K, V> {
    private static final Object NULL_ITEM = new Object();
    private final long cleanupDelayMillis;
    private final int capacity;
    private final ConcurrentMap<Object, Node<K>> cache;
    private final AtomicBoolean cleanupScheduled = new AtomicBoolean(false);
    private static final ScheduledExecutorService scheduler = Executors.newSingleThreadScheduledExecutor();

    public ThreadedLRUCacheStrategy(int capacity, int cleanupDelayMillis) {
        if (capacity < 1) {
            throw new IllegalArgumentException("Capacity must be at least 1.");
        }
        if (cleanupDelayMillis < 10) {
            throw new IllegalArgumentException("cleanupDelayMillis must be at least 10 milliseconds.");
        }
        this.capacity = capacity;
        this.cache = new ConcurrentHashMap<Object, Node<K>>(capacity);
        this.cleanupDelayMillis = cleanupDelayMillis;
        this.schedulePurgeTask();
    }

    private void schedulePurgeTask() {
        WeakReference cacheRef = new WeakReference(this);
        PurgeTask purgeTask = new PurgeTask(cacheRef);
        scheduler.scheduleAtFixedRate(purgeTask, this.cleanupDelayMillis, this.cleanupDelayMillis, TimeUnit.MILLISECONDS);
    }

    private void cleanup() {
        int size = this.cache.size();
        if (size > this.capacity) {
            int nodesToRemove = size - this.capacity;
            Node[] nodes = this.cache.values().toArray(new Node[0]);
            Arrays.sort(nodes, Comparator.comparingLong(node -> node.timestamp));
            for (int i = 0; i < nodesToRemove; ++i) {
                Node node2 = nodes[i];
                this.cache.remove(this.toCacheItem(node2.key), node2);
            }
            this.cleanupScheduled.set(false);
            if (this.cache.size() > this.capacity) {
                this.scheduleImmediateCleanup();
            }
        }
    }

    private void scheduleImmediateCleanup() {
        if (this.cleanupScheduled.compareAndSet(false, true)) {
            scheduler.schedule(this::cleanup, this.cleanupDelayMillis, TimeUnit.MILLISECONDS);
        }
    }

    @Override
    public V get(Object key) {
        Object cacheKey = this.toCacheItem(key);
        Node node = (Node)this.cache.get(cacheKey);
        if (node != null) {
            node.updateTimestamp();
            return (V)this.fromCacheItem(node.value);
        }
        return null;
    }

    @Override
    public V put(K key, V value) {
        Object cacheValue;
        Node<K> newNode;
        Object cacheKey = this.toCacheItem(key);
        Node<K> oldNode = this.cache.put(cacheKey, newNode = new Node<K>(key, cacheValue = this.toCacheItem(value)));
        if (oldNode != null) {
            newNode.updateTimestamp();
            return (V)this.fromCacheItem(oldNode.value);
        }
        if (this.size() > this.capacity) {
            this.scheduleImmediateCleanup();
        }
        return null;
    }

    @Override
    public void putAll(Map<? extends K, ? extends V> m) {
        for (Map.Entry<K, V> entry : m.entrySet()) {
            this.put(entry.getKey(), entry.getValue());
        }
    }

    @Override
    public boolean isEmpty() {
        return this.cache.isEmpty();
    }

    @Override
    public V remove(Object key) {
        Object cacheKey = this.toCacheItem(key);
        Node node = (Node)this.cache.remove(cacheKey);
        if (node != null) {
            return (V)this.fromCacheItem(node.value);
        }
        return null;
    }

    @Override
    public void clear() {
        this.cache.clear();
    }

    @Override
    public int size() {
        return this.cache.size();
    }

    @Override
    public boolean containsKey(Object key) {
        return this.cache.containsKey(this.toCacheItem(key));
    }

    @Override
    public boolean containsValue(Object value) {
        Object cacheValue = this.toCacheItem(value);
        for (Node node : this.cache.values()) {
            if (!Objects.equals(node.value, cacheValue)) continue;
            return true;
        }
        return false;
    }

    @Override
    public Set<Map.Entry<K, V>> entrySet() {
        ConcurrentHashMap.KeySetView entrySet = ConcurrentHashMap.newKeySet();
        for (Node node : this.cache.values()) {
            entrySet.add(new AbstractMap.SimpleEntry(this.fromCacheItem(node.key), this.fromCacheItem(node.value)));
        }
        return Collections.unmodifiableSet(entrySet);
    }

    @Override
    public Set<K> keySet() {
        ConcurrentHashMap.KeySetView keySet = ConcurrentHashMap.newKeySet();
        for (Node node : this.cache.values()) {
            keySet.add(this.fromCacheItem(node.key));
        }
        return Collections.unmodifiableSet(keySet);
    }

    @Override
    public Collection<V> values() {
        ArrayList values = new ArrayList();
        for (Node node : this.cache.values()) {
            values.add(this.fromCacheItem(node.value));
        }
        return Collections.unmodifiableCollection(values);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof Map)) {
            return false;
        }
        Map other = (Map)o;
        return this.entrySet().equals(other.entrySet());
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        for (Node node : this.cache.values()) {
            Object key = this.fromCacheItem(node.key);
            Object value = this.fromCacheItem(node.value);
            hashCode = 31 * hashCode + (key == null ? 0 : key.hashCode());
            hashCode = 31 * hashCode + (value == null ? 0 : value.hashCode());
        }
        return hashCode;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        Iterator<Map.Entry<K, V>> it = this.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<K, V> entry = it.next();
            sb.append(entry.getKey()).append("=").append(entry.getValue());
            if (!it.hasNext()) continue;
            sb.append(", ");
        }
        sb.append("}");
        return sb.toString();
    }

    private Object toCacheItem(Object item) {
        return item == null ? NULL_ITEM : item;
    }

    private <T> T fromCacheItem(Object cacheItem) {
        return (T)(cacheItem == NULL_ITEM ? null : cacheItem);
    }

    public static void shutdown() {
        scheduler.shutdown();
        try {
            if (!scheduler.awaitTermination(5L, TimeUnit.SECONDS)) {
                scheduler.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private static class PurgeTask<K, V>
    implements Runnable {
        private final WeakReference<ThreadedLRUCacheStrategy<K, V>> cacheRef;

        PurgeTask(WeakReference<ThreadedLRUCacheStrategy<K, V>> cacheRef) {
            this.cacheRef = cacheRef;
        }

        @Override
        public void run() {
            ThreadedLRUCacheStrategy cache = (ThreadedLRUCacheStrategy)this.cacheRef.get();
            if (cache != null) {
                cache.cleanup();
            }
        }
    }

    private static class Node<K> {
        final K key;
        volatile Object value;
        volatile long timestamp;

        Node(K key, Object value) {
            this.key = key;
            this.value = value;
            this.timestamp = System.nanoTime();
        }

        void updateTimestamp() {
            this.timestamp = System.nanoTime();
        }
    }
}

