/*
 * Decompiled with CFR 0.152.
 */
package com.facebook.cache.disk;

import com.facebook.binaryresource.BinaryResource;
import com.facebook.binaryresource.FileBinaryResource;
import com.facebook.cache.common.CacheErrorLogger;
import com.facebook.cache.common.WriterCallback;
import com.facebook.cache.disk.DiskStorage;
import com.facebook.common.file.FileTree;
import com.facebook.common.file.FileTreeVisitor;
import com.facebook.common.file.FileUtils;
import com.facebook.common.internal.CountingOutputStream;
import com.facebook.common.internal.Lists;
import com.facebook.common.internal.Preconditions;
import com.facebook.common.internal.VisibleForTesting;
import com.facebook.common.time.Clock;
import com.facebook.common.time.SystemClock;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

public class DefaultDiskStorage
implements DiskStorage {
    private static final Class<?> TAG = DefaultDiskStorage.class;
    private static final String CONTENT_FILE_EXTENSION = ".cnt";
    private static final String TEMP_FILE_EXTENSION = ".tmp";
    private static final String DEFAULT_DISK_STORAGE_VERSION_PREFIX = "v2";
    private static final int SHARDING_BUCKET_COUNT = 100;
    static final long TEMP_FILE_LIFETIME_MS = TimeUnit.MINUTES.toMillis(30L);
    private final File mRootDirectory;
    private final File mVersionDirectory;
    private final CacheErrorLogger mCacheErrorLogger;
    private final Clock mClock;

    public DefaultDiskStorage(File rootDirectory, int version, CacheErrorLogger cacheErrorLogger) {
        Preconditions.checkNotNull((Object)rootDirectory);
        this.mRootDirectory = rootDirectory;
        this.mVersionDirectory = new File(this.mRootDirectory, DefaultDiskStorage.getVersionSubdirectoryName(version));
        this.mCacheErrorLogger = cacheErrorLogger;
        this.recreateDirectoryIfVersionChanges();
        this.mClock = SystemClock.get();
    }

    @VisibleForTesting
    static String getVersionSubdirectoryName(int version) {
        return String.format((Locale)null, "%s.ols%d.%d", DEFAULT_DISK_STORAGE_VERSION_PREFIX, 100, version);
    }

    @Override
    public boolean isEnabled() {
        return true;
    }

    private void recreateDirectoryIfVersionChanges() {
        boolean recreateBase = false;
        if (!this.mRootDirectory.exists()) {
            recreateBase = true;
        } else if (!this.mVersionDirectory.exists()) {
            recreateBase = true;
            FileTree.deleteRecursively((File)this.mRootDirectory);
        }
        if (recreateBase) {
            try {
                FileUtils.mkdirs((File)this.mVersionDirectory);
            }
            catch (FileUtils.CreateDirectoryException e) {
                this.mCacheErrorLogger.logError(CacheErrorLogger.CacheErrorCategory.WRITE_CREATE_DIR, TAG, "version directory could not be created: " + this.mVersionDirectory, null);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void updateResource(String resourceId, BinaryResource resource, WriterCallback callback, Object debugInfo) throws IOException {
        FileBinaryResource fileBinaryResource = (FileBinaryResource)resource;
        File file = fileBinaryResource.getFile();
        FileOutputStream fileStream = null;
        try {
            fileStream = new FileOutputStream(file);
        }
        catch (FileNotFoundException fne) {
            this.mCacheErrorLogger.logError(CacheErrorLogger.CacheErrorCategory.WRITE_UPDATE_FILE_NOT_FOUND, TAG, "updateResource", fne);
            throw fne;
        }
        long length = -1L;
        try {
            CountingOutputStream countingStream = new CountingOutputStream((OutputStream)fileStream);
            callback.write((OutputStream)countingStream);
            countingStream.flush();
            length = countingStream.getCount();
        }
        finally {
            fileStream.close();
        }
        if (file.length() != length) {
            throw new IncompleteFileException(length, file.length());
        }
    }

    @VisibleForTesting
    File getContentFileFor(String resourceId) {
        FileInfo fileInfo = new FileInfo(FileType.CONTENT, resourceId);
        File parent = this.getSubdirectory(fileInfo.resourceId);
        return fileInfo.toFile(parent);
    }

    private File getSubdirectory(String resourceId) {
        String subdirectory = String.valueOf(Math.abs(resourceId.hashCode() % 100));
        return new File(this.mVersionDirectory, subdirectory);
    }

    @Override
    public void purgeUnexpectedResources() {
        FileTree.walkFileTree((File)this.mRootDirectory, (FileTreeVisitor)new PurgingVisitor());
    }

    private void mkdirs(File directory, String message) throws IOException {
        try {
            FileUtils.mkdirs((File)directory);
        }
        catch (FileUtils.CreateDirectoryException cde) {
            this.mCacheErrorLogger.logError(CacheErrorLogger.CacheErrorCategory.WRITE_CREATE_DIR, TAG, message, cde);
            throw cde;
        }
    }

    @Override
    public FileBinaryResource createTemporary(String resourceId, Object debugInfo) throws IOException {
        FileInfo info = new FileInfo(FileType.TEMP, resourceId);
        File parent = this.getSubdirectory(info.resourceId);
        if (!parent.exists()) {
            this.mkdirs(parent, "createTemporary");
        }
        try {
            File file = info.createTempFile(parent);
            return FileBinaryResource.createOrNull(file);
        }
        catch (IOException ioe) {
            this.mCacheErrorLogger.logError(CacheErrorLogger.CacheErrorCategory.WRITE_CREATE_TEMPFILE, TAG, "createTemporary", ioe);
            throw ioe;
        }
    }

    @Override
    public FileBinaryResource commit(String resourceId, BinaryResource temporary, Object debugInfo) throws IOException {
        FileBinaryResource tempFileResource = (FileBinaryResource)temporary;
        File tempFile = tempFileResource.getFile();
        File targetFile = this.getContentFileFor(resourceId);
        try {
            FileUtils.rename((File)tempFile, (File)targetFile);
        }
        catch (FileUtils.RenameException re) {
            Throwable cause = re.getCause();
            CacheErrorLogger.CacheErrorCategory category = cause == null ? CacheErrorLogger.CacheErrorCategory.WRITE_RENAME_FILE_OTHER : (cause instanceof FileUtils.ParentDirNotFoundException ? CacheErrorLogger.CacheErrorCategory.WRITE_RENAME_FILE_TEMPFILE_PARENT_NOT_FOUND : (cause instanceof FileNotFoundException ? CacheErrorLogger.CacheErrorCategory.WRITE_RENAME_FILE_TEMPFILE_NOT_FOUND : CacheErrorLogger.CacheErrorCategory.WRITE_RENAME_FILE_OTHER));
            this.mCacheErrorLogger.logError(category, TAG, "commit", re);
            throw re;
        }
        if (targetFile.exists()) {
            targetFile.setLastModified(this.mClock.now());
        }
        return FileBinaryResource.createOrNull(targetFile);
    }

    @Override
    public FileBinaryResource getResource(String resourceId, Object debugInfo) {
        File file = this.getContentFileFor(resourceId);
        if (file.exists()) {
            file.setLastModified(this.mClock.now());
            return FileBinaryResource.createOrNull(file);
        }
        return null;
    }

    @Override
    public boolean contains(String resourceId, Object debugInfo) {
        return this.query(resourceId, false);
    }

    @Override
    public boolean touch(String resourceId, Object debugInfo) {
        return this.query(resourceId, true);
    }

    private boolean query(String resourceId, boolean touch) {
        File contentFile = this.getContentFileFor(resourceId);
        boolean exists = contentFile.exists();
        if (touch && exists) {
            contentFile.setLastModified(this.mClock.now());
        }
        return exists;
    }

    @Override
    public long remove(DiskStorage.Entry entry) {
        EntryImpl entryImpl = (EntryImpl)entry;
        FileBinaryResource resource = entryImpl.getResource();
        return this.doRemove(resource.getFile());
    }

    @Override
    public long remove(String resourceId) {
        return this.doRemove(this.getContentFileFor(resourceId));
    }

    private long doRemove(File contentFile) {
        if (!contentFile.exists()) {
            return 0L;
        }
        long fileSize = contentFile.length();
        if (contentFile.delete()) {
            return fileSize;
        }
        return -1L;
    }

    @Override
    public void clearAll() {
        FileTree.deleteContents((File)this.mRootDirectory);
    }

    @Override
    public DiskStorage.DiskDumpInfo getDumpInfo() throws IOException {
        Collection entries = this.getEntries();
        DiskStorage.DiskDumpInfo dumpInfo = new DiskStorage.DiskDumpInfo();
        for (DiskStorage.Entry entry : entries) {
            DiskStorage.DiskDumpInfoEntry infoEntry = this.dumpCacheEntry(entry);
            String type = infoEntry.type;
            if (!dumpInfo.typeCounts.containsKey(type)) {
                dumpInfo.typeCounts.put(type, 0);
            }
            dumpInfo.typeCounts.put(type, dumpInfo.typeCounts.get(type) + 1);
            dumpInfo.entries.add(infoEntry);
        }
        return dumpInfo;
    }

    private DiskStorage.DiskDumpInfoEntry dumpCacheEntry(DiskStorage.Entry entry) throws IOException {
        EntryImpl entryImpl = (EntryImpl)entry;
        String firstBits = "";
        byte[] bytes = entryImpl.getResource().read();
        String type = this.typeOfBytes(bytes);
        if (type.equals("undefined") && bytes.length >= 4) {
            firstBits = String.format((Locale)null, "0x%02X 0x%02X 0x%02X 0x%02X", bytes[0], bytes[1], bytes[2], bytes[3]);
        }
        String path = entryImpl.getResource().getFile().getPath();
        return new DiskStorage.DiskDumpInfoEntry(path, type, entryImpl.getSize(), firstBits);
    }

    private String typeOfBytes(byte[] bytes) {
        if (bytes.length >= 2) {
            if (bytes[0] == -1 && bytes[1] == -40) {
                return "jpg";
            }
            if (bytes[0] == -119 && bytes[1] == 80) {
                return "png";
            }
            if (bytes[0] == 82 && bytes[1] == 73) {
                return "webp";
            }
            if (bytes[0] == 71 && bytes[1] == 73) {
                return "gif";
            }
        }
        return "undefined";
    }

    public List<DiskStorage.Entry> getEntries() throws IOException {
        EntriesCollector collector = new EntriesCollector();
        FileTree.walkFileTree((File)this.mVersionDirectory, (FileTreeVisitor)collector);
        return collector.getEntries();
    }

    private FileInfo getShardFileInfo(File file) {
        FileInfo info = FileInfo.fromFile(file);
        if (info == null) {
            return null;
        }
        File expectedDirectory = this.getSubdirectory(info.resourceId);
        boolean isCorrect = expectedDirectory.equals(file.getParentFile());
        return isCorrect ? info : null;
    }

    private static class FileInfo {
        public final FileType type;
        public final String resourceId;

        private FileInfo(FileType type, String resourceId) {
            this.type = type;
            this.resourceId = resourceId;
        }

        public String toString() {
            return (Object)((Object)this.type) + "(" + this.resourceId + ")";
        }

        public File toFile(File parentDir) {
            return new File(parentDir, this.resourceId + this.type.extension);
        }

        public File createTempFile(File parent) throws IOException {
            File f = File.createTempFile(this.resourceId + ".", DefaultDiskStorage.TEMP_FILE_EXTENSION, parent);
            return f;
        }

        public static FileInfo fromFile(File file) {
            String name = file.getName();
            int pos = name.lastIndexOf(46);
            if (pos <= 0) {
                return null;
            }
            String ext = name.substring(pos);
            FileType type = FileType.fromExtension(ext);
            if (type == null) {
                return null;
            }
            String resourceId = name.substring(0, pos);
            if (type.equals((Object)FileType.TEMP)) {
                int numPos = resourceId.lastIndexOf(46);
                if (numPos <= 0) {
                    return null;
                }
                resourceId = resourceId.substring(0, numPos);
            }
            return new FileInfo(type, resourceId);
        }
    }

    private static enum FileType {
        CONTENT(".cnt"),
        TEMP(".tmp");

        public final String extension;

        private FileType(String extension) {
            this.extension = extension;
        }

        public static FileType fromExtension(String extension) {
            for (FileType ft : FileType.values()) {
                if (!ft.extension.equals(extension)) continue;
                return ft;
            }
            return null;
        }
    }

    @VisibleForTesting
    class EntryImpl
    implements DiskStorage.Entry {
        private final FileBinaryResource resource;
        private long size;
        private long timestamp;

        private EntryImpl(File cachedFile) {
            Preconditions.checkNotNull((Object)cachedFile);
            this.resource = FileBinaryResource.createOrNull(cachedFile);
            this.size = -1L;
            this.timestamp = -1L;
        }

        @Override
        public long getTimestamp() {
            if (this.timestamp < 0L) {
                File cachedFile = this.resource.getFile();
                this.timestamp = cachedFile.lastModified();
            }
            return this.timestamp;
        }

        @Override
        public FileBinaryResource getResource() {
            return this.resource;
        }

        @Override
        public long getSize() {
            if (this.size < 0L) {
                this.size = this.resource.size();
            }
            return this.size;
        }
    }

    private class PurgingVisitor
    implements FileTreeVisitor {
        private boolean insideBaseDirectory;

        private PurgingVisitor() {
        }

        public void preVisitDirectory(File directory) {
            if (!this.insideBaseDirectory && directory.equals(DefaultDiskStorage.this.mVersionDirectory)) {
                this.insideBaseDirectory = true;
            }
        }

        public void visitFile(File file) {
            if (!this.insideBaseDirectory || !this.isExpectedFile(file)) {
                file.delete();
            }
        }

        public void postVisitDirectory(File directory) {
            if (!DefaultDiskStorage.this.mRootDirectory.equals(directory) && !this.insideBaseDirectory) {
                directory.delete();
            }
            if (this.insideBaseDirectory && directory.equals(DefaultDiskStorage.this.mVersionDirectory)) {
                this.insideBaseDirectory = false;
            }
        }

        private boolean isExpectedFile(File file) {
            FileInfo info = DefaultDiskStorage.this.getShardFileInfo(file);
            if (info == null) {
                return false;
            }
            if (info.type == FileType.TEMP) {
                return this.isRecentFile(file);
            }
            Preconditions.checkState((info.type == FileType.CONTENT ? 1 : 0) != 0);
            return true;
        }

        private boolean isRecentFile(File file) {
            return file.lastModified() > DefaultDiskStorage.this.mClock.now() - TEMP_FILE_LIFETIME_MS;
        }
    }

    private class EntriesCollector
    implements FileTreeVisitor {
        private final List<DiskStorage.Entry> result = Lists.newArrayList();

        private EntriesCollector() {
        }

        public void preVisitDirectory(File directory) {
        }

        public void visitFile(File file) {
            FileInfo info = DefaultDiskStorage.this.getShardFileInfo(file);
            if (info != null && info.type == FileType.CONTENT) {
                this.result.add(new EntryImpl(file));
            }
        }

        public void postVisitDirectory(File directory) {
        }

        public List<DiskStorage.Entry> getEntries() {
            return Collections.unmodifiableList(this.result);
        }
    }

    private static class IncompleteFileException
    extends IOException {
        public final long expected;
        public final long actual;

        public IncompleteFileException(long expected, long actual) {
            super("File was not written completely. Expected: " + expected + ", found: " + actual);
            this.expected = expected;
            this.actual = actual;
        }
    }
}

