/*
 * Decompiled with CFR 0.152.
 */
package com.newrelic.agent.service.module;

import com.newrelic.agent.bridge.ManifestUtils;
import com.newrelic.agent.config.AgentConfig;
import com.newrelic.agent.deps.com.google.common.annotations.VisibleForTesting;
import com.newrelic.agent.deps.com.google.common.collect.ImmutableList;
import com.newrelic.agent.deps.com.google.common.util.concurrent.RateLimiter;
import com.newrelic.agent.service.module.EmbeddedJars;
import com.newrelic.agent.service.module.JarData;
import com.newrelic.agent.service.module.JarInfo;
import com.newrelic.agent.service.module.ShaChecksums;
import com.newrelic.api.agent.Logger;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.function.Function;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;
import java.util.logging.Level;

public class JarCollectorServiceProcessor
implements Function<URL, JarData> {
    static final String SHA1_CHECKSUM_KEY = "sha1Checksum";
    static final String SHA512_CHECKSUM_KEY = "sha512Checksum";
    static final String JAR_EXTENSION = ".jar";
    static final String JAR_PROTOCOL = "jar";
    static final String UNKNOWN_VERSION = " ";
    private static final List<String> ATTRIBUTES_TO_COLLECT = ImmutableList.of(Attributes.Name.IMPLEMENTATION_VENDOR.toString(), Attributes.Name.IMPLEMENTATION_VENDOR_ID.toString());
    private final Logger logger;
    private final boolean skipTempJars;
    private final List<String> ignoreJars;
    private final RateLimiter processUrlRateLimiter;
    private static final File TEMP_DIRECTORY = new File(System.getProperty("java.io.tmpdir"));

    public JarCollectorServiceProcessor(Logger logger, AgentConfig agentConfig) {
        this.logger = logger;
        this.skipTempJars = agentConfig.getJarCollectorConfig().skipTempJars();
        if (!this.skipTempJars) {
            logger.log(Level.FINEST, "temporary jars will be transmitted to the host");
        }
        this.ignoreJars = new ArrayList<String>(agentConfig.getIgnoreJars());
        int jarsPerSecond = agentConfig.getJarCollectorConfig().getJarsPerSecond();
        if (jarsPerSecond <= 0) {
            logger.log(Level.INFO, "Jars per second must be greater than 0. Defaulting to {0}.", (Object)10);
            jarsPerSecond = 10;
        }
        this.processUrlRateLimiter = RateLimiter.create(jarsPerSecond);
    }

    @Override
    public JarData apply(URL url) {
        this.processUrlRateLimiter.acquire(1);
        try {
            return this.tryProcessSingleURL(url);
        }
        catch (Throwable t2) {
            this.logger.log(Level.FINE, t2, "Exception processing jar at {0}", (Object)url);
            return null;
        }
    }

    @VisibleForTesting
    JarData tryProcessSingleURL(URL url) throws URISyntaxException {
        if (this.skipTempJars && JarCollectorServiceProcessor.isTempFile(url)) {
            this.logger.log(Level.FINE, "{0} Skipping temp jar file", (Object)url);
            return null;
        }
        if (!url.getFile().endsWith(JAR_EXTENSION)) {
            this.logger.log(Level.FINE, "{0} Skipping file with non-jar extension", (Object)url);
            return null;
        }
        JarInfo jarInfo = this.getJarInfoSafe(url);
        return this.addJarAndVersion(url, jarInfo);
    }

    static boolean isTempFile(URL address) throws URISyntaxException {
        if (!"file".equals(address.getProtocol())) {
            return false;
        }
        return JarCollectorServiceProcessor.isTempFile(new File(address.toURI()));
    }

    static boolean isTempFile(File fileParam) {
        File file = fileParam;
        while (file != null) {
            if (!TEMP_DIRECTORY.equals(file = file.getParentFile())) continue;
            return true;
        }
        return false;
    }

    JarInfo getJarInfoSafe(URL url) {
        JarInfo jarInfo;
        HashMap<String, String> attributes = new HashMap<String, String>();
        try {
            String sha1Checksum = ShaChecksums.computeSha(url);
            attributes.put(SHA1_CHECKSUM_KEY, sha1Checksum);
        }
        catch (Exception ex) {
            this.logger.log(Level.FINE, (Throwable)ex, "{0} Error getting jar file sha1 checksum", (Object)url);
        }
        try {
            String sha512Checksum = ShaChecksums.computeSha512(url);
            attributes.put(SHA512_CHECKSUM_KEY, sha512Checksum);
        }
        catch (Exception ex) {
            this.logger.log(Level.FINE, (Throwable)ex, "{0} Error getting jar file sha512 checksum", (Object)url);
        }
        try {
            jarInfo = this.getJarInfo(url, attributes);
        }
        catch (Exception e) {
            this.logger.log(Level.FINE, (Throwable)e, "{0} Trouble getting version from jar. Adding jar without version.", (Object)url);
            jarInfo = new JarInfo(UNKNOWN_VERSION, attributes);
        }
        return jarInfo;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private JarInfo getJarInfo(URL url, Map<String, String> attributes) throws IOException {
        JarInputStream jarFile;
        block13: {
            JarInfo jarInfo;
            block14: {
                jarFile = EmbeddedJars.getJarInputStream(url);
                Throwable throwable = null;
                try {
                    JarCollectorServiceProcessor.getExtraAttributes(jarFile, attributes);
                    Map<String, String> pom = JarCollectorServiceProcessor.getPom(jarFile);
                    if (pom == null) break block13;
                    attributes.putAll(pom);
                    jarInfo = new JarInfo(pom.get("version"), attributes);
                    if (jarFile == null) return jarInfo;
                    if (throwable == null) break block14;
                }
                catch (Exception ex) {
                    this.logger.log(Level.FINEST, (Throwable)ex, "{0} Exception getting extra attributes or pom.", (Object)url);
                    break block13;
                }
                catch (Throwable throwable3) {
                    throwable = throwable3;
                    throw throwable3;
                }
                catch (Throwable throwable4) {
                    throw throwable4;
                }
                try {
                    jarFile.close();
                    return jarInfo;
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                    return jarInfo;
                }
            }
            jarFile.close();
            return jarInfo;
        }
        String version = JarCollectorServiceProcessor.getVersion(jarFile);
        if (version != null) return new JarInfo(version, attributes);
        version = UNKNOWN_VERSION;
        return new JarInfo(version, attributes);
    }

    private static Map<String, String> getPom(JarInputStream jarFile) throws IOException {
        Properties pom = null;
        JarEntry entry = jarFile.getNextJarEntry();
        while (entry != null) {
            if (entry.getName().startsWith("META-INF/maven") && entry.getName().endsWith("pom.properties")) {
                if (pom != null) {
                    return null;
                }
                Properties props = new Properties();
                props.load(jarFile);
                pom = props;
            }
            entry = jarFile.getNextJarEntry();
        }
        return pom;
    }

    static void getExtraAttributes(JarInputStream jarFile, Map<String, String> map) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return;
        }
        Attributes attributes = manifest.getMainAttributes();
        for (String name : ATTRIBUTES_TO_COLLECT) {
            String value = attributes.getValue(name);
            if (null == value) continue;
            map.put(name, value);
        }
    }

    static String getVersion(JarInputStream jarFile) {
        Manifest manifest = jarFile.getManifest();
        if (manifest == null) {
            return null;
        }
        return ManifestUtils.getVersionFromManifest((Manifest)manifest);
    }

    JarData addJarAndVersion(URL url, JarInfo jarInfo) throws URISyntaxException {
        String jarFile;
        if (jarInfo == null) {
            jarInfo = JarInfo.MISSING;
        }
        if (this.shouldAttemptAdd(jarFile = this.parseJarName(url))) {
            this.logger.log(Level.FINE, "{0} Adding the file {1} with version {2}.", (Object)url, (Object)jarFile, (Object)jarInfo.version);
            return new JarData(jarFile, jarInfo);
        }
        return null;
    }

    String parseJarName(URL url) throws URISyntaxException {
        if ("file".equals(url.getProtocol())) {
            File file = new File(url.toURI());
            return file.getName().trim();
        }
        this.logger.log(Level.FINEST, "{0} Parsing jar file name", (Object)url);
        String path = url.getFile();
        int end = path.lastIndexOf(JAR_EXTENSION);
        if (end > 0) {
            int start = (path = path.substring(0, end)).lastIndexOf(File.separator);
            if (start > -1) {
                return path.substring(start + 1) + JAR_EXTENSION;
            }
            return path + JAR_EXTENSION;
        }
        throw new URISyntaxException(url.getPath(), "Unable to parse the jar file name from a URL");
    }

    private boolean shouldAttemptAdd(String jarFile) {
        return !this.ignoreJars.contains(jarFile);
    }
}

