package com.tietoevry.quarkus.resteasy.problem.postprocessing;

import com.tietoevry.quarkus.resteasy.problem.HttpProblem;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import org.slf4j.LoggerFactory;

/**
 * Container for prioritised list of Problem post-processors. This class is thread-safe.
 */
public final class PostProcessorsRegistry {

    private final List<ProblemPostProcessor> processors = new CopyOnWriteArrayList<>();

    public PostProcessorsRegistry() {
        reset();
    }

    /**
     * Removes all registered post-processors and registers default ones. Used mainly for Quarkus dev mode (live-reload) tests
     * where there's a need to reset registered processors because of config change.
     */
    synchronized public void reset() {
        processors.clear();
        register(new ProblemLogger(LoggerFactory.getLogger("http-problem")));
        register(new ProblemDefaultsProvider());
    }

    synchronized public void register(ProblemPostProcessor processor) {
        processors.add(processor);
        processors.sort(ProblemPostProcessor.DEFAULT_ORDERING);
    }

    /**
     * Applies all registered post-processors on a given Problem, in prioritized order.
     *
     * @param problem Original Problem produced by Exception Mapper
     * @param context Additional info on cause (original exception caught by ExceptionMapper) and HTTP request
     * @return Enhanced version of original Problem
     */
    public HttpProblem applyPostProcessing(HttpProblem problem, ProblemContext context) {
        HttpProblem finalProblem = problem;
        for (ProblemPostProcessor processor : processors) {
            finalProblem = processor.apply(finalProblem, context);
        }
        return finalProblem;
    }

}
