/*
 * Decompiled with CFR 0.152.
 */
package com.yugabyte.ysql;

import com.yugabyte.Driver;
import com.yugabyte.jdbc.PgConnection;
import com.yugabyte.util.GT;
import com.yugabyte.util.HostSpec;
import com.yugabyte.util.PSQLException;
import com.yugabyte.util.PSQLState;
import com.yugabyte.ysql.LoadBalanceProperties;
import com.yugabyte.ysql.LoadBalancer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Logger;

public class LoadBalanceService {
    private static final ConcurrentHashMap<String, NodeInfo> clusterInfoMap = new ConcurrentHashMap();
    private static Connection controlConnection = null;
    protected static final String GET_SERVERS_QUERY = "select * from yb_servers()";
    protected static final Logger LOGGER = Logger.getLogger("com.yugabyte." + LoadBalanceService.class.getName());
    private static long lastRefreshTime;
    private static boolean forceRefreshOnce;
    private static Boolean useHostColumn;

    public static void printHostToConnectionMap() {
        System.out.println("Current load on servers");
        System.out.println("-------------------");
        for (Map.Entry<String, NodeInfo> e : clusterInfoMap.entrySet()) {
            System.out.println(e.getKey() + " - " + e.getValue().connectionCount);
        }
    }

    static synchronized void clear() {
        LOGGER.warning("Clearing LoadBalanceService state for testing purposes");
        clusterInfoMap.clear();
        controlConnection = null;
        lastRefreshTime = 0L;
        forceRefreshOnce = false;
        useHostColumn = null;
    }

    static long getLastRefreshTime() {
        return lastRefreshTime;
    }

    private static boolean needsRefresh(long refreshInterval) {
        if (forceRefreshOnce) {
            LOGGER.fine("forceRefreshOnce is set to true");
            return true;
        }
        long elapsed = (System.currentTimeMillis() - lastRefreshTime) / 1000L;
        if (elapsed >= refreshInterval) {
            LOGGER.fine("Needs refresh as list of servers may be stale or being fetched for the first time, refreshInterval: " + refreshInterval);
            return true;
        }
        LOGGER.fine("Refresh not required, refreshInterval: " + refreshInterval);
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static synchronized boolean refresh(Connection conn, long refreshInterval) throws SQLException {
        forceRefreshOnce = false;
        Statement st = conn.createStatement();
        LOGGER.fine("Executing query: select * from yb_servers() to fetch list of servers");
        ResultSet rs = st.executeQuery(GET_SERVERS_QUERY);
        InetAddress hostConnectedInetAddress = LoadBalanceService.getConnectedInetAddress(conn);
        boolean publicIPsGivenForAll = true;
        while (rs.next()) {
            InetAddress publicHostInetAddr;
            InetAddress hostInetAddr;
            NodeInfo nodeInfo;
            String host = rs.getString("host");
            LOGGER.finest("Received entry for host " + host);
            String publicHost = rs.getString("public_ip");
            publicHost = publicHost == null ? "" : publicHost;
            String port = rs.getString("port");
            String cloud = rs.getString("cloud");
            String region = rs.getString("region");
            String zone = rs.getString("zone");
            NodeInfo nodeInfo2 = nodeInfo = clusterInfoMap.containsKey(host) ? clusterInfoMap.get(host) : new NodeInfo();
            synchronized (nodeInfo2) {
                nodeInfo.host = host;
                nodeInfo.publicIP = publicHost;
                publicIPsGivenForAll = !publicHost.isEmpty();
                nodeInfo.placement = new CloudPlacement(cloud, region, zone);
                try {
                    nodeInfo.port = Integer.valueOf(port);
                }
                catch (NumberFormatException nfe) {
                    LOGGER.warning("Could not parse port " + port + " for host " + host + ", using 5433 instead.");
                    nodeInfo.port = 5433;
                }
                long failedHostTTL = Long.getLong("failed-host-reconnect-delay-secs", 5L);
                if (nodeInfo.isDown) {
                    if (System.currentTimeMillis() - nodeInfo.isDownSince > failedHostTTL * 1000L) {
                        LOGGER.fine("Marking " + nodeInfo.host + " as UP since failed-host-reconnect-delay-secs (" + failedHostTTL + "s) has elapsed");
                        nodeInfo.isDown = false;
                    } else {
                        LOGGER.fine("Keeping " + nodeInfo.host + " as DOWN since failed-host-reconnect-delay-secs (" + failedHostTTL + "s) has not elapsed");
                    }
                }
            }
            clusterInfoMap.putIfAbsent(host, nodeInfo);
            try {
                hostInetAddr = InetAddress.getByName(host);
            }
            catch (UnknownHostException e) {
                hostInetAddr = null;
            }
            try {
                publicHostInetAddr = !publicHost.isEmpty() ? InetAddress.getByName(publicHost) : null;
            }
            catch (UnknownHostException e) {
                publicHostInetAddr = null;
            }
            if (useHostColumn != null) continue;
            if (hostConnectedInetAddress.equals(hostInetAddr)) {
                useHostColumn = Boolean.TRUE;
                continue;
            }
            if (!hostConnectedInetAddress.equals(publicHostInetAddr)) continue;
            useHostColumn = Boolean.FALSE;
        }
        if (useHostColumn != null && !useHostColumn.booleanValue() || useHostColumn == null && publicIPsGivenForAll) {
            LOGGER.info("Will be using publicIPs for establishing connections");
            Enumeration<String> hosts = clusterInfoMap.keys();
            while (hosts.hasMoreElements()) {
                NodeInfo info = clusterInfoMap.get(hosts.nextElement());
                clusterInfoMap.put(info.publicIP, info);
                clusterInfoMap.remove(info.host);
            }
        } else if (useHostColumn == null) {
            LOGGER.warning("Unable to identify set of addresses to use for establishing connections. Using private addresses.");
        }
        lastRefreshTime = System.currentTimeMillis();
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void markAsFailed(String host) {
        NodeInfo info = clusterInfoMap.get(host);
        if (info == null) {
            return;
        }
        NodeInfo nodeInfo = info;
        synchronized (nodeInfo) {
            String previous = info.isDown ? "DOWN" : "UP";
            info.isDown = true;
            info.isDownSince = System.currentTimeMillis();
            info.connectionCount = 0;
            LOGGER.info("Marked " + host + " as DOWN (was " + previous + " earlier)");
        }
    }

    static int getLoad(String host) {
        NodeInfo info = clusterInfoMap.get(host);
        return info == null ? 0 : info.connectionCount;
    }

    static ArrayList<String> getAllEligibleHosts(LoadBalancer policy) {
        ArrayList<String> list = new ArrayList<String>();
        Set<Map.Entry<String, NodeInfo>> set = clusterInfoMap.entrySet();
        for (Map.Entry<String, NodeInfo> e : set) {
            if (policy.isHostEligible(e)) {
                list.add(e.getKey());
                continue;
            }
            LOGGER.finest("Skipping " + e + " because it is not eligible.");
        }
        return list;
    }

    private static ArrayList<String> getAllAvailableHosts(ArrayList<String> attempted) {
        ArrayList<String> list = new ArrayList<String>();
        Enumeration<String> hosts = clusterInfoMap.keys();
        while (hosts.hasMoreElements()) {
            String h = hosts.nextElement();
            if (attempted.contains(h) || clusterInfoMap.get(h).isDown) continue;
            list.add(h);
        }
        return list;
    }

    private static int getPort(String host) {
        NodeInfo info = clusterInfoMap.get(host);
        return info != null ? info.port : 5433;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static boolean incrementConnectionCount(String host) {
        NodeInfo info = clusterInfoMap.get(host);
        if (info != null) {
            NodeInfo nodeInfo = info;
            synchronized (nodeInfo) {
                if (info.connectionCount < 0) {
                    info.connectionCount = 0;
                    LOGGER.fine("Resetting connection count for " + host + " to zero from " + info.connectionCount);
                }
                NodeInfo nodeInfo2 = info;
                nodeInfo2.connectionCount = nodeInfo2.connectionCount + 1;
                return true;
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean decrementConnectionCount(String host) {
        NodeInfo info = clusterInfoMap.get(host);
        if (info != null) {
            NodeInfo nodeInfo = info;
            synchronized (nodeInfo) {
                NodeInfo nodeInfo2 = info;
                nodeInfo2.connectionCount = nodeInfo2.connectionCount - 1;
                LOGGER.fine("Decremented connection count for " + host + " by one: " + info.connectionCount);
                if (info.connectionCount < 0) {
                    info.connectionCount = 0;
                    LOGGER.fine("Resetting connection count for " + host + " to zero.");
                }
                return true;
            }
        }
        return false;
    }

    public static Connection getConnection(String url, Properties properties, String user, String database, LoadBalanceProperties lbProperties, ArrayList<String> timedOutHosts) {
        if (lbProperties.hasLoadBalance()) {
            Connection conn = LoadBalanceService.getConnection(lbProperties, properties, user, database, timedOutHosts);
            if (conn != null) {
                return conn;
            }
            LOGGER.warning("Failed to apply load balance. Trying normal connection");
            properties.setProperty("PGHOST", lbProperties.getOriginalProperties().getProperty("PGHOST"));
            properties.setProperty("PGPORT", lbProperties.getOriginalProperties().getProperty("PGPORT"));
        }
        return null;
    }

    private static Connection getConnection(LoadBalanceProperties loadBalanceProperties, Properties props, String user, String dbName, ArrayList<String> timedOutHosts) {
        LoadBalancer lb = loadBalanceProperties.getAppropriateLoadBalancer();
        String url = loadBalanceProperties.getStrippedURL();
        if (!LoadBalanceService.checkAndRefresh(loadBalanceProperties, lb, user, dbName)) {
            LOGGER.fine("Attempt to refresh info from yb_servers() failed");
            return null;
        }
        ArrayList<String> failedHosts = new ArrayList<String>();
        String chosenHost = lb.getLeastLoadedServer(true, failedHosts, timedOutHosts);
        SQLException firstException = null;
        while (chosenHost != null) {
            try {
                props.setProperty("PGHOST", chosenHost);
                props.setProperty("PGPORT", String.valueOf(LoadBalanceService.getPort(chosenHost)));
                if (timedOutHosts != null) {
                    timedOutHosts.add(chosenHost);
                }
                PgConnection newConnection = new PgConnection(Driver.hostSpecs(props), user, dbName, props, url);
                newConnection.setLoadBalancer(lb);
                LOGGER.fine("Created connection to " + chosenHost);
                return newConnection;
            }
            catch (SQLException ex) {
                forceRefreshOnce = true;
                failedHosts.add(chosenHost);
                LoadBalanceService.decrementConnectionCount(chosenHost);
                if (PSQLState.CONNECTION_UNABLE_TO_CONNECT.getState().equals(ex.getSQLState())) {
                    if (firstException == null) {
                        firstException = ex;
                    }
                    LOGGER.fine("couldn't connect to " + chosenHost + ", adding it to failed host list");
                    LoadBalanceService.markAsFailed(chosenHost);
                } else {
                    LOGGER.warning("got exception " + ex.getMessage() + " while connecting to " + chosenHost);
                }
            }
            catch (Throwable e) {
                LOGGER.fine("Received Throwable: " + e);
                LoadBalanceService.decrementConnectionCount(chosenHost);
                throw e;
            }
            chosenHost = lb.getLeastLoadedServer(false, failedHosts, timedOutHosts);
        }
        LOGGER.fine("No host could be chosen");
        return null;
    }

    private static synchronized boolean checkAndRefresh(LoadBalanceProperties loadBalanceProperties, LoadBalancer lb, String user, String dbName) {
        if (LoadBalanceService.needsRefresh(lb.getRefreshListSeconds())) {
            Properties props = loadBalanceProperties.getOriginalProperties();
            String url = loadBalanceProperties.getStrippedURL();
            HostSpec[] hspec = Driver.hostSpecs(props);
            ArrayList<String> hosts = LoadBalanceService.getAllAvailableHosts(new ArrayList<String>());
            while (true) {
                boolean refreshFailed = false;
                try {
                    if (controlConnection == null || controlConnection.isClosed()) {
                        controlConnection = new PgConnection(hspec, user, dbName, props, url);
                    }
                    try {
                        LoadBalanceService.refresh(controlConnection, lb.getRefreshListSeconds());
                    }
                    catch (SQLException e) {
                        refreshFailed = true;
                        throw e;
                    }
                }
                catch (SQLException ex) {
                    if (refreshFailed) {
                        LOGGER.fine("Exception while refreshing: " + ex + ", " + ex.getSQLState());
                        String failed = ((PgConnection)controlConnection).getQueryExecutor().getHostSpec().getHost();
                        LoadBalanceService.markAsFailed(failed);
                    } else {
                        String msg = hspec.length > 1 ? " and others" : "";
                        LOGGER.fine("Exception while creating control connection to " + hspec[0].getHost() + msg + ": " + ex + ", " + ex.getSQLState());
                        for (HostSpec h : hspec) {
                            hosts.remove(h.getHost());
                        }
                    }
                    if (PSQLState.UNDEFINED_FUNCTION.getState().equals(ex.getSQLState())) {
                        LOGGER.warning("Received UNDEFINED_FUNCTION for yb_servers() (SQLState=42883). You may be using an older version of YugabyteDB, consider upgrading it.");
                        return false;
                    }
                    if (hosts.isEmpty()) {
                        LOGGER.fine("Failed to establish control connection to available servers");
                        return false;
                    }
                    if (!refreshFailed) {
                        HostSpec hs = new HostSpec(hosts.get(0), LoadBalanceService.getPort(hosts.get(0)), loadBalanceProperties.getOriginalProperties().getProperty("localSocketAddress"));
                        hspec = new HostSpec[]{hs};
                    }
                    controlConnection = null;
                    continue;
                }
                break;
            }
        }
        return true;
    }

    private static InetAddress getConnectedInetAddress(Connection conn) throws SQLException {
        InetAddress hostConnectedInetAddr;
        String hostConnectedTo = ((PgConnection)conn).getQueryExecutor().getHostSpec().getHost();
        boolean isIpv6Addresses = hostConnectedTo.contains(":");
        if (isIpv6Addresses) {
            hostConnectedTo = hostConnectedTo.replace("[", "").replace("]", "");
        }
        try {
            hostConnectedInetAddr = InetAddress.getByName(hostConnectedTo);
        }
        catch (UnknownHostException e) {
            throw new PSQLException(GT.tr("Unexpected UnknownHostException for ${0} ", hostConnectedTo), PSQLState.UNKNOWN_STATE, (Throwable)e);
        }
        return hostConnectedInetAddr;
    }

    static {
        forceRefreshOnce = false;
        useHostColumn = null;
    }

    static class CloudPlacement {
        private final String cloud;
        private final String region;
        private final String zone;

        CloudPlacement(String cloud, String region, String zone) {
            this.cloud = cloud;
            this.region = region;
            this.zone = zone;
        }

        public boolean isContainedIn(Set<CloudPlacement> set) {
            if (this.zone.equals("*")) {
                for (CloudPlacement cp : set) {
                    if (!cp.cloud.equalsIgnoreCase(this.cloud) || !cp.region.equalsIgnoreCase(this.region)) continue;
                    return true;
                }
            } else {
                for (CloudPlacement cp : set) {
                    if (!cp.cloud.equalsIgnoreCase(this.cloud) || !cp.region.equalsIgnoreCase(this.region) || !cp.zone.equalsIgnoreCase(this.zone) && !cp.zone.equals("*")) continue;
                    return true;
                }
            }
            return false;
        }

        public int hashCode() {
            return this.cloud.hashCode() ^ this.region.hashCode() ^ this.zone.hashCode();
        }

        public boolean equals(Object other) {
            boolean equal = false;
            LOGGER.fine("equals called for this: " + this + " and other = " + other);
            if (other instanceof CloudPlacement) {
                CloudPlacement o = (CloudPlacement)other;
                equal = this.cloud.equalsIgnoreCase(o.cloud) && this.region.equalsIgnoreCase(o.region) && this.zone.equalsIgnoreCase(o.zone);
            }
            LOGGER.fine("equals returning: " + equal);
            return equal;
        }

        public String toString() {
            return "CloudPlacement: " + this.cloud + "." + this.region + "." + this.zone;
        }
    }

    static class NodeInfo {
        private String host;
        private int port;
        private CloudPlacement placement;
        private String publicIP;
        private int connectionCount;
        private boolean isDown;
        private long isDownSince;

        NodeInfo() {
        }

        public boolean isDown() {
            return this.isDown;
        }

        public String getHost() {
            return this.host;
        }

        public String getPublicIP() {
            return this.publicIP;
        }

        public CloudPlacement getPlacement() {
            return this.placement;
        }

        public int getPort() {
            return this.port;
        }

        public int getConnectionCount() {
            return this.connectionCount;
        }

        public long getIsDownSince() {
            return this.isDownSince;
        }
    }
}

