package de.flapdoodle.embed.mongo.packageresolver;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link ToolVersionRange}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableToolVersionRange.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableToolVersionRange.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableToolVersionRange
    extends ToolVersionRange {
  private final NumericVersion min;
  private final NumericVersion max;

  private ImmutableToolVersionRange(
      NumericVersion min,
      NumericVersion max) {
    this.min = Objects.requireNonNull(min, "min");
    this.max = Objects.requireNonNull(max, "max");
  }

  private ImmutableToolVersionRange(
      ImmutableToolVersionRange original,
      NumericVersion min,
      NumericVersion max) {
    this.min = min;
    this.max = max;
  }

  /**
   * @return The value of the {@code min} attribute
   */
  @Override
  NumericVersion min() {
    return min;
  }

  /**
   * @return The value of the {@code max} attribute
   */
  @Override
  NumericVersion max() {
    return max;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ToolVersionRange#min() min} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for min
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableToolVersionRange withMin(NumericVersion value) {
    if (this.min == value) return this;
    NumericVersion newValue = Objects.requireNonNull(value, "min");
    return validate(new ImmutableToolVersionRange(this, newValue, this.max));
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ToolVersionRange#max() max} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for max
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableToolVersionRange withMax(NumericVersion value) {
    if (this.max == value) return this;
    NumericVersion newValue = Objects.requireNonNull(value, "max");
    return validate(new ImmutableToolVersionRange(this, this.min, newValue));
  }

  /**
   * This instance is equal to all instances of {@code ImmutableToolVersionRange} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableToolVersionRange
        && equalTo(0, (ImmutableToolVersionRange) another);
  }

  private boolean equalTo(int synthetic, ImmutableToolVersionRange another) {
    return min.equals(another.min)
        && max.equals(another.max);
  }

  /**
   * Computes a hash code from attributes: {@code min}, {@code max}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + min.hashCode();
    h += (h << 5) + max.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code ToolVersionRange} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ToolVersionRange{"
        + "min=" + min
        + ", max=" + max
        + "}";
  }

  /**
   * Construct a new immutable {@code ToolVersionRange} instance.
   * @param min The value for the {@code min} attribute
   * @param max The value for the {@code max} attribute
   * @return An immutable ToolVersionRange instance
   */
  public static ImmutableToolVersionRange of(NumericVersion min, NumericVersion max) {
    return validate(new ImmutableToolVersionRange(min, max));
  }

  private static ImmutableToolVersionRange validate(ImmutableToolVersionRange instance) {
    instance.check();
    return instance;
  }

  /**
   * Creates an immutable copy of a {@link ToolVersionRange} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ToolVersionRange instance
   */
  public static ImmutableToolVersionRange copyOf(ToolVersionRange instance) {
    if (instance instanceof ImmutableToolVersionRange) {
      return (ImmutableToolVersionRange) instance;
    }
    return ImmutableToolVersionRange.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableToolVersionRange ImmutableToolVersionRange}.
   * <pre>
   * ImmutableToolVersionRange.builder()
   *    .min(de.flapdoodle.embed.mongo.packageresolver.NumericVersion) // required {@link ToolVersionRange#min() min}
   *    .max(de.flapdoodle.embed.mongo.packageresolver.NumericVersion) // required {@link ToolVersionRange#max() max}
   *    .build();
   * </pre>
   * @return A new ImmutableToolVersionRange builder
   */
  public static ImmutableToolVersionRange.Builder builder() {
    return new ImmutableToolVersionRange.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableToolVersionRange ImmutableToolVersionRange}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_MIN = 0x1L;
    private static final long INIT_BIT_MAX = 0x2L;
    private long initBits = 0x3L;

    private NumericVersion min;
    private NumericVersion max;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ToolVersionRange} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ToolVersionRange instance) {
      Objects.requireNonNull(instance, "instance");
      this.min(instance.min());
      this.max(instance.max());
      return this;
    }

    /**
     * Initializes the value for the {@link ToolVersionRange#min() min} attribute.
     * @param min The value for min 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder min(NumericVersion min) {
      this.min = Objects.requireNonNull(min, "min");
      initBits &= ~INIT_BIT_MIN;
      return this;
    }

    /**
     * Initializes the value for the {@link ToolVersionRange#max() max} attribute.
     * @param max The value for max 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder max(NumericVersion max) {
      this.max = Objects.requireNonNull(max, "max");
      initBits &= ~INIT_BIT_MAX;
      return this;
    }

    /**
     * Builds a new {@link ImmutableToolVersionRange ImmutableToolVersionRange}.
     * @return An immutable instance of ToolVersionRange
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableToolVersionRange build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return ImmutableToolVersionRange.validate(new ImmutableToolVersionRange(null, min, max));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_MIN) != 0) attributes.add("min");
      if ((initBits & INIT_BIT_MAX) != 0) attributes.add("max");
      return "Cannot build ToolVersionRange, some of required attributes are not set " + attributes;
    }
  }
}
