package de.flapdoodle.embed.mongo.transitions;

import de.flapdoodle.embed.mongo.packageresolver.Command;
import de.flapdoodle.embed.mongo.types.DistributionBaseUrl;
import de.flapdoodle.embed.process.config.store.Package;
import de.flapdoodle.embed.process.distribution.Distribution;
import de.flapdoodle.embed.process.distribution.PackageResolver;
import de.flapdoodle.reverse.StateID;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * Immutable implementation of {@link PackageOfCommandDistribution}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePackageOfCommandDistribution.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutablePackageOfCommandDistribution
    extends PackageOfCommandDistribution {
  private final Function<Command, PackageResolver> commandPackageResolver;
  private final StateID<Command> command;
  private final StateID<Distribution> distribution;
  private final StateID<DistributionBaseUrl> distributionBaseUrl;
  private final StateID<Package> destination;

  private ImmutablePackageOfCommandDistribution(ImmutablePackageOfCommandDistribution.Builder builder) {
    if (builder.commandPackageResolver != null) {
      initShim.commandPackageResolver(builder.commandPackageResolver);
    }
    if (builder.command != null) {
      initShim.command(builder.command);
    }
    if (builder.distribution != null) {
      initShim.distribution(builder.distribution);
    }
    if (builder.distributionBaseUrl != null) {
      initShim.distributionBaseUrl(builder.distributionBaseUrl);
    }
    if (builder.destination != null) {
      initShim.destination(builder.destination);
    }
    this.commandPackageResolver = initShim.commandPackageResolver();
    this.command = initShim.command();
    this.distribution = initShim.distribution();
    this.distributionBaseUrl = initShim.distributionBaseUrl();
    this.destination = initShim.destination();
    this.initShim = null;
  }

  private ImmutablePackageOfCommandDistribution(
      Function<Command, PackageResolver> commandPackageResolver,
      StateID<Command> command,
      StateID<Distribution> distribution,
      StateID<DistributionBaseUrl> distributionBaseUrl,
      StateID<Package> destination) {
    this.commandPackageResolver = commandPackageResolver;
    this.command = command;
    this.distribution = distribution;
    this.distributionBaseUrl = distributionBaseUrl;
    this.destination = destination;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte commandPackageResolverBuildStage = STAGE_UNINITIALIZED;
    private Function<Command, PackageResolver> commandPackageResolver;

    Function<Command, PackageResolver> commandPackageResolver() {
      if (commandPackageResolverBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (commandPackageResolverBuildStage == STAGE_UNINITIALIZED) {
        commandPackageResolverBuildStage = STAGE_INITIALIZING;
        this.commandPackageResolver = Objects.requireNonNull(ImmutablePackageOfCommandDistribution.super.commandPackageResolver(), "commandPackageResolver");
        commandPackageResolverBuildStage = STAGE_INITIALIZED;
      }
      return this.commandPackageResolver;
    }

    void commandPackageResolver(Function<Command, PackageResolver> commandPackageResolver) {
      this.commandPackageResolver = commandPackageResolver;
      commandPackageResolverBuildStage = STAGE_INITIALIZED;
    }

    private byte commandBuildStage = STAGE_UNINITIALIZED;
    private StateID<Command> command;

    StateID<Command> command() {
      if (commandBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (commandBuildStage == STAGE_UNINITIALIZED) {
        commandBuildStage = STAGE_INITIALIZING;
        this.command = Objects.requireNonNull(ImmutablePackageOfCommandDistribution.super.command(), "command");
        commandBuildStage = STAGE_INITIALIZED;
      }
      return this.command;
    }

    void command(StateID<Command> command) {
      this.command = command;
      commandBuildStage = STAGE_INITIALIZED;
    }

    private byte distributionBuildStage = STAGE_UNINITIALIZED;
    private StateID<Distribution> distribution;

    StateID<Distribution> distribution() {
      if (distributionBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distributionBuildStage == STAGE_UNINITIALIZED) {
        distributionBuildStage = STAGE_INITIALIZING;
        this.distribution = Objects.requireNonNull(ImmutablePackageOfCommandDistribution.super.distribution(), "distribution");
        distributionBuildStage = STAGE_INITIALIZED;
      }
      return this.distribution;
    }

    void distribution(StateID<Distribution> distribution) {
      this.distribution = distribution;
      distributionBuildStage = STAGE_INITIALIZED;
    }

    private byte distributionBaseUrlBuildStage = STAGE_UNINITIALIZED;
    private StateID<DistributionBaseUrl> distributionBaseUrl;

    StateID<DistributionBaseUrl> distributionBaseUrl() {
      if (distributionBaseUrlBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (distributionBaseUrlBuildStage == STAGE_UNINITIALIZED) {
        distributionBaseUrlBuildStage = STAGE_INITIALIZING;
        this.distributionBaseUrl = Objects.requireNonNull(ImmutablePackageOfCommandDistribution.super.distributionBaseUrl(), "distributionBaseUrl");
        distributionBaseUrlBuildStage = STAGE_INITIALIZED;
      }
      return this.distributionBaseUrl;
    }

    void distributionBaseUrl(StateID<DistributionBaseUrl> distributionBaseUrl) {
      this.distributionBaseUrl = distributionBaseUrl;
      distributionBaseUrlBuildStage = STAGE_INITIALIZED;
    }

    private byte destinationBuildStage = STAGE_UNINITIALIZED;
    private StateID<Package> destination;

    StateID<Package> destination() {
      if (destinationBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (destinationBuildStage == STAGE_UNINITIALIZED) {
        destinationBuildStage = STAGE_INITIALIZING;
        this.destination = Objects.requireNonNull(ImmutablePackageOfCommandDistribution.super.destination(), "destination");
        destinationBuildStage = STAGE_INITIALIZED;
      }
      return this.destination;
    }

    void destination(StateID<Package> destination) {
      this.destination = destination;
      destinationBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (commandPackageResolverBuildStage == STAGE_INITIALIZING) attributes.add("commandPackageResolver");
      if (commandBuildStage == STAGE_INITIALIZING) attributes.add("command");
      if (distributionBuildStage == STAGE_INITIALIZING) attributes.add("distribution");
      if (distributionBaseUrlBuildStage == STAGE_INITIALIZING) attributes.add("distributionBaseUrl");
      if (destinationBuildStage == STAGE_INITIALIZING) attributes.add("destination");
      return "Cannot build PackageOfCommandDistribution, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code commandPackageResolver} attribute
   */
  @Override
  protected Function<Command, PackageResolver> commandPackageResolver() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.commandPackageResolver()
        : this.commandPackageResolver;
  }

  /**
   * @return The value of the {@code command} attribute
   */
  @Override
  public StateID<Command> command() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.command()
        : this.command;
  }

  /**
   * @return The value of the {@code distribution} attribute
   */
  @Override
  public StateID<Distribution> distribution() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distribution()
        : this.distribution;
  }

  /**
   * @return The value of the {@code distributionBaseUrl} attribute
   */
  @Override
  public StateID<DistributionBaseUrl> distributionBaseUrl() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.distributionBaseUrl()
        : this.distributionBaseUrl;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<Package> destination() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.destination()
        : this.destination;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfCommandDistribution#commandPackageResolver() commandPackageResolver} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for commandPackageResolver
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackageOfCommandDistribution withCommandPackageResolver(Function<Command, PackageResolver> value) {
    if (this.commandPackageResolver == value) return this;
    Function<Command, PackageResolver> newValue = Objects.requireNonNull(value, "commandPackageResolver");
    return new ImmutablePackageOfCommandDistribution(newValue, this.command, this.distribution, this.distributionBaseUrl, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfCommandDistribution#command() command} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for command
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackageOfCommandDistribution withCommand(StateID<Command> value) {
    if (this.command == value) return this;
    StateID<Command> newValue = Objects.requireNonNull(value, "command");
    return new ImmutablePackageOfCommandDistribution(
        this.commandPackageResolver,
        newValue,
        this.distribution,
        this.distributionBaseUrl,
        this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfCommandDistribution#distribution() distribution} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distribution
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackageOfCommandDistribution withDistribution(StateID<Distribution> value) {
    if (this.distribution == value) return this;
    StateID<Distribution> newValue = Objects.requireNonNull(value, "distribution");
    return new ImmutablePackageOfCommandDistribution(this.commandPackageResolver, this.command, newValue, this.distributionBaseUrl, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfCommandDistribution#distributionBaseUrl() distributionBaseUrl} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for distributionBaseUrl
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackageOfCommandDistribution withDistributionBaseUrl(StateID<DistributionBaseUrl> value) {
    if (this.distributionBaseUrl == value) return this;
    StateID<DistributionBaseUrl> newValue = Objects.requireNonNull(value, "distributionBaseUrl");
    return new ImmutablePackageOfCommandDistribution(this.commandPackageResolver, this.command, this.distribution, newValue, this.destination);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageOfCommandDistribution#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutablePackageOfCommandDistribution withDestination(StateID<Package> value) {
    if (this.destination == value) return this;
    StateID<Package> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutablePackageOfCommandDistribution(
        this.commandPackageResolver,
        this.command,
        this.distribution,
        this.distributionBaseUrl,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePackageOfCommandDistribution} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePackageOfCommandDistribution
        && equalTo(0, (ImmutablePackageOfCommandDistribution) another);
  }

  private boolean equalTo(int synthetic, ImmutablePackageOfCommandDistribution another) {
    return commandPackageResolver.equals(another.commandPackageResolver)
        && command.equals(another.command)
        && distribution.equals(another.distribution)
        && distributionBaseUrl.equals(another.distributionBaseUrl)
        && destination.equals(another.destination);
  }

  /**
   * Computes a hash code from attributes: {@code commandPackageResolver}, {@code command}, {@code distribution}, {@code distributionBaseUrl}, {@code destination}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + commandPackageResolver.hashCode();
    h += (h << 5) + command.hashCode();
    h += (h << 5) + distribution.hashCode();
    h += (h << 5) + distributionBaseUrl.hashCode();
    h += (h << 5) + destination.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code PackageOfCommandDistribution} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "PackageOfCommandDistribution{"
        + "commandPackageResolver=" + commandPackageResolver
        + ", command=" + command
        + ", distribution=" + distribution
        + ", distributionBaseUrl=" + distributionBaseUrl
        + ", destination=" + destination
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link PackageOfCommandDistribution} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable PackageOfCommandDistribution instance
   */
  public static ImmutablePackageOfCommandDistribution copyOf(PackageOfCommandDistribution instance) {
    if (instance instanceof ImmutablePackageOfCommandDistribution) {
      return (ImmutablePackageOfCommandDistribution) instance;
    }
    return ImmutablePackageOfCommandDistribution.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePackageOfCommandDistribution ImmutablePackageOfCommandDistribution}.
   * <pre>
   * ImmutablePackageOfCommandDistribution.builder()
   *    .commandPackageResolver(function.Function&amp;lt;de.flapdoodle.embed.mongo.packageresolver.Command, de.flapdoodle.embed.process.distribution.PackageResolver&amp;gt;) // optional {@link PackageOfCommandDistribution#commandPackageResolver() commandPackageResolver}
   *    .command(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.mongo.packageresolver.Command&amp;gt;) // optional {@link PackageOfCommandDistribution#command() command}
   *    .distribution(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.distribution.Distribution&amp;gt;) // optional {@link PackageOfCommandDistribution#distribution() distribution}
   *    .distributionBaseUrl(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.mongo.types.DistributionBaseUrl&amp;gt;) // optional {@link PackageOfCommandDistribution#distributionBaseUrl() distributionBaseUrl}
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.config.store.Package&amp;gt;) // optional {@link PackageOfCommandDistribution#destination() destination}
   *    .build();
   * </pre>
   * @return A new ImmutablePackageOfCommandDistribution builder
   */
  public static ImmutablePackageOfCommandDistribution.Builder builder() {
    return new ImmutablePackageOfCommandDistribution.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePackageOfCommandDistribution ImmutablePackageOfCommandDistribution}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private Function<Command, PackageResolver> commandPackageResolver;
    private StateID<Command> command;
    private StateID<Distribution> distribution;
    private StateID<DistributionBaseUrl> distributionBaseUrl;
    private StateID<Package> destination;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code PackageOfCommandDistribution} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(PackageOfCommandDistribution instance) {
      Objects.requireNonNull(instance, "instance");
      commandPackageResolver(instance.commandPackageResolver());
      command(instance.command());
      distribution(instance.distribution());
      distributionBaseUrl(instance.distributionBaseUrl());
      destination(instance.destination());
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfCommandDistribution#commandPackageResolver() commandPackageResolver} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfCommandDistribution#commandPackageResolver() commandPackageResolver}.</em>
     * @param commandPackageResolver The value for commandPackageResolver 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder commandPackageResolver(Function<Command, PackageResolver> commandPackageResolver) {
      this.commandPackageResolver = Objects.requireNonNull(commandPackageResolver, "commandPackageResolver");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfCommandDistribution#command() command} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfCommandDistribution#command() command}.</em>
     * @param command The value for command 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder command(StateID<Command> command) {
      this.command = Objects.requireNonNull(command, "command");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfCommandDistribution#distribution() distribution} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfCommandDistribution#distribution() distribution}.</em>
     * @param distribution The value for distribution 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distribution(StateID<Distribution> distribution) {
      this.distribution = Objects.requireNonNull(distribution, "distribution");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfCommandDistribution#distributionBaseUrl() distributionBaseUrl} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfCommandDistribution#distributionBaseUrl() distributionBaseUrl}.</em>
     * @param distributionBaseUrl The value for distributionBaseUrl 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder distributionBaseUrl(StateID<DistributionBaseUrl> distributionBaseUrl) {
      this.distributionBaseUrl = Objects.requireNonNull(distributionBaseUrl, "distributionBaseUrl");
      return this;
    }

    /**
     * Initializes the value for the {@link PackageOfCommandDistribution#destination() destination} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link PackageOfCommandDistribution#destination() destination}.</em>
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder destination(StateID<Package> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      return this;
    }

    /**
     * Builds a new {@link ImmutablePackageOfCommandDistribution ImmutablePackageOfCommandDistribution}.
     * @return An immutable instance of PackageOfCommandDistribution
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePackageOfCommandDistribution build() {
      return new ImmutablePackageOfCommandDistribution(this);
    }
  }
}
