
package de.flapdoodle.embed.mongo.client;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link AuthenticationSetup.User}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableUser.builder()}.
 * Use the static factory method to create immutable instances:
 * {@code ImmutableUser.of()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableUser implements AuthenticationSetup.User {
  private final String database;
  private final UsernamePassword user;
  private final List<String> roles;

  private ImmutableUser(String database, UsernamePassword user) {
    this.database = Objects.requireNonNull(database, "database");
    this.user = Objects.requireNonNull(user, "user");
    this.roles = Collections.emptyList();
  }

  private ImmutableUser(
      String database,
      UsernamePassword user,
      List<String> roles) {
    this.database = database;
    this.user = user;
    this.roles = roles;
  }

  /**
   * @return The value of the {@code database} attribute
   */
  @Override
  public String database() {
    return database;
  }

  /**
   * @return The value of the {@code user} attribute
   */
  @Override
  public UsernamePassword user() {
    return user;
  }

  /**
   * @return The value of the {@code roles} attribute
   */
  @Override
  public List<String> roles() {
    return roles;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup.User#database() database} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for database
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUser withDatabase(String value) {
    String newValue = Objects.requireNonNull(value, "database");
    if (this.database.equals(newValue)) return this;
    return new ImmutableUser(newValue, this.user, this.roles);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link AuthenticationSetup.User#user() user} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for user
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableUser withUser(UsernamePassword value) {
    if (this.user == value) return this;
    UsernamePassword newValue = Objects.requireNonNull(value, "user");
    return new ImmutableUser(this.database, newValue, this.roles);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup.User#roles() roles}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableUser withRoles(String... elements) {
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableUser(this.database, this.user, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link AuthenticationSetup.User#roles() roles}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of roles elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableUser withRoles(Iterable<String> elements) {
    if (this.roles == elements) return this;
    List<String> newValue = createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableUser(this.database, this.user, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableUser} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableUser
        && equalsByValue((ImmutableUser) another);
  }

  private boolean equalsByValue(ImmutableUser another) {
    return database.equals(another.database)
        && user.equals(another.user)
        && roles.equals(another.roles);
  }

  /**
   * Computes a hash code from attributes: {@code database}, {@code user}, {@code roles}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + database.hashCode();
    h += (h << 5) + user.hashCode();
    h += (h << 5) + roles.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code User} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "User{"
        + "database=" + database
        + ", user=" + user
        + ", roles=" + roles
        + "}";
  }

  /**
   * Construct a new immutable {@code User} instance.
   * @param database The value for the {@code database} attribute
   * @param user The value for the {@code user} attribute
   * @return An immutable User instance
   */
  public static ImmutableUser of(String database, UsernamePassword user) {
    return new ImmutableUser(database, user);
  }

  /**
   * Creates an immutable copy of a {@link AuthenticationSetup.User} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable User instance
   */
  public static ImmutableUser copyOf(AuthenticationSetup.User instance) {
    if (instance instanceof ImmutableUser) {
      return (ImmutableUser) instance;
    }
    return ImmutableUser.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableUser ImmutableUser}.
   * <pre>
   * ImmutableUser.builder()
   *    .database(String) // required {@link AuthenticationSetup.User#database() database}
   *    .user(de.flapdoodle.embed.mongo.client.UsernamePassword) // required {@link AuthenticationSetup.User#user() user}
   *    .addRoles|addAllRoles(String) // {@link AuthenticationSetup.User#roles() roles} elements
   *    .build();
   * </pre>
   * @return A new ImmutableUser builder
   */
  public static ImmutableUser.Builder builder() {
    return new ImmutableUser.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableUser ImmutableUser}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long INIT_BIT_DATABASE = 0x1L;
    private static final long INIT_BIT_USER = 0x2L;
    private long initBits = 0x3L;

    private String database;
    private UsernamePassword user;
    private List<String> roles = new ArrayList<String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code User} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(AuthenticationSetup.User instance) {
      Objects.requireNonNull(instance, "instance");
      this.database(instance.database());
      this.user(instance.user());
      addAllRoles(instance.roles());
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup.User#database() database} attribute.
     * @param database The value for database 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder database(String database) {
      this.database = Objects.requireNonNull(database, "database");
      initBits &= ~INIT_BIT_DATABASE;
      return this;
    }

    /**
     * Initializes the value for the {@link AuthenticationSetup.User#user() user} attribute.
     * @param user The value for user 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder user(UsernamePassword user) {
      this.user = Objects.requireNonNull(user, "user");
      initBits &= ~INIT_BIT_USER;
      return this;
    }

    /**
     * Adds one element to {@link AuthenticationSetup.User#roles() roles} list.
     * @param element A roles element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addRoles(String element) {
      this.roles.add(Objects.requireNonNull(element, "roles element"));
      return this;
    }

    /**
     * Adds elements to {@link AuthenticationSetup.User#roles() roles} list.
     * @param elements An array of roles elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addRoles(String... elements) {
      for (String element : elements) {
        this.roles.add(Objects.requireNonNull(element, "roles element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link AuthenticationSetup.User#roles() roles} list.
     * @param elements An iterable of roles elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder roles(Iterable<String> elements) {
      this.roles.clear();
      return addAllRoles(elements);
    }

    /**
     * Adds elements to {@link AuthenticationSetup.User#roles() roles} list.
     * @param elements An iterable of roles elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllRoles(Iterable<String> elements) {
      for (String element : elements) {
        this.roles.add(Objects.requireNonNull(element, "roles element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableUser ImmutableUser}.
     * @return An immutable instance of User
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableUser build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableUser(database, user, createUnmodifiableList(true, roles));
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DATABASE) != 0) attributes.add("database");
      if ((initBits & INIT_BIT_USER) != 0) attributes.add("user");
      return "Cannot build User, some of required attributes are not set " + attributes;
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
