package de.flapdoodle.embed.process.config;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link TimeoutConfig}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableTimeoutConfig.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableTimeoutConfig implements TimeoutConfig {
  private final int connectionTimeout;
  private final int readTimeout;

  private ImmutableTimeoutConfig(ImmutableTimeoutConfig.Builder builder) {
    if (builder.connectionTimeoutIsSet()) {
      initShim.connectionTimeout(builder.connectionTimeout);
    }
    if (builder.readTimeoutIsSet()) {
      initShim.readTimeout(builder.readTimeout);
    }
    this.connectionTimeout = initShim.getConnectionTimeout();
    this.readTimeout = initShim.getReadTimeout();
    this.initShim = null;
  }

  private ImmutableTimeoutConfig(int connectionTimeout, int readTimeout) {
    this.connectionTimeout = connectionTimeout;
    this.readTimeout = readTimeout;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte connectionTimeoutBuildStage = STAGE_UNINITIALIZED;
    private int connectionTimeout;

    int getConnectionTimeout() {
      if (connectionTimeoutBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (connectionTimeoutBuildStage == STAGE_UNINITIALIZED) {
        connectionTimeoutBuildStage = STAGE_INITIALIZING;
        this.connectionTimeout = getConnectionTimeoutInitialize();
        connectionTimeoutBuildStage = STAGE_INITIALIZED;
      }
      return this.connectionTimeout;
    }

    void connectionTimeout(int connectionTimeout) {
      this.connectionTimeout = connectionTimeout;
      connectionTimeoutBuildStage = STAGE_INITIALIZED;
    }

    private byte readTimeoutBuildStage = STAGE_UNINITIALIZED;
    private int readTimeout;

    int getReadTimeout() {
      if (readTimeoutBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (readTimeoutBuildStage == STAGE_UNINITIALIZED) {
        readTimeoutBuildStage = STAGE_INITIALIZING;
        this.readTimeout = getReadTimeoutInitialize();
        readTimeoutBuildStage = STAGE_INITIALIZED;
      }
      return this.readTimeout;
    }

    void readTimeout(int readTimeout) {
      this.readTimeout = readTimeout;
      readTimeoutBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (connectionTimeoutBuildStage == STAGE_INITIALIZING) attributes.add("connectionTimeout");
      if (readTimeoutBuildStage == STAGE_INITIALIZING) attributes.add("readTimeout");
      return "Cannot build TimeoutConfig, attribute initializers form cycle " + attributes;
    }
  }

  private int getConnectionTimeoutInitialize() {
    return TimeoutConfig.super.getConnectionTimeout();
  }

  private int getReadTimeoutInitialize() {
    return TimeoutConfig.super.getReadTimeout();
  }

  /**
   * @return The value of the {@code connectionTimeout} attribute
   */
  @Override
  public int getConnectionTimeout() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.getConnectionTimeout()
        : this.connectionTimeout;
  }

  /**
   * @return The value of the {@code readTimeout} attribute
   */
  @Override
  public int getReadTimeout() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.getReadTimeout()
        : this.readTimeout;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TimeoutConfig#getConnectionTimeout() connectionTimeout} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for connectionTimeout
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTimeoutConfig withConnectionTimeout(int value) {
    if (this.connectionTimeout == value) return this;
    return new ImmutableTimeoutConfig(value, this.readTimeout);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link TimeoutConfig#getReadTimeout() readTimeout} attribute.
   * A value equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for readTimeout
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableTimeoutConfig withReadTimeout(int value) {
    if (this.readTimeout == value) return this;
    return new ImmutableTimeoutConfig(this.connectionTimeout, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableTimeoutConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableTimeoutConfig
        && equalTo(0, (ImmutableTimeoutConfig) another);
  }

  private boolean equalTo(int synthetic, ImmutableTimeoutConfig another) {
    return connectionTimeout == another.connectionTimeout
        && readTimeout == another.readTimeout;
  }

  /**
   * Computes a hash code from attributes: {@code connectionTimeout}, {@code readTimeout}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + connectionTimeout;
    h += (h << 5) + readTimeout;
    return h;
  }

  /**
   * Prints the immutable value {@code TimeoutConfig} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "TimeoutConfig{"
        + "connectionTimeout=" + connectionTimeout
        + ", readTimeout=" + readTimeout
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link TimeoutConfig} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable TimeoutConfig instance
   */
  public static ImmutableTimeoutConfig copyOf(TimeoutConfig instance) {
    if (instance instanceof ImmutableTimeoutConfig) {
      return (ImmutableTimeoutConfig) instance;
    }
    return ImmutableTimeoutConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableTimeoutConfig ImmutableTimeoutConfig}.
   * <pre>
   * ImmutableTimeoutConfig.builder()
   *    .connectionTimeout(int) // optional {@link TimeoutConfig#getConnectionTimeout() connectionTimeout}
   *    .readTimeout(int) // optional {@link TimeoutConfig#getReadTimeout() readTimeout}
   *    .build();
   * </pre>
   * @return A new ImmutableTimeoutConfig builder
   */
  public static ImmutableTimeoutConfig.Builder builder() {
    return new ImmutableTimeoutConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableTimeoutConfig ImmutableTimeoutConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private static final long OPT_BIT_CONNECTION_TIMEOUT = 0x1L;
    private static final long OPT_BIT_READ_TIMEOUT = 0x2L;
    private long optBits;

    private int connectionTimeout;
    private int readTimeout;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code TimeoutConfig} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(TimeoutConfig instance) {
      Objects.requireNonNull(instance, "instance");
      connectionTimeout(instance.getConnectionTimeout());
      readTimeout(instance.getReadTimeout());
      return this;
    }

    /**
     * Initializes the value for the {@link TimeoutConfig#getConnectionTimeout() connectionTimeout} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link TimeoutConfig#getConnectionTimeout() connectionTimeout}.</em>
     * @param connectionTimeout The value for connectionTimeout 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder connectionTimeout(int connectionTimeout) {
      this.connectionTimeout = connectionTimeout;
      optBits |= OPT_BIT_CONNECTION_TIMEOUT;
      return this;
    }

    /**
     * Initializes the value for the {@link TimeoutConfig#getReadTimeout() readTimeout} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link TimeoutConfig#getReadTimeout() readTimeout}.</em>
     * @param readTimeout The value for readTimeout 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder readTimeout(int readTimeout) {
      this.readTimeout = readTimeout;
      optBits |= OPT_BIT_READ_TIMEOUT;
      return this;
    }

    /**
     * Builds a new {@link ImmutableTimeoutConfig ImmutableTimeoutConfig}.
     * @return An immutable instance of TimeoutConfig
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableTimeoutConfig build() {
      return new ImmutableTimeoutConfig(this);
    }

    private boolean connectionTimeoutIsSet() {
      return (optBits & OPT_BIT_CONNECTION_TIMEOUT) != 0;
    }

    private boolean readTimeoutIsSet() {
      return (optBits & OPT_BIT_READ_TIMEOUT) != 0;
    }
  }
}
