package de.flapdoodle.embed.process.transitions;

import de.flapdoodle.embed.process.archives.ExtractedFileSet;
import de.flapdoodle.embed.process.config.SupportConfig;
import de.flapdoodle.embed.process.io.ProcessOutput;
import de.flapdoodle.embed.process.types.*;
import de.flapdoodle.embed.process.types.ProcessConfig;
import de.flapdoodle.embed.process.types.RunningProcess;
import de.flapdoodle.embed.process.types.RunningProcessFactory;
import de.flapdoodle.reverse.StateID;
import de.flapdoodle.reverse.naming.HasLabel;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link Starter}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableStarter.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableStarter<T extends RunningProcess>
    extends Starter<T> {
  private final StateID<ExtractedFileSet> processExecutable;
  private final StateID<ProcessWorkingDir> processWorkingDir;
  private final StateID<ProcessConfig> processConfig;
  private final StateID<ProcessEnv> processEnv;
  private final StateID<ProcessArguments> arguments;
  private final StateID<ProcessOutput> processOutput;
  private final StateID<SupportConfig> supportConfig;
  private final String transitionLabel;
  private final StateID<T> destination;
  private final RunningProcessFactory<T> runningProcessFactory;

  private ImmutableStarter(ImmutableStarter.Builder<T> builder) {
    this.destination = builder.destination;
    this.runningProcessFactory = builder.runningProcessFactory;
    if (builder.processExecutable != null) {
      initShim.processExecutable(builder.processExecutable);
    }
    if (builder.processWorkingDir != null) {
      initShim.processWorkingDir(builder.processWorkingDir);
    }
    if (builder.processConfig != null) {
      initShim.processConfig(builder.processConfig);
    }
    if (builder.processEnv != null) {
      initShim.processEnv(builder.processEnv);
    }
    if (builder.arguments != null) {
      initShim.arguments(builder.arguments);
    }
    if (builder.processOutput != null) {
      initShim.processOutput(builder.processOutput);
    }
    if (builder.supportConfig != null) {
      initShim.supportConfig(builder.supportConfig);
    }
    if (builder.transitionLabel != null) {
      initShim.transitionLabel(builder.transitionLabel);
    }
    this.processExecutable = initShim.processExecutable();
    this.processWorkingDir = initShim.processWorkingDir();
    this.processConfig = initShim.processConfig();
    this.processEnv = initShim.processEnv();
    this.arguments = initShim.arguments();
    this.processOutput = initShim.processOutput();
    this.supportConfig = initShim.supportConfig();
    this.transitionLabel = initShim.transitionLabel();
    this.initShim = null;
  }

  private ImmutableStarter(
      StateID<ExtractedFileSet> processExecutable,
      StateID<ProcessWorkingDir> processWorkingDir,
      StateID<ProcessConfig> processConfig,
      StateID<ProcessEnv> processEnv,
      StateID<ProcessArguments> arguments,
      StateID<ProcessOutput> processOutput,
      StateID<SupportConfig> supportConfig,
      String transitionLabel,
      StateID<T> destination,
      RunningProcessFactory<T> runningProcessFactory) {
    this.processExecutable = processExecutable;
    this.processWorkingDir = processWorkingDir;
    this.processConfig = processConfig;
    this.processEnv = processEnv;
    this.arguments = arguments;
    this.processOutput = processOutput;
    this.supportConfig = supportConfig;
    this.transitionLabel = transitionLabel;
    this.destination = destination;
    this.runningProcessFactory = runningProcessFactory;
    this.initShim = null;
  }

  private static final byte STAGE_INITIALIZING = -1;
  private static final byte STAGE_UNINITIALIZED = 0;
  private static final byte STAGE_INITIALIZED = 1;
  private transient volatile InitShim initShim = new InitShim();

  private final class InitShim {
    private byte processExecutableBuildStage = STAGE_UNINITIALIZED;
    private StateID<ExtractedFileSet> processExecutable;

    StateID<ExtractedFileSet> processExecutable() {
      if (processExecutableBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processExecutableBuildStage == STAGE_UNINITIALIZED) {
        processExecutableBuildStage = STAGE_INITIALIZING;
        this.processExecutable = Objects.requireNonNull(ImmutableStarter.super.processExecutable(), "processExecutable");
        processExecutableBuildStage = STAGE_INITIALIZED;
      }
      return this.processExecutable;
    }

    void processExecutable(StateID<ExtractedFileSet> processExecutable) {
      this.processExecutable = processExecutable;
      processExecutableBuildStage = STAGE_INITIALIZED;
    }

    private byte processWorkingDirBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessWorkingDir> processWorkingDir;

    StateID<ProcessWorkingDir> processWorkingDir() {
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processWorkingDirBuildStage == STAGE_UNINITIALIZED) {
        processWorkingDirBuildStage = STAGE_INITIALIZING;
        this.processWorkingDir = Objects.requireNonNull(ImmutableStarter.super.processWorkingDir(), "processWorkingDir");
        processWorkingDirBuildStage = STAGE_INITIALIZED;
      }
      return this.processWorkingDir;
    }

    void processWorkingDir(StateID<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = processWorkingDir;
      processWorkingDirBuildStage = STAGE_INITIALIZED;
    }

    private byte processConfigBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessConfig> processConfig;

    StateID<ProcessConfig> processConfig() {
      if (processConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processConfigBuildStage == STAGE_UNINITIALIZED) {
        processConfigBuildStage = STAGE_INITIALIZING;
        this.processConfig = Objects.requireNonNull(ImmutableStarter.super.processConfig(), "processConfig");
        processConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.processConfig;
    }

    void processConfig(StateID<ProcessConfig> processConfig) {
      this.processConfig = processConfig;
      processConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte processEnvBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessEnv> processEnv;

    StateID<ProcessEnv> processEnv() {
      if (processEnvBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processEnvBuildStage == STAGE_UNINITIALIZED) {
        processEnvBuildStage = STAGE_INITIALIZING;
        this.processEnv = Objects.requireNonNull(ImmutableStarter.super.processEnv(), "processEnv");
        processEnvBuildStage = STAGE_INITIALIZED;
      }
      return this.processEnv;
    }

    void processEnv(StateID<ProcessEnv> processEnv) {
      this.processEnv = processEnv;
      processEnvBuildStage = STAGE_INITIALIZED;
    }

    private byte argumentsBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessArguments> arguments;

    StateID<ProcessArguments> arguments() {
      if (argumentsBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (argumentsBuildStage == STAGE_UNINITIALIZED) {
        argumentsBuildStage = STAGE_INITIALIZING;
        this.arguments = Objects.requireNonNull(ImmutableStarter.super.arguments(), "arguments");
        argumentsBuildStage = STAGE_INITIALIZED;
      }
      return this.arguments;
    }

    void arguments(StateID<ProcessArguments> arguments) {
      this.arguments = arguments;
      argumentsBuildStage = STAGE_INITIALIZED;
    }

    private byte processOutputBuildStage = STAGE_UNINITIALIZED;
    private StateID<ProcessOutput> processOutput;

    StateID<ProcessOutput> processOutput() {
      if (processOutputBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (processOutputBuildStage == STAGE_UNINITIALIZED) {
        processOutputBuildStage = STAGE_INITIALIZING;
        this.processOutput = Objects.requireNonNull(ImmutableStarter.super.processOutput(), "processOutput");
        processOutputBuildStage = STAGE_INITIALIZED;
      }
      return this.processOutput;
    }

    void processOutput(StateID<ProcessOutput> processOutput) {
      this.processOutput = processOutput;
      processOutputBuildStage = STAGE_INITIALIZED;
    }

    private byte supportConfigBuildStage = STAGE_UNINITIALIZED;
    private StateID<SupportConfig> supportConfig;

    StateID<SupportConfig> supportConfig() {
      if (supportConfigBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (supportConfigBuildStage == STAGE_UNINITIALIZED) {
        supportConfigBuildStage = STAGE_INITIALIZING;
        this.supportConfig = Objects.requireNonNull(ImmutableStarter.super.supportConfig(), "supportConfig");
        supportConfigBuildStage = STAGE_INITIALIZED;
      }
      return this.supportConfig;
    }

    void supportConfig(StateID<SupportConfig> supportConfig) {
      this.supportConfig = supportConfig;
      supportConfigBuildStage = STAGE_INITIALIZED;
    }

    private byte transitionLabelBuildStage = STAGE_UNINITIALIZED;
    private String transitionLabel;

    String transitionLabel() {
      if (transitionLabelBuildStage == STAGE_INITIALIZING) throw new IllegalStateException(formatInitCycleMessage());
      if (transitionLabelBuildStage == STAGE_UNINITIALIZED) {
        transitionLabelBuildStage = STAGE_INITIALIZING;
        this.transitionLabel = Objects.requireNonNull(ImmutableStarter.super.transitionLabel(), "transitionLabel");
        transitionLabelBuildStage = STAGE_INITIALIZED;
      }
      return this.transitionLabel;
    }

    void transitionLabel(String transitionLabel) {
      this.transitionLabel = transitionLabel;
      transitionLabelBuildStage = STAGE_INITIALIZED;
    }

    private String formatInitCycleMessage() {
      List<String> attributes = new ArrayList<>();
      if (processExecutableBuildStage == STAGE_INITIALIZING) attributes.add("processExecutable");
      if (processWorkingDirBuildStage == STAGE_INITIALIZING) attributes.add("processWorkingDir");
      if (processConfigBuildStage == STAGE_INITIALIZING) attributes.add("processConfig");
      if (processEnvBuildStage == STAGE_INITIALIZING) attributes.add("processEnv");
      if (argumentsBuildStage == STAGE_INITIALIZING) attributes.add("arguments");
      if (processOutputBuildStage == STAGE_INITIALIZING) attributes.add("processOutput");
      if (supportConfigBuildStage == STAGE_INITIALIZING) attributes.add("supportConfig");
      if (transitionLabelBuildStage == STAGE_INITIALIZING) attributes.add("transitionLabel");
      return "Cannot build Starter, attribute initializers form cycle " + attributes;
    }
  }

  /**
   * @return The value of the {@code processExecutable} attribute
   */
  @Override
  public StateID<ExtractedFileSet> processExecutable() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processExecutable()
        : this.processExecutable;
  }

  /**
   * @return The value of the {@code processWorkingDir} attribute
   */
  @Override
  public StateID<ProcessWorkingDir> processWorkingDir() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processWorkingDir()
        : this.processWorkingDir;
  }

  /**
   * @return The value of the {@code processConfig} attribute
   */
  @Override
  public StateID<ProcessConfig> processConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processConfig()
        : this.processConfig;
  }

  /**
   * @return The value of the {@code processEnv} attribute
   */
  @Override
  public StateID<ProcessEnv> processEnv() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processEnv()
        : this.processEnv;
  }

  /**
   * @return The value of the {@code arguments} attribute
   */
  @Override
  public StateID<ProcessArguments> arguments() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.arguments()
        : this.arguments;
  }

  /**
   * @return The value of the {@code processOutput} attribute
   */
  @Override
  public StateID<ProcessOutput> processOutput() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.processOutput()
        : this.processOutput;
  }

  /**
   * @return The value of the {@code supportConfig} attribute
   */
  @Override
  public StateID<SupportConfig> supportConfig() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.supportConfig()
        : this.supportConfig;
  }

  /**
   * @return The value of the {@code transitionLabel} attribute
   */
  @Override
  public String transitionLabel() {
    InitShim shim = this.initShim;
    return shim != null
        ? shim.transitionLabel()
        : this.transitionLabel;
  }

  /**
   * @return The value of the {@code destination} attribute
   */
  @Override
  public StateID<T> destination() {
    return destination;
  }

  /**
   * @return The value of the {@code runningProcessFactory} attribute
   */
  @Override
  protected RunningProcessFactory<T> runningProcessFactory() {
    return runningProcessFactory;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#processExecutable() processExecutable} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processExecutable
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withProcessExecutable(StateID<ExtractedFileSet> value) {
    if (this.processExecutable == value) return this;
    StateID<ExtractedFileSet> newValue = Objects.requireNonNull(value, "processExecutable");
    return new ImmutableStarter<>(
        newValue,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#processWorkingDir() processWorkingDir} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processWorkingDir
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withProcessWorkingDir(StateID<ProcessWorkingDir> value) {
    if (this.processWorkingDir == value) return this;
    StateID<ProcessWorkingDir> newValue = Objects.requireNonNull(value, "processWorkingDir");
    return new ImmutableStarter<>(
        this.processExecutable,
        newValue,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#processConfig() processConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withProcessConfig(StateID<ProcessConfig> value) {
    if (this.processConfig == value) return this;
    StateID<ProcessConfig> newValue = Objects.requireNonNull(value, "processConfig");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        newValue,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#processEnv() processEnv} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processEnv
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withProcessEnv(StateID<ProcessEnv> value) {
    if (this.processEnv == value) return this;
    StateID<ProcessEnv> newValue = Objects.requireNonNull(value, "processEnv");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        newValue,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#arguments() arguments} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for arguments
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withArguments(StateID<ProcessArguments> value) {
    if (this.arguments == value) return this;
    StateID<ProcessArguments> newValue = Objects.requireNonNull(value, "arguments");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        newValue,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#processOutput() processOutput} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for processOutput
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withProcessOutput(StateID<ProcessOutput> value) {
    if (this.processOutput == value) return this;
    StateID<ProcessOutput> newValue = Objects.requireNonNull(value, "processOutput");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        newValue,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#supportConfig() supportConfig} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for supportConfig
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withSupportConfig(StateID<SupportConfig> value) {
    if (this.supportConfig == value) return this;
    StateID<SupportConfig> newValue = Objects.requireNonNull(value, "supportConfig");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        newValue,
        this.transitionLabel,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#transitionLabel() transitionLabel} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for transitionLabel
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withTransitionLabel(String value) {
    String newValue = Objects.requireNonNull(value, "transitionLabel");
    if (this.transitionLabel.equals(newValue)) return this;
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        newValue,
        this.destination,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#destination() destination} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for destination
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withDestination(StateID<T> value) {
    if (this.destination == value) return this;
    StateID<T> newValue = Objects.requireNonNull(value, "destination");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        newValue,
        this.runningProcessFactory);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Starter#runningProcessFactory() runningProcessFactory} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for runningProcessFactory
   * @return A modified copy of the {@code this} object
   */
  public final ImmutableStarter<T> withRunningProcessFactory(RunningProcessFactory<T> value) {
    if (this.runningProcessFactory == value) return this;
    RunningProcessFactory<T> newValue = Objects.requireNonNull(value, "runningProcessFactory");
    return new ImmutableStarter<>(
        this.processExecutable,
        this.processWorkingDir,
        this.processConfig,
        this.processEnv,
        this.arguments,
        this.processOutput,
        this.supportConfig,
        this.transitionLabel,
        this.destination,
        newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableStarter} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableStarter<?>
        && equalTo(0, (ImmutableStarter<?>) another);
  }

  private boolean equalTo(int synthetic, ImmutableStarter<?> another) {
    return processExecutable.equals(another.processExecutable)
        && processWorkingDir.equals(another.processWorkingDir)
        && processConfig.equals(another.processConfig)
        && processEnv.equals(another.processEnv)
        && arguments.equals(another.arguments)
        && processOutput.equals(another.processOutput)
        && supportConfig.equals(another.supportConfig)
        && transitionLabel.equals(another.transitionLabel)
        && destination.equals(another.destination)
        && runningProcessFactory.equals(another.runningProcessFactory);
  }

  /**
   * Computes a hash code from attributes: {@code processExecutable}, {@code processWorkingDir}, {@code processConfig}, {@code processEnv}, {@code arguments}, {@code processOutput}, {@code supportConfig}, {@code transitionLabel}, {@code destination}, {@code runningProcessFactory}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + processExecutable.hashCode();
    h += (h << 5) + processWorkingDir.hashCode();
    h += (h << 5) + processConfig.hashCode();
    h += (h << 5) + processEnv.hashCode();
    h += (h << 5) + arguments.hashCode();
    h += (h << 5) + processOutput.hashCode();
    h += (h << 5) + supportConfig.hashCode();
    h += (h << 5) + transitionLabel.hashCode();
    h += (h << 5) + destination.hashCode();
    h += (h << 5) + runningProcessFactory.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Starter} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Starter{"
        + "processExecutable=" + processExecutable
        + ", processWorkingDir=" + processWorkingDir
        + ", processConfig=" + processConfig
        + ", processEnv=" + processEnv
        + ", arguments=" + arguments
        + ", processOutput=" + processOutput
        + ", supportConfig=" + supportConfig
        + ", transitionLabel=" + transitionLabel
        + ", destination=" + destination
        + ", runningProcessFactory=" + runningProcessFactory
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Starter} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param <T> generic parameter T
   * @param instance The instance to copy
   * @return A copied immutable Starter instance
   */
  public static <T extends RunningProcess> ImmutableStarter<T> copyOf(Starter<T> instance) {
    if (instance instanceof ImmutableStarter<?>) {
      return (ImmutableStarter<T>) instance;
    }
    return ImmutableStarter.<T>builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableStarter ImmutableStarter}.
   * <pre>
   * ImmutableStarter.&amp;lt;T&amp;gt;builder()
   *    .processExecutable(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.archives.ExtractedFileSet&amp;gt;) // optional {@link Starter#processExecutable() processExecutable}
   *    .processWorkingDir(de.flapdoodle.reverse.StateID&amp;lt;ProcessWorkingDir&amp;gt;) // optional {@link Starter#processWorkingDir() processWorkingDir}
   *    .processConfig(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.types.ProcessConfig&amp;gt;) // optional {@link Starter#processConfig() processConfig}
   *    .processEnv(de.flapdoodle.reverse.StateID&amp;lt;ProcessEnv&amp;gt;) // optional {@link Starter#processEnv() processEnv}
   *    .arguments(de.flapdoodle.reverse.StateID&amp;lt;ProcessArguments&amp;gt;) // optional {@link Starter#arguments() arguments}
   *    .processOutput(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.io.ProcessOutput&amp;gt;) // optional {@link Starter#processOutput() processOutput}
   *    .supportConfig(de.flapdoodle.reverse.StateID&amp;lt;de.flapdoodle.embed.process.config.SupportConfig&amp;gt;) // optional {@link Starter#supportConfig() supportConfig}
   *    .transitionLabel(String) // optional {@link Starter#transitionLabel() transitionLabel}
   *    .destination(de.flapdoodle.reverse.StateID&amp;lt;T&amp;gt;) // required {@link Starter#destination() destination}
   *    .runningProcessFactory(de.flapdoodle.embed.process.types.RunningProcessFactory&amp;lt;T&amp;gt;) // required {@link Starter#runningProcessFactory() runningProcessFactory}
   *    .build();
   * </pre>
   * @param <T> generic parameter T
   * @param runningProcessFactory {@code runningProcessFactory} parameter
   * @return A new ImmutableStarter builder
   */
  public static <T extends RunningProcess> ImmutableStarter.Builder<T> builder(RunningProcessFactory<T> runningProcessFactory) {
    return new ImmutableStarter.Builder<>(runningProcessFactory);
  }

  static <T extends RunningProcess> ImmutableStarter.Builder<T> builder() {
    return new ImmutableStarter.Builder<>();
  }

  /**
   * Builds instances of type {@link ImmutableStarter ImmutableStarter}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder<T extends RunningProcess> {
    private static final long INIT_BIT_DESTINATION = 0x1L;
    private static final long INIT_BIT_RUNNING_PROCESS_FACTORY = 0x2L;
    private long initBits = 0x3L;

    private StateID<ExtractedFileSet> processExecutable;
    private StateID<ProcessWorkingDir> processWorkingDir;
    private StateID<ProcessConfig> processConfig;
    private StateID<ProcessEnv> processEnv;
    private StateID<ProcessArguments> arguments;
    private StateID<ProcessOutput> processOutput;
    private StateID<SupportConfig> supportConfig;
    private String transitionLabel;
    private StateID<T> destination;
    private RunningProcessFactory<T> runningProcessFactory;

    private Builder(RunningProcessFactory<T> runningProcessFactory) {
      runningProcessFactory(runningProcessFactory);
    }

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.embed.process.transitions.Starter} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> from(Starter<T> instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    /**
     * Fill a builder with attribute values from the provided {@code de.flapdoodle.reverse.naming.HasLabel} instance.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> from(HasLabel instance) {
      Objects.requireNonNull(instance, "instance");
      from((short) 0, (Object) instance);
      return this;
    }

    @SuppressWarnings("unchecked")
    private void from(short _unused, Object object) {
      long bits = 0;
      if (object instanceof Starter<?>) {
        Starter<T> instance = (Starter<T>) object;
        supportConfig(instance.supportConfig());
        processExecutable(instance.processExecutable());
        runningProcessFactory(instance.runningProcessFactory());
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
        processConfig(instance.processConfig());
        processEnv(instance.processEnv());
        destination(instance.destination());
        processOutput(instance.processOutput());
        arguments(instance.arguments());
        processWorkingDir(instance.processWorkingDir());
      }
      if (object instanceof HasLabel) {
        HasLabel instance = (HasLabel) object;
        if ((bits & 0x1L) == 0) {
          transitionLabel(instance.transitionLabel());
          bits |= 0x1L;
        }
      }
    }

    /**
     * Initializes the value for the {@link Starter#processExecutable() processExecutable} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#processExecutable() processExecutable}.</em>
     * @param processExecutable The value for processExecutable 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> processExecutable(StateID<ExtractedFileSet> processExecutable) {
      this.processExecutable = Objects.requireNonNull(processExecutable, "processExecutable");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#processWorkingDir() processWorkingDir} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#processWorkingDir() processWorkingDir}.</em>
     * @param processWorkingDir The value for processWorkingDir 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> processWorkingDir(StateID<ProcessWorkingDir> processWorkingDir) {
      this.processWorkingDir = Objects.requireNonNull(processWorkingDir, "processWorkingDir");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#processConfig() processConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#processConfig() processConfig}.</em>
     * @param processConfig The value for processConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> processConfig(StateID<ProcessConfig> processConfig) {
      this.processConfig = Objects.requireNonNull(processConfig, "processConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#processEnv() processEnv} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#processEnv() processEnv}.</em>
     * @param processEnv The value for processEnv 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> processEnv(StateID<ProcessEnv> processEnv) {
      this.processEnv = Objects.requireNonNull(processEnv, "processEnv");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#arguments() arguments} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#arguments() arguments}.</em>
     * @param arguments The value for arguments 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> arguments(StateID<ProcessArguments> arguments) {
      this.arguments = Objects.requireNonNull(arguments, "arguments");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#processOutput() processOutput} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#processOutput() processOutput}.</em>
     * @param processOutput The value for processOutput 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> processOutput(StateID<ProcessOutput> processOutput) {
      this.processOutput = Objects.requireNonNull(processOutput, "processOutput");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#supportConfig() supportConfig} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#supportConfig() supportConfig}.</em>
     * @param supportConfig The value for supportConfig 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> supportConfig(StateID<SupportConfig> supportConfig) {
      this.supportConfig = Objects.requireNonNull(supportConfig, "supportConfig");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#transitionLabel() transitionLabel} attribute.
     * <p><em>If not set, this attribute will have a default value as returned by the initializer of {@link Starter#transitionLabel() transitionLabel}.</em>
     * @param transitionLabel The value for transitionLabel 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> transitionLabel(String transitionLabel) {
      this.transitionLabel = Objects.requireNonNull(transitionLabel, "transitionLabel");
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#destination() destination} attribute.
     * @param destination The value for destination 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder<T> destination(StateID<T> destination) {
      this.destination = Objects.requireNonNull(destination, "destination");
      initBits &= ~INIT_BIT_DESTINATION;
      return this;
    }

    /**
     * Initializes the value for the {@link Starter#runningProcessFactory() runningProcessFactory} attribute.
     * @param runningProcessFactory The value for runningProcessFactory 
     * @return {@code this} builder for use in a chained invocation
     */
    final Builder<T> runningProcessFactory(RunningProcessFactory<T> runningProcessFactory) {
      this.runningProcessFactory = Objects.requireNonNull(runningProcessFactory, "runningProcessFactory");
      initBits &= ~INIT_BIT_RUNNING_PROCESS_FACTORY;
      return this;
    }

    /**
     * Builds a new {@link ImmutableStarter ImmutableStarter}.
     * @return An immutable instance of Starter
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableStarter<T> build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableStarter<T>(this);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_DESTINATION) != 0) attributes.add("destination");
      if ((initBits & INIT_BIT_RUNNING_PROCESS_FACTORY) != 0) attributes.add("runningProcessFactory");
      return "Cannot build Starter, some of required attributes are not set " + attributes;
    }
  }
}
