/*
 * Decompiled with CFR 0.152.
 */
package io.druid.indexer;

import com.google.common.base.Supplier;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.ByteArrayDataInput;
import com.google.common.io.ByteArrayDataOutput;
import com.google.common.io.ByteStreams;
import io.druid.data.input.InputRow;
import io.druid.data.input.MapBasedInputRow;
import io.druid.data.input.Rows;
import io.druid.data.input.impl.DimensionSchema;
import io.druid.data.input.impl.DimensionsSpec;
import io.druid.java.util.common.IAE;
import io.druid.java.util.common.StringUtils;
import io.druid.java.util.common.logger.Logger;
import io.druid.java.util.common.parsers.ParseException;
import io.druid.query.aggregation.Aggregator;
import io.druid.query.aggregation.AggregatorFactory;
import io.druid.segment.DimensionHandlerUtils;
import io.druid.segment.VirtualColumns;
import io.druid.segment.column.ValueType;
import io.druid.segment.incremental.IncrementalIndex;
import io.druid.segment.serde.ComplexMetricSerde;
import io.druid.segment.serde.ComplexMetrics;
import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.io.WritableUtils;

public class InputRowSerde {
    private static final Logger log = new Logger(InputRowSerde.class);
    private static final IndexSerdeTypeHelper STRING_HELPER = new StringIndexSerdeTypeHelper();
    private static final IndexSerdeTypeHelper LONG_HELPER = new LongIndexSerdeTypeHelper();
    private static final IndexSerdeTypeHelper FLOAT_HELPER = new FloatIndexSerdeTypeHelper();
    private static final IndexSerdeTypeHelper DOUBLE_HELPER = new DoubleIndexSerdeTypeHelper();

    public static Map<String, IndexSerdeTypeHelper> getTypeHelperMap(DimensionsSpec dimensionsSpec) {
        HashMap typeHelperMap = Maps.newHashMap();
        for (DimensionSchema dimensionSchema : dimensionsSpec.getDimensions()) {
            IndexSerdeTypeHelper typeHelper;
            switch (dimensionSchema.getValueType()) {
                case STRING: {
                    typeHelper = STRING_HELPER;
                    break;
                }
                case LONG: {
                    typeHelper = LONG_HELPER;
                    break;
                }
                case FLOAT: {
                    typeHelper = FLOAT_HELPER;
                    break;
                }
                case DOUBLE: {
                    typeHelper = DOUBLE_HELPER;
                    break;
                }
                default: {
                    throw new IAE("Invalid type: [%s]", new Object[]{dimensionSchema.getValueType()});
                }
            }
            typeHelperMap.put(dimensionSchema.getName(), typeHelper);
        }
        return typeHelperMap;
    }

    public static final byte[] toBytes(Map<String, IndexSerdeTypeHelper> typeHelperMap, final InputRow row, AggregatorFactory[] aggs, boolean reportParseExceptions) {
        try {
            ByteArrayDataOutput out = ByteStreams.newDataOutput();
            out.writeLong(row.getTimestampFromEpoch());
            List dimList = row.getDimensions();
            WritableUtils.writeVInt((DataOutput)out, (int)dimList.size());
            if (dimList != null) {
                for (String dim : dimList) {
                    IndexSerdeTypeHelper typeHelper = typeHelperMap.get(dim);
                    if (typeHelper == null) {
                        typeHelper = STRING_HELPER;
                    }
                    InputRowSerde.writeString(dim, out);
                    typeHelper.serialize(out, row.getRaw(dim), reportParseExceptions);
                }
            }
            Supplier<InputRow> supplier = new Supplier<InputRow>(){

                public InputRow get() {
                    return row;
                }
            };
            WritableUtils.writeVInt((DataOutput)out, (int)aggs.length);
            for (AggregatorFactory aggFactory : aggs) {
                String k = aggFactory.getName();
                InputRowSerde.writeString(k, out);
                try (Aggregator agg = aggFactory.factorize(IncrementalIndex.makeColumnSelectorFactory((VirtualColumns)VirtualColumns.EMPTY, (AggregatorFactory)aggFactory, (Supplier)supplier, (boolean)true));){
                    try {
                        agg.aggregate();
                    }
                    catch (ParseException e) {
                        if (reportParseExceptions) {
                            throw new ParseException((Throwable)e, "Encountered parse error for aggregator[%s]", new Object[]{k});
                        }
                        log.debug((Throwable)e, "Encountered parse error, skipping aggregator[%s].", new Object[]{k});
                    }
                    String t = aggFactory.getTypeName();
                    if (t.equals("float")) {
                        out.writeFloat(agg.getFloat());
                        continue;
                    }
                    if (t.equals("long")) {
                        WritableUtils.writeVLong((DataOutput)out, (long)agg.getLong());
                        continue;
                    }
                    if (t.equals("double")) {
                        out.writeDouble(agg.getDouble());
                        continue;
                    }
                    Object val = agg.get();
                    ComplexMetricSerde serde = InputRowSerde.getComplexMetricSerde(t);
                    InputRowSerde.writeBytes(serde.toBytes(val), out);
                }
            }
            return out.toByteArray();
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    private static void writeBytes(byte[] value, ByteArrayDataOutput out) throws IOException {
        WritableUtils.writeVInt((DataOutput)out, (int)value.length);
        out.write(value, 0, value.length);
    }

    private static void writeString(String value, ByteArrayDataOutput out) throws IOException {
        InputRowSerde.writeBytes(StringUtils.toUtf8((String)value), out);
    }

    private static void writeStringArray(List<String> values, ByteArrayDataOutput out) throws IOException {
        if (values == null || values.size() == 0) {
            WritableUtils.writeVInt((DataOutput)out, (int)0);
            return;
        }
        WritableUtils.writeVInt((DataOutput)out, (int)values.size());
        for (String value : values) {
            InputRowSerde.writeString(value, out);
        }
    }

    private static String readString(DataInput in) throws IOException {
        byte[] result = InputRowSerde.readBytes(in);
        return StringUtils.fromUtf8((byte[])result);
    }

    private static byte[] readBytes(DataInput in) throws IOException {
        int size = WritableUtils.readVInt((DataInput)in);
        byte[] result = new byte[size];
        in.readFully(result, 0, size);
        return result;
    }

    private static List<String> readStringArray(DataInput in) throws IOException {
        int count = WritableUtils.readVInt((DataInput)in);
        if (count == 0) {
            return null;
        }
        ArrayList values = Lists.newArrayListWithCapacity((int)count);
        for (int i = 0; i < count; ++i) {
            values.add(InputRowSerde.readString(in));
        }
        return values;
    }

    public static final InputRow fromBytes(Map<String, IndexSerdeTypeHelper> typeHelperMap, byte[] data, AggregatorFactory[] aggs) {
        try {
            ByteArrayDataInput in = ByteStreams.newDataInput((byte[])data);
            long timestamp = in.readLong();
            HashMap event = Maps.newHashMap();
            ArrayList dimensions = Lists.newArrayList();
            int dimNum = WritableUtils.readVInt((DataInput)in);
            for (int i = 0; i < dimNum; ++i) {
                Object dimValues;
                String dimension = InputRowSerde.readString((DataInput)in);
                dimensions.add(dimension);
                IndexSerdeTypeHelper typeHelper = typeHelperMap.get(dimension);
                if (typeHelper == null) {
                    typeHelper = STRING_HELPER;
                }
                if ((dimValues = typeHelper.deserialize(in)) == null) continue;
                if (typeHelper.getType() == ValueType.STRING) {
                    List dimensionValues = (List)dimValues;
                    if (dimensionValues.size() == 1) {
                        event.put(dimension, dimensionValues.get(0));
                        continue;
                    }
                    event.put(dimension, dimensionValues);
                    continue;
                }
                event.put(dimension, dimValues);
            }
            int metricSize = WritableUtils.readVInt((DataInput)in);
            for (int i = 0; i < metricSize; ++i) {
                String metric = InputRowSerde.readString((DataInput)in);
                String type = InputRowSerde.getType(metric, aggs, i);
                if (type.equals("float")) {
                    event.put(metric, Float.valueOf(in.readFloat()));
                    continue;
                }
                if (type.equals("long")) {
                    event.put(metric, WritableUtils.readVLong((DataInput)in));
                    continue;
                }
                if (type.equals("double")) {
                    event.put(metric, in.readDouble());
                    continue;
                }
                ComplexMetricSerde serde = InputRowSerde.getComplexMetricSerde(type);
                byte[] value = InputRowSerde.readBytes((DataInput)in);
                event.put(metric, serde.fromBytes(value, 0, value.length));
            }
            return new MapBasedInputRow(timestamp, (List)dimensions, (Map)event);
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    private static String getType(String metric, AggregatorFactory[] aggs, int i) {
        if (aggs[i].getName().equals(metric)) {
            return aggs[i].getTypeName();
        }
        log.warn("Aggs disordered, fall backs to loop.", new Object[0]);
        for (AggregatorFactory agg : aggs) {
            if (!agg.getName().equals(metric)) continue;
            return agg.getTypeName();
        }
        return null;
    }

    private static ComplexMetricSerde getComplexMetricSerde(String type) {
        ComplexMetricSerde serde = ComplexMetrics.getSerdeForType((String)type);
        if (serde == null) {
            throw new IAE("Unknown type[%s]", new Object[]{type});
        }
        return serde;
    }

    public static class DoubleIndexSerdeTypeHelper
    implements IndexSerdeTypeHelper<Double> {
        @Override
        public ValueType getType() {
            return ValueType.DOUBLE;
        }

        @Override
        public void serialize(ByteArrayDataOutput out, Object value, boolean reportParseExceptions) {
            Double ret = DimensionHandlerUtils.convertObjectToDouble((Object)value, (boolean)reportParseExceptions);
            if (ret == null) {
                ret = DimensionHandlerUtils.ZERO_DOUBLE;
            }
            out.writeDouble(ret.doubleValue());
        }

        @Override
        public Double deserialize(ByteArrayDataInput in) {
            return in.readDouble();
        }
    }

    public static class FloatIndexSerdeTypeHelper
    implements IndexSerdeTypeHelper<Float> {
        @Override
        public ValueType getType() {
            return ValueType.FLOAT;
        }

        @Override
        public void serialize(ByteArrayDataOutput out, Object value, boolean reportParseExceptions) {
            Float ret = DimensionHandlerUtils.convertObjectToFloat((Object)value, (boolean)reportParseExceptions);
            if (ret == null) {
                ret = DimensionHandlerUtils.ZERO_FLOAT;
            }
            out.writeFloat(ret.floatValue());
        }

        @Override
        public Float deserialize(ByteArrayDataInput in) {
            return Float.valueOf(in.readFloat());
        }
    }

    public static class LongIndexSerdeTypeHelper
    implements IndexSerdeTypeHelper<Long> {
        @Override
        public ValueType getType() {
            return ValueType.LONG;
        }

        @Override
        public void serialize(ByteArrayDataOutput out, Object value, boolean reportParseExceptions) {
            Long ret = DimensionHandlerUtils.convertObjectToLong((Object)value, (boolean)reportParseExceptions);
            if (ret == null) {
                ret = DimensionHandlerUtils.ZERO_LONG;
            }
            out.writeLong(ret.longValue());
        }

        @Override
        public Long deserialize(ByteArrayDataInput in) {
            return in.readLong();
        }
    }

    public static class StringIndexSerdeTypeHelper
    implements IndexSerdeTypeHelper<List<String>> {
        @Override
        public ValueType getType() {
            return ValueType.STRING;
        }

        @Override
        public void serialize(ByteArrayDataOutput out, Object value, boolean reportParseExceptions) {
            List values = Rows.objectToStrings((Object)value);
            try {
                InputRowSerde.writeStringArray(values, out);
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        }

        @Override
        public List<String> deserialize(ByteArrayDataInput in) {
            try {
                return InputRowSerde.readStringArray((DataInput)in);
            }
            catch (IOException ioe) {
                throw new RuntimeException(ioe);
            }
        }
    }

    public static interface IndexSerdeTypeHelper<T> {
        public ValueType getType();

        public void serialize(ByteArrayDataOutput var1, Object var2, boolean var3);

        public T deserialize(ByteArrayDataInput var1);
    }
}

