/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.util;

import com.google.common.net.UrlEscapers;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import io.fabric8.maven.docker.access.AuthConfig;
import io.fabric8.maven.docker.access.ecr.EcrExtendedAuth;
import io.fabric8.maven.docker.util.CredentialHelperClient;
import io.fabric8.maven.docker.util.DockerFileUtil;
import io.fabric8.maven.docker.util.EnvUtil;
import io.fabric8.maven.docker.util.Logger;
import io.fabric8.maven.docker.util.aws.AwsSdkAuthConfigFactory;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.lang.reflect.Method;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.Settings;
import org.codehaus.plexus.PlexusContainer;
import org.codehaus.plexus.component.repository.exception.ComponentLookupException;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.sonatype.plexus.components.sec.dispatcher.SecDispatcher;

public class AuthConfigFactory {
    private static final String AUTH_USE_OPENSHIFT_AUTH = "useOpenShiftAuth";
    static final String DOCKER_LOGIN_DEFAULT_REGISTRY = "https://index.docker.io/v1/";
    private final PlexusContainer container;
    private Logger log;
    private static final String[] DEFAULT_REGISTRIES = new String[]{"docker.io", "index.docker.io", "registry.hub.docker.com"};

    public AuthConfigFactory(PlexusContainer container) {
        this.container = container;
    }

    public void setLog(Logger log) {
        this.log = log;
    }

    public AuthConfig createAuthConfig(boolean isPush, boolean skipExtendedAuth, Map authConfig, Settings settings, String user, String registry) throws MojoExecutionException {
        AuthConfig ret = this.createStandardAuthConfig(isPush, authConfig, settings, user, registry);
        if (ret != null) {
            if (registry == null || skipExtendedAuth) {
                return ret;
            }
            try {
                return this.extendedAuthentication(ret, registry);
            }
            catch (IOException e) {
                throw new MojoExecutionException(e.getMessage(), (Exception)e);
            }
        }
        ret = this.getAuthConfigFromDockerConfig(registry);
        if (ret != null) {
            this.log.debug("AuthConfig: credentials from ~/.docker/config.json", new Object[0]);
            return ret;
        }
        this.log.debug("AuthConfig: no credentials found", new Object[0]);
        return null;
    }

    private AuthConfig extendedAuthentication(AuthConfig standardAuthConfig, String registry) throws IOException, MojoExecutionException {
        EcrExtendedAuth ecr = new EcrExtendedAuth(this.log, registry);
        if (ecr.isAwsRegistry()) {
            return ecr.extendedAuth(standardAuthConfig);
        }
        return standardAuthConfig;
    }

    private AuthConfig createStandardAuthConfig(boolean isPush, Map authConfigMap, Settings settings, String user, String registry) throws MojoExecutionException {
        AuthConfig ret;
        for (LookupMode lookupMode : new LookupMode[]{this.getLookupMode(isPush), LookupMode.DEFAULT, LookupMode.REGISTRY}) {
            ret = this.getAuthConfigFromSystemProperties(lookupMode);
            if (ret != null) {
                this.log.debug("AuthConfig: credentials from system properties", new Object[0]);
                return ret;
            }
            if (lookupMode != LookupMode.REGISTRY && (ret = this.getAuthConfigFromOpenShiftConfig(lookupMode, authConfigMap)) != null) {
                this.log.debug("AuthConfig: OpenShift credentials", new Object[0]);
                return ret;
            }
            ret = this.getAuthConfigFromPluginConfiguration(lookupMode, authConfigMap);
            if (ret == null) continue;
            this.log.debug("AuthConfig: credentials from plugin config", new Object[0]);
            return ret;
        }
        ret = this.getAuthConfigFromSettings(settings, user, registry);
        if (ret != null) {
            this.log.debug("AuthConfig: credentials from ~/.m2/setting.xml", new Object[0]);
            return ret;
        }
        if (EcrExtendedAuth.isAwsRegistry(registry)) {
            ret = this.getAuthConfigViaAwsSdk();
            if (ret != null) {
                this.log.debug("AuthConfig: AWS credentials from AWS SDK", new Object[0]);
                return ret;
            }
            ret = this.getAuthConfigFromAwsEnvironmentVariables();
            if (ret != null) {
                this.log.debug("AuthConfig: AWS credentials from ENV variables", new Object[0]);
                return ret;
            }
            try {
                ret = this.getAuthConfigFromEC2InstanceRole();
            }
            catch (ConnectTimeoutException ex) {
                this.log.debug("Connection timeout while retrieving instance meta-data, likely not an EC2 instance (%s)", ex.getMessage());
            }
            catch (IOException ex) {
                this.log.warn("Error while retrieving EC2 instance credentials: %s", ex.getMessage());
            }
            if (ret != null) {
                this.log.debug("AuthConfig: credentials from EC2 instance role", new Object[0]);
                return ret;
            }
            try {
                ret = this.getAuthConfigFromTaskRole();
            }
            catch (ConnectTimeoutException ex) {
                this.log.debug("Connection timeout while retrieving ECS meta-data, likely not an ECS instance (%s)", ex.getMessage());
            }
            catch (IOException ex) {
                this.log.warn("Error while retrieving ECS Task role credentials: %s", ex.getMessage());
            }
            if (ret != null) {
                this.log.debug("AuthConfig: credentials from ECS Task role", new Object[0]);
                return ret;
            }
        }
        return null;
    }

    private AuthConfig getAuthConfigViaAwsSdk() {
        try {
            Class.forName("com.amazonaws.auth.DefaultAWSCredentialsProviderChain");
        }
        catch (ClassNotFoundException e) {
            this.log.info("It appears that you're using AWS ECR. Consider integrating the AWS SDK in order to make use of common AWS authentication mechanisms, see https://dmp.fabric8.io/#extended-authentication", new Object[0]);
            return null;
        }
        return new AwsSdkAuthConfigFactory(this.log).createAuthConfig();
    }

    private AuthConfig getAuthConfigFromAwsEnvironmentVariables() {
        String accessKeyId = System.getenv("AWS_ACCESS_KEY_ID");
        if (accessKeyId == null) {
            this.log.debug("System environment not set for variable AWS_ACCESS_KEY_ID, no AWS credentials found", new Object[0]);
            return null;
        }
        String secretAccessKey = System.getenv("AWS_SECRET_ACCESS_KEY");
        if (secretAccessKey == null) {
            this.log.warn("System environment set for variable AWS_ACCESS_KEY_ID, but NOT for variable AWS_SECRET_ACCESS_KEY!", new Object[0]);
            return null;
        }
        return new AuthConfig(accessKeyId, secretAccessKey, "none", System.getenv("AWS_SESSION_TOKEN"));
    }

    private AuthConfig getAuthConfigFromEC2InstanceRole() throws IOException {
        this.log.debug("No user and password set for ECR, checking EC2 instance role", new Object[0]);
        try (CloseableHttpClient client = HttpClients.custom().useSystemProperties().build();){
            String instanceRole;
            RequestConfig conf = RequestConfig.custom().setConnectionRequestTimeout(1000).setConnectTimeout(1000).setSocketTimeout(1000).build();
            HttpGet request = new HttpGet("http://169.254.169.254/latest/meta-data/iam/security-credentials");
            request.setConfig(conf);
            try (CloseableHttpResponse response = client.execute((HttpUriRequest)request);){
                if (response.getStatusLine().getStatusCode() != 200) {
                    this.log.debug("No instance role found, return code was %d", response.getStatusLine().getStatusCode());
                    AuthConfig authConfig = null;
                    return authConfig;
                }
                try (InputStream is = response.getEntity().getContent();){
                    instanceRole = IOUtils.toString((InputStream)is, (Charset)StandardCharsets.UTF_8);
                }
            }
            this.log.debug("Found instance role %s, getting temporary security credentials", instanceRole);
            request = new HttpGet("http://169.254.169.254/latest/meta-data/iam/security-credentials/" + UrlEscapers.urlPathSegmentEscaper().escape(instanceRole));
            request.setConfig(conf);
            AuthConfig authConfig = this.readAwsCredentials(client, request);
            return authConfig;
        }
    }

    private AuthConfig getAuthConfigFromTaskRole() throws IOException {
        this.log.debug("No user and password set for ECR, checking ECS Task role", new Object[0]);
        URI uri = this.getMetadataEndpointForCredentials();
        if (uri == null) {
            return null;
        }
        this.log.debug("Getting temporary security credentials from: %s", uri);
        try (CloseableHttpClient client = HttpClients.custom().useSystemProperties().build();){
            RequestConfig conf = RequestConfig.custom().setConnectionRequestTimeout(1000).setConnectTimeout(1000).setSocketTimeout(1000).build();
            HttpGet request = new HttpGet(uri);
            request.setConfig(conf);
            AuthConfig authConfig = this.readAwsCredentials(client, request);
            return authConfig;
        }
    }

    private AuthConfig readAwsCredentials(CloseableHttpClient client, HttpGet request) throws IOException {
        try (CloseableHttpResponse response = client.execute((HttpUriRequest)request);){
            AuthConfig authConfig;
            if (response.getStatusLine().getStatusCode() != 200) {
                this.log.debug("No security credential found, return code was %d", response.getStatusLine().getStatusCode());
                AuthConfig authConfig2 = null;
                return authConfig2;
            }
            try (InputStreamReader r = new InputStreamReader(response.getEntity().getContent(), StandardCharsets.UTF_8);){
                JsonObject securityCredentials = (JsonObject)new Gson().fromJson((Reader)r, JsonObject.class);
                String user = securityCredentials.getAsJsonPrimitive("AccessKeyId").getAsString();
                String password = securityCredentials.getAsJsonPrimitive("SecretAccessKey").getAsString();
                String token = securityCredentials.getAsJsonPrimitive("Token").getAsString();
                this.log.debug("Received temporary access key %s...", user.substring(0, 8));
                authConfig = new AuthConfig(user, password, "none", token);
            }
            return authConfig;
        }
    }

    private URI getMetadataEndpointForCredentials() {
        String ecsMetadataEndpoint;
        String awsContainerCredentialsUri = System.getenv("AWS_CONTAINER_CREDENTIALS_RELATIVE_URI");
        if (awsContainerCredentialsUri == null) {
            this.log.debug("System environment not set for variable AWS_CONTAINER_CREDENTIALS_RELATIVE_URI, no task role found", new Object[0]);
            return null;
        }
        if (awsContainerCredentialsUri.charAt(0) != '/') {
            awsContainerCredentialsUri = "/" + awsContainerCredentialsUri;
        }
        if ((ecsMetadataEndpoint = System.getenv("ECS_METADATA_ENDPOINT")) == null) {
            ecsMetadataEndpoint = "http://169.254.170.2";
        }
        try {
            return new URI(ecsMetadataEndpoint + awsContainerCredentialsUri);
        }
        catch (URISyntaxException e) {
            this.log.warn("Failed to construct path to ECS metadata endpoint for credentials", e);
            return null;
        }
    }

    private AuthConfig getAuthConfigFromSystemProperties(LookupMode lookupMode) throws MojoExecutionException {
        Properties props = System.getProperties();
        String userKey = lookupMode.asSysProperty("username");
        String passwordKey = lookupMode.asSysProperty("password");
        if (props.containsKey(userKey)) {
            if (!props.containsKey(passwordKey)) {
                throw new MojoExecutionException("No " + passwordKey + " provided for username " + props.getProperty(userKey));
            }
            return new AuthConfig(props.getProperty(userKey), this.decrypt(props.getProperty(passwordKey)), props.getProperty(lookupMode.asSysProperty("email")), this.getAuthProperty(props, lookupMode));
        }
        return null;
    }

    private String getAuthProperty(Properties props, LookupMode lookupMode) {
        String authProp = props.getProperty(lookupMode.asSysProperty("auth"));
        if (authProp != null) {
            return authProp;
        }
        return props.getProperty(lookupMode.asSysProperty("authToken"));
    }

    private AuthConfig getAuthConfigFromOpenShiftConfig(LookupMode lookupMode, Map authConfigMap) throws MojoExecutionException {
        String useOpenAuthModeProp;
        Properties props = System.getProperties();
        if (props.containsKey(useOpenAuthModeProp = lookupMode.asSysProperty(AUTH_USE_OPENSHIFT_AUTH))) {
            boolean useOpenShift = Boolean.valueOf(props.getProperty(useOpenAuthModeProp));
            if (useOpenShift) {
                return this.validateMandatoryOpenShiftLogin(this.parseOpenShiftConfig(), useOpenAuthModeProp);
            }
            return null;
        }
        Map mapToCheck = this.getAuthConfigMapToCheck(lookupMode, authConfigMap);
        if (mapToCheck != null && mapToCheck.containsKey(AUTH_USE_OPENSHIFT_AUTH) && Boolean.valueOf((String)mapToCheck.get(AUTH_USE_OPENSHIFT_AUTH)).booleanValue()) {
            return this.validateMandatoryOpenShiftLogin(this.parseOpenShiftConfig(), useOpenAuthModeProp);
        }
        return null;
    }

    private AuthConfig getAuthConfigFromPluginConfiguration(LookupMode lookupMode, Map authConfig) throws MojoExecutionException {
        Map mapToCheck = this.getAuthConfigMapToCheck(lookupMode, authConfig);
        if (mapToCheck != null && mapToCheck.containsKey("username")) {
            if (!mapToCheck.containsKey("password")) {
                throw new MojoExecutionException("No 'password' given while using <authConfig> in configuration for mode " + (Object)((Object)lookupMode));
            }
            HashMap<String, String> cloneConfig = new HashMap<String, String>(mapToCheck);
            cloneConfig.put("password", this.decrypt((String)cloneConfig.get("password")));
            return new AuthConfig(cloneConfig);
        }
        return null;
    }

    private AuthConfig getAuthConfigFromSettings(Settings settings, String user, String registry) throws MojoExecutionException {
        Server defaultServer = null;
        for (Server server : settings.getServers()) {
            Server found;
            String id = server.getId();
            if (defaultServer == null) {
                defaultServer = this.checkForServer(server, id, registry, null);
            }
            if ((found = this.checkForServer(server, id, registry, user)) == null) continue;
            return this.createAuthConfigFromServer(found);
        }
        return defaultServer != null ? this.createAuthConfigFromServer(defaultServer) : null;
    }

    private AuthConfig getAuthConfigFromDockerConfig(String registry) throws MojoExecutionException {
        String registryToLookup;
        JsonObject dockerConfig = DockerFileUtil.readDockerConfig();
        if (dockerConfig == null) {
            return null;
        }
        String string = registryToLookup = registry != null ? registry : DOCKER_LOGIN_DEFAULT_REGISTRY;
        if (dockerConfig.has("credHelpers") || dockerConfig.has("credsStore")) {
            JsonObject credHelpers;
            if (dockerConfig.has("credHelpers") && (credHelpers = dockerConfig.getAsJsonObject("credHelpers")).has(registryToLookup)) {
                return this.extractAuthConfigFromCredentialsHelper(registryToLookup, credHelpers.get(registryToLookup).getAsString());
            }
            if (dockerConfig.has("credsStore")) {
                return this.extractAuthConfigFromCredentialsHelper(registryToLookup, dockerConfig.get("credsStore").getAsString());
            }
        }
        if (dockerConfig.has("auths")) {
            return this.extractAuthConfigFromDockerConfigAuths(registryToLookup, dockerConfig.getAsJsonObject("auths"));
        }
        return null;
    }

    private AuthConfig extractAuthConfigFromDockerConfigAuths(String registryToLookup, JsonObject auths) {
        JsonObject credentials = this.getCredentialsNode(auths, registryToLookup);
        if (credentials == null || !credentials.has("auth")) {
            return null;
        }
        String auth = credentials.get("auth").getAsString();
        String identityToken = credentials.has("identitytoken") ? credentials.get("identitytoken").getAsString() : null;
        String email = credentials.has("email") && !credentials.get("email").isJsonNull() ? credentials.get("email").getAsString() : null;
        return new AuthConfig(auth, email, identityToken);
    }

    private AuthConfig extractAuthConfigFromCredentialsHelper(String registryToLookup, String credConfig) throws MojoExecutionException {
        CredentialHelperClient credentialHelper = new CredentialHelperClient(this.log, credConfig);
        String version = credentialHelper.getVersion();
        this.log.debug("AuthConfig: credentials from credential helper/store %s%s", credentialHelper.getName(), version != null ? " version " + version : "");
        return credentialHelper.getAuthConfig(registryToLookup);
    }

    private JsonObject getCredentialsNode(JsonObject auths, String registryToLookup) {
        if (auths.has(registryToLookup)) {
            return auths.getAsJsonObject(registryToLookup);
        }
        String registryWithScheme = EnvUtil.ensureRegistryHttpUrl(registryToLookup);
        if (auths.has(registryWithScheme)) {
            return auths.getAsJsonObject(registryWithScheme);
        }
        return null;
    }

    private Map getAuthConfigMapToCheck(LookupMode lookupMode, Map authConfigMap) {
        String configMapKey = lookupMode.getConfigMapKey();
        if (configMapKey == null) {
            return authConfigMap;
        }
        if (authConfigMap != null) {
            return (Map)authConfigMap.get(configMapKey);
        }
        return null;
    }

    private AuthConfig parseOpenShiftConfig() {
        Map<String, ?> kubeConfig = DockerFileUtil.readKubeConfig();
        if (kubeConfig == null) {
            return null;
        }
        String currentContextName = (String)kubeConfig.get("current-context");
        if (currentContextName == null) {
            return null;
        }
        for (Map contextMap : (List)kubeConfig.get("contexts")) {
            if (!currentContextName.equals(contextMap.get("name"))) continue;
            return this.parseContext(kubeConfig, (Map)contextMap.get("context"));
        }
        return null;
    }

    private AuthConfig parseContext(Map kubeConfig, Map context) {
        if (context == null) {
            return null;
        }
        String userName = (String)context.get("user");
        if (userName == null) {
            return null;
        }
        List users = (List)kubeConfig.get("users");
        if (users == null) {
            return null;
        }
        for (Map userMap : users) {
            if (!userName.equals(userMap.get("name"))) continue;
            return this.parseUser(userName, (Map)userMap.get("user"));
        }
        return null;
    }

    private AuthConfig parseUser(String userName, Map user) {
        if (user == null) {
            return null;
        }
        String token = (String)user.get("token");
        if (token == null) {
            return null;
        }
        Matcher matcher = Pattern.compile("^([^/]+).*$").matcher(userName);
        return new AuthConfig(matcher.matches() ? matcher.group(1) : userName, token, null, null);
    }

    private AuthConfig validateMandatoryOpenShiftLogin(AuthConfig openShiftAuthConfig, String useOpenAuthModeProp) throws MojoExecutionException {
        if (openShiftAuthConfig != null) {
            return openShiftAuthConfig;
        }
        String kubeConfigEnv = System.getenv("KUBECONFIG");
        throw new MojoExecutionException(String.format("System property %s set, but not active user and/or token found in %s. Please use 'oc login' for connecting to OpenShift.", useOpenAuthModeProp, kubeConfigEnv != null ? kubeConfigEnv : "~/.kube/config"));
    }

    private Server checkForServer(Server server, String id, String registry, String user) {
        String[] registries;
        String[] stringArray;
        if (registry != null) {
            String[] stringArray2 = new String[1];
            stringArray = stringArray2;
            stringArray2[0] = registry;
        } else {
            stringArray = DEFAULT_REGISTRIES;
        }
        for (String reg : registries = stringArray) {
            if (!id.equals(user == null ? reg : reg + "/" + user)) continue;
            return server;
        }
        return null;
    }

    private String decrypt(String password) throws MojoExecutionException {
        try {
            Object secDispatcher = this.container.lookup(SecDispatcher.ROLE, "maven");
            Method method = secDispatcher.getClass().getMethod("decrypt", String.class);
            return (String)method.invoke(secDispatcher, password);
        }
        catch (ComponentLookupException e) {
            throw new MojoExecutionException("Error looking security dispatcher", (Exception)((Object)e));
        }
        catch (ReflectiveOperationException e) {
            throw new MojoExecutionException("Cannot decrypt password: " + e.getCause(), (Exception)e);
        }
    }

    private AuthConfig createAuthConfigFromServer(Server server) throws MojoExecutionException {
        return new AuthConfig(server.getUsername(), this.decrypt(server.getPassword()), this.extractFromServerConfiguration(server.getConfiguration(), "email"), this.extractFromServerConfiguration(server.getConfiguration(), "auth"));
    }

    private String extractFromServerConfiguration(Object configuration, String prop) {
        Xpp3Dom dom;
        Xpp3Dom element;
        if (configuration != null && (element = (dom = (Xpp3Dom)configuration).getChild(prop)) != null) {
            return element.getValue();
        }
        return null;
    }

    private LookupMode getLookupMode(boolean isPush) {
        return isPush ? LookupMode.PUSH : LookupMode.PULL;
    }

    private static enum LookupMode {
        PUSH("docker.push.", "push"),
        PULL("docker.pull.", "pull"),
        REGISTRY("registry.", null),
        DEFAULT("docker.", null);

        private final String sysPropPrefix;
        private String configMapKey;

        private LookupMode(String sysPropPrefix, String configMapKey) {
            this.sysPropPrefix = sysPropPrefix;
            this.configMapKey = configMapKey;
        }

        public String asSysProperty(String prop) {
            return this.sysPropPrefix + prop;
        }

        public String getConfigMapKey() {
            return this.configMapKey;
        }
    }
}

