/*
 * Decompiled with CFR 0.152.
 */
package io.fabric8.maven.docker.util;

import com.google.cloud.tools.jib.api.CacheDirectoryCreationException;
import com.google.cloud.tools.jib.api.Containerizer;
import com.google.cloud.tools.jib.api.Credential;
import com.google.cloud.tools.jib.api.InvalidImageReferenceException;
import com.google.cloud.tools.jib.api.Jib;
import com.google.cloud.tools.jib.api.JibContainerBuilder;
import com.google.cloud.tools.jib.api.LogEvent;
import com.google.cloud.tools.jib.api.RegistryException;
import com.google.cloud.tools.jib.api.RegistryImage;
import com.google.cloud.tools.jib.api.TarImage;
import com.google.cloud.tools.jib.api.buildplan.AbsoluteUnixPath;
import com.google.cloud.tools.jib.api.buildplan.FileEntriesLayer;
import com.google.cloud.tools.jib.api.buildplan.ImageFormat;
import com.google.cloud.tools.jib.api.buildplan.Port;
import com.google.cloud.tools.jib.event.events.ProgressEvent;
import com.google.cloud.tools.jib.event.progress.ProgressEventHandler;
import io.fabric8.maven.docker.assembly.AssemblyFiles;
import io.fabric8.maven.docker.config.Arguments;
import io.fabric8.maven.docker.config.BuildImageConfiguration;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.docker.util.ImageName;
import io.fabric8.maven.docker.util.Logger;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.fusesource.jansi.Ansi;

public class JibServiceUtil {
    private static final String HEADER = "Executing tasks:";
    private static final int PROGRESS_BAR_COUNT = 30;
    public static final String JIB_LOG_PREFIX = "JIB> ";
    private static final long JIB_EXECUTOR_SHUTDOWN_TIMEOUT_SECONDS = 10L;
    protected static final String BUSYBOX = "busybox:latest";

    private JibServiceUtil() {
    }

    public static void buildContainer(JibContainerBuilder jibContainerBuilder, TarImage image, Logger logger) throws InterruptedException {
        ExecutorService jibBuildExecutor = Executors.newCachedThreadPool();
        try {
            jibContainerBuilder.setCreationTime(Instant.now());
            jibContainerBuilder.containerize(Containerizer.to((TarImage)image).setAllowInsecureRegistries(true).setExecutorService(jibBuildExecutor).addEventHandler(LogEvent.class, JibServiceUtil.log(logger)).addEventHandler(ProgressEvent.class, (Consumer)new ProgressEventHandler(JibServiceUtil.logUpdate())));
            JibServiceUtil.logUpdateFinished();
        }
        catch (CacheDirectoryCreationException | RegistryException | IOException | ExecutionException ex) {
            logger.error("Unable to build the image tarball: ", ex);
            throw new IllegalStateException(ex);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw ex;
        }
        finally {
            jibBuildExecutor.shutdown();
            jibBuildExecutor.awaitTermination(10L, TimeUnit.SECONDS);
        }
    }

    public static JibContainerBuilder containerFromImageConfiguration(String jibImageFormat, ImageConfiguration imageConfiguration, Credential pullRegistryCredential) throws InvalidImageReferenceException {
        JibContainerBuilder containerBuilder = Jib.from((RegistryImage)JibServiceUtil.getRegistryImage(JibServiceUtil.getBaseImage(imageConfiguration), pullRegistryCredential)).setFormat(JibServiceUtil.getImageFormat(jibImageFormat));
        return JibServiceUtil.populateContainerBuilderFromImageConfiguration(containerBuilder, imageConfiguration);
    }

    public static String getFullImageName(ImageConfiguration imageConfiguration, String tag) {
        ImageName imageName = tag != null ? new ImageName(imageConfiguration.getName(), tag) : new ImageName(imageConfiguration.getName());
        return imageName.getFullName();
    }

    public static void jibPush(ImageConfiguration imageConfiguration, Credential pushCredentials, File tarArchive, Logger log) {
        BuildImageConfiguration buildImageConfiguration = imageConfiguration.getBuildConfiguration();
        String imageName = JibServiceUtil.getFullImageName(imageConfiguration, null);
        try {
            for (String tag : JibServiceUtil.getAllImageTags(buildImageConfiguration.getTags(), imageName)) {
                String imageNameWithTag = JibServiceUtil.getFullImageName(imageConfiguration, tag);
                log.info("Pushing image: %s", imageNameWithTag);
                JibServiceUtil.pushImage(TarImage.at((Path)tarArchive.toPath()), imageNameWithTag, pushCredentials, log);
            }
        }
        catch (IllegalStateException e) {
            log.error("Exception occurred while pushing the image: %s", imageConfiguration.getName());
            throw e;
        }
        catch (InterruptedException e) {
            log.error("Thread interrupted", e);
            Thread.currentThread().interrupt();
        }
    }

    private static void pushImage(TarImage baseImage, String targetImageName, Credential credential, Logger logger) throws InterruptedException {
        ExecutorService jibBuildExecutor = Executors.newCachedThreadPool();
        try {
            JibServiceUtil.submitPushToJib(baseImage, JibServiceUtil.getRegistryImage(targetImageName, credential), jibBuildExecutor, logger);
        }
        catch (CacheDirectoryCreationException | InvalidImageReferenceException | RegistryException | IOException | ExecutionException e) {
            logger.error("Exception occurred while pushing the image: %s, %s", targetImageName, e.getMessage());
            throw new IllegalStateException(e.getMessage(), e);
        }
        catch (InterruptedException ex) {
            logger.error("Thread interrupted", ex);
            throw ex;
        }
        finally {
            jibBuildExecutor.shutdown();
            jibBuildExecutor.awaitTermination(10L, TimeUnit.SECONDS);
        }
    }

    private static JibContainerBuilder populateContainerBuilderFromImageConfiguration(JibContainerBuilder containerBuilder, ImageConfiguration imageConfiguration) {
        Optional<BuildImageConfiguration> bic = Optional.ofNullable(Objects.requireNonNull(imageConfiguration).getBuildConfiguration());
        bic.map(BuildImageConfiguration::getEntryPoint).map(Arguments::asStrings).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setEntrypoint(arg_0));
        bic.map(BuildImageConfiguration::getEnv).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setEnvironment(arg_0));
        bic.map(BuildImageConfiguration::getPorts).map(Collection::stream).map(s -> s.map(Integer::parseInt).map(Port::tcp)).map(s -> s.collect(Collectors.toSet())).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setExposedPorts(arg_0));
        bic.map(BuildImageConfiguration::getLabels).map(Map::entrySet).ifPresent(labels -> labels.forEach(l -> {
            if (l.getKey() != null && l.getValue() != null) {
                containerBuilder.addLabel((String)l.getKey(), (String)l.getValue());
            }
        }));
        bic.map(BuildImageConfiguration::getCmd).map(Arguments::asStrings).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setProgramArguments(arg_0));
        bic.map(BuildImageConfiguration::getUser).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setUser(arg_0));
        bic.map(BuildImageConfiguration::getVolumes).map(Collection::stream).map(s -> s.map(AbsoluteUnixPath::get)).map(s -> s.collect(Collectors.toSet())).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setVolumes(arg_0));
        bic.map(BuildImageConfiguration::getWorkdir).filter(((Predicate<String>)String::isEmpty).negate()).map(AbsoluteUnixPath::get).ifPresent(arg_0 -> ((JibContainerBuilder)containerBuilder).setWorkingDirectory(arg_0));
        return containerBuilder;
    }

    static Set<String> getAllImageTags(List<String> tags, String imageName) {
        ImageName tempImage = new ImageName(imageName);
        Set<String> tagSet = tags.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (!tempImage.getTag().isEmpty()) {
            tagSet.add(tempImage.getTag());
        }
        return tagSet;
    }

    static ImageFormat getImageFormat(String jibImageFormat) {
        if (jibImageFormat != null && jibImageFormat.equalsIgnoreCase("oci")) {
            return ImageFormat.OCI;
        }
        return ImageFormat.Docker;
    }

    private static void submitPushToJib(TarImage baseImage, RegistryImage targetImage, ExecutorService jibBuildExecutor, Logger logger) throws InterruptedException, ExecutionException, RegistryException, CacheDirectoryCreationException, IOException {
        Jib.from((TarImage)baseImage).setCreationTime(Instant.now()).containerize(Containerizer.to((RegistryImage)targetImage).setAllowInsecureRegistries(true).setExecutorService(jibBuildExecutor).addEventHandler(LogEvent.class, JibServiceUtil.log(logger)).addEventHandler(ProgressEvent.class, (Consumer)new ProgressEventHandler(JibServiceUtil.logUpdate())));
        JibServiceUtil.logUpdateFinished();
    }

    private static RegistryImage getRegistryImage(String targetImage, Credential credential) throws InvalidImageReferenceException {
        RegistryImage registryImage = RegistryImage.named((String)targetImage);
        if (credential != null && !credential.getUsername().isEmpty() && !credential.getPassword().isEmpty()) {
            registryImage.addCredential(credential.getUsername(), credential.getPassword());
        }
        return registryImage;
    }

    private static Consumer<LogEvent> log(Logger logger) {
        return le -> {
            if (le.getLevel() != LogEvent.Level.DEBUG || logger.isVerboseEnabled() || logger.isDebugEnabled()) {
                System.out.println(Ansi.ansi().cursorUpLine(1).eraseLine().a(JIB_LOG_PREFIX).a(StringUtils.rightPad((String)le.getMessage(), (int)120)).a("\n"));
            }
        };
    }

    private static Consumer<ProgressEventHandler.Update> logUpdate() {
        return update -> {
            List<String> progressDisplay = JibServiceUtil.generateProgressDisplay(update.getProgress(), (List<String>)update.getUnfinishedLeafTasks());
            if (progressDisplay.size() > 2 && progressDisplay.stream().allMatch(Objects::nonNull)) {
                String progressBar = progressDisplay.get(1);
                String task = progressDisplay.get(2);
                System.out.println(Ansi.ansi().cursorUpLine(1).eraseLine().a(JIB_LOG_PREFIX).a(progressBar).a(" ").a(task));
            }
        };
    }

    private static void logUpdateFinished() {
        System.out.println(JIB_LOG_PREFIX + JibServiceUtil.generateProgressBar(1.0));
    }

    public static String getBaseImage(ImageConfiguration imageConfiguration) {
        return Optional.ofNullable(imageConfiguration).map(ImageConfiguration::getBuildConfiguration).map(BuildImageConfiguration::getFrom).filter(((Predicate<String>)String::isEmpty).negate()).orElse(BUSYBOX);
    }

    public static void copyToContainer(final JibContainerBuilder containerBuilder, File directory, final String targetDir, Map<File, AssemblyFiles.Entry> files) throws IOException {
        Files.walkFileTree(directory.toPath(), (FileVisitor<? super Path>)new FileVisitor<Path>(){
            boolean notParentDir = false;

            @Override
            public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                if (!this.notParentDir) {
                    this.notParentDir = true;
                    return FileVisitResult.CONTINUE;
                }
                String fileFullpath = dir.toAbsolutePath().toString();
                String relativePath = fileFullpath.substring(targetDir.length());
                AbsoluteUnixPath absoluteUnixPath = AbsoluteUnixPath.fromPath((Path)Paths.get(relativePath, new String[0]));
                containerBuilder.addFileEntriesLayer(FileEntriesLayer.builder().addEntryRecursive(dir, absoluteUnixPath).build());
                return FileVisitResult.SKIP_SUBTREE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                String fileFullpath = file.toAbsolutePath().toString();
                String relativePath = fileFullpath.substring(targetDir.length());
                AbsoluteUnixPath absoluteUnixPath = AbsoluteUnixPath.fromPath((Path)Paths.get(relativePath, new String[0]));
                containerBuilder.addFileEntriesLayer(FileEntriesLayer.builder().addEntryRecursive(file, absoluteUnixPath).build());
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                if (exc != null) {
                    throw new IOException(exc);
                }
                return FileVisitResult.TERMINATE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                if (exc != null) {
                    throw new IOException(exc);
                }
                return FileVisitResult.CONTINUE;
            }
        });
    }

    private static List<String> generateProgressDisplay(double progress, List<String> unfinishedLeafTasks) {
        ArrayList<String> lines = new ArrayList<String>();
        lines.add(HEADER);
        lines.add(JibServiceUtil.generateProgressBar(progress));
        for (String task : unfinishedLeafTasks) {
            lines.add("> " + task);
        }
        return lines;
    }

    private static String generateProgressBar(double progress) {
        StringBuilder progressBar = new StringBuilder();
        progressBar.append('[');
        int barsToDisplay = (int)Math.round(30.0 * progress);
        for (int barIndex = 0; barIndex < 30; ++barIndex) {
            progressBar.append(barIndex < barsToDisplay ? (char)'=' : ' ');
        }
        return progressBar.append(']').append(String.format(" %.1f", progress * 100.0)).append("% complete").toString();
    }
}

