/**
 * Copyright 2016 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version
 * 2.0 (the "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied.  See the License for the specific language governing
 * permissions and limitations under the License.
 */
package io.fabric8.maven.enricher.standard;

import io.fabric8.kubernetes.api.builder.TypedVisitor;
import io.fabric8.kubernetes.api.model.KubernetesListBuilder;
import io.fabric8.kubernetes.api.model.PodSpec;
import io.fabric8.kubernetes.api.model.PodSpecBuilder;
import io.fabric8.kubernetes.api.model.apps.DeploymentBuilder;
import io.fabric8.kubernetes.api.model.apps.DeploymentFluent;
import io.fabric8.kubernetes.api.model.apps.DeploymentSpec;
import io.fabric8.kubernetes.api.model.apps.StatefulSetBuilder;
import io.fabric8.kubernetes.api.model.apps.StatefulSetFluent;
import io.fabric8.kubernetes.api.model.apps.StatefulSetSpec;
import io.fabric8.maven.core.config.PlatformMode;
import io.fabric8.maven.core.config.ResourceConfig;
import io.fabric8.maven.core.handler.DeploymentHandler;
import io.fabric8.maven.core.handler.HandlerHub;
import io.fabric8.maven.core.handler.StatefulSetHandler;
import io.fabric8.maven.core.util.Configs;
import io.fabric8.maven.core.util.MavenUtil;
import io.fabric8.maven.core.util.kubernetes.KubernetesResourceUtil;
import io.fabric8.maven.docker.config.ImageConfiguration;
import io.fabric8.maven.enricher.api.BaseEnricher;
import io.fabric8.maven.enricher.api.MavenEnricherContext;

import java.util.Collections;
import java.util.List;

public class ControllerViaPluginConfigurationEnricher extends BaseEnricher {
    protected static final String[] POD_CONTROLLER_KINDS =
            { "ReplicationController", "ReplicaSet", "Deployment", "DeploymentConfig", "StatefulSet", "DaemonSet", "Job" };

    private final DeploymentHandler deployHandler;
    private final StatefulSetHandler statefulSetHandler;

    // Available configuration keys
    private enum Config implements Configs.Key {
        name,
        pullPolicy           {{ d = "IfNotPresent"; }},
        type                 {{ d = "deployment"; }},
        replicaCount         {{ d = "1"; }};

        public String def() { return d; } protected String d;
    }

    public ControllerViaPluginConfigurationEnricher(MavenEnricherContext context) {
        super(context, "fmp-controller-from-configuration");
        HandlerHub handlers = new HandlerHub(
                getContext().getGav(), getContext().getConfiguration().getProperties());
        deployHandler = handlers.getDeploymentHandler();
        statefulSetHandler = handlers.getStatefulSetHandler();
    }

    @Override
    public void create(PlatformMode platformMode, KubernetesListBuilder builder) {
        final String name = getConfig(Config.name, MavenUtil.createDefaultResourceName(getContext().getGav().getSanitizedArtifactId()));
        final ResourceConfig config = new ResourceConfig.Builder()
                .controllerName(name)
                .imagePullPolicy(getConfig(Config.pullPolicy))
                .withReplicas(Configs.asInt(getConfig(Config.replicaCount)))
                .build();

        final List<ImageConfiguration> images = getImages().orElse(Collections.emptyList());

        // Check if at least a replica set is added. If not add a default one
        if (KubernetesResourceUtil.checkForKind(builder, POD_CONTROLLER_KINDS)) {
            // At least one image must be present, otherwise the resulting config will be invalid
            if (KubernetesResourceUtil.checkForKind(builder, "StatefulSet")) {
                final StatefulSetSpec spec = statefulSetHandler.getStatefulSet(config, images).getSpec();
                if (spec != null) {
                    builder.accept(new TypedVisitor<StatefulSetBuilder>() {
                        @Override
                        public void visit(StatefulSetBuilder statefulSetBuilder) {
                            statefulSetBuilder.editOrNewSpec().editOrNewTemplate().editOrNewSpec().endSpec().endTemplate().endSpec();
                            mergeStatefulSetSpec(statefulSetBuilder, spec);
                        }
                    });

                    if (spec.getTemplate() != null && spec.getTemplate().getSpec() != null) {
                        final PodSpec podSpec = spec.getTemplate().getSpec();
                        builder.accept(new TypedVisitor<PodSpecBuilder>() {
                            @Override
                            public void visit(PodSpecBuilder builder) {
                                KubernetesResourceUtil.mergePodSpec(builder, podSpec, name);
                            }
                        });
                    }
                }
            } else {
                final DeploymentSpec spec = deployHandler.getDeployment(config, images).getSpec();
                if (spec != null) {
                    builder.accept(new TypedVisitor<DeploymentBuilder>() {
                        @Override
                        public void visit(DeploymentBuilder deploymentBuilder) {
                            deploymentBuilder.editOrNewSpec().editOrNewTemplate().editOrNewSpec().endSpec().endTemplate().endSpec();
                            mergeDeploymentSpec(deploymentBuilder, spec);
                        }
                    });

                    if (spec.getTemplate() != null && spec.getTemplate().getSpec() != null) {
                        final PodSpec podSpec = spec.getTemplate().getSpec();
                        builder.accept(new TypedVisitor<PodSpecBuilder>() {
                            @Override
                            public void visit(PodSpecBuilder builder) {
                                KubernetesResourceUtil.mergePodSpec(builder, podSpec, name);
                            }
                        });
                    }
                }
            }
        }
    }

    private void mergeDeploymentSpec(DeploymentBuilder builder, DeploymentSpec spec) {
        DeploymentFluent.SpecNested<DeploymentBuilder> specBuilder = builder.editSpec();
        KubernetesResourceUtil.mergeSimpleFields(specBuilder, spec);
        specBuilder.endSpec();
    }

    private void mergeStatefulSetSpec(StatefulSetBuilder builder, StatefulSetSpec spec) {
        StatefulSetFluent.SpecNested<StatefulSetBuilder> specBuilder = builder.editSpec();
        KubernetesResourceUtil.mergeSimpleFields(specBuilder, spec);
        specBuilder.endSpec();
    }

}
