/*
 * Copyright 2011-2019 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.http.cache

import java.net.InetAddress

import io.gatling.commons.util.RoundRobin
import io.gatling.core.session.{ Session, SessionPrivateAttributes }
import io.gatling.http.protocol.HttpProtocol
import io.gatling.http.util.HttpTypeCaster

object LocalAddressSupport {

  val LocalAddressAttributeName: String = SessionPrivateAttributes.PrivateAttributePrefix + "http.cache.localAddress"
}

trait LocalAddressSupport {

  import LocalAddressSupport._

  def setLocalAddress(httpProtocol: HttpProtocol): Session => Session = {
    httpProtocol.enginePart.localAddresses match {
      case Nil            => identity
      case address :: Nil => _.set(LocalAddressAttributeName, address)
      case address =>
        val it = RoundRobin(address.toVector)
        _.set(LocalAddressAttributeName, it.next())
    }
  }

  val localAddress: Session => Option[InetAddress] = {
    // import optimized TypeCaster
    import HttpTypeCaster._
    _(LocalAddressAttributeName).asOption[InetAddress]
  }
}
