/*
 * Copyright 2011-2023 GatlingCorp (https://gatling.io)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.gatling.javaapi.mqtt;

import static io.gatling.javaapi.core.internal.Converters.*;
import static io.gatling.javaapi.core.internal.Expressions.*;

import io.gatling.javaapi.core.ActionBuilder;
import io.gatling.javaapi.core.Body;
import io.gatling.javaapi.core.CheckBuilder;
import io.gatling.javaapi.core.Session;
import io.gatling.javaapi.mqtt.internal.MqttChecks;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.function.Function;
import javax.annotation.Nonnull;

/**
 * DSL for actions that publish MQTT messages
 *
 * <p>Immutable, so all methods return a new occurrence and leave the original unmodified.
 */
public final class PublishActionBuilder implements ActionBuilder {

  private final io.gatling.mqtt.action.builder.PublishBuilder wrapped;

  public static final class Base {
    private final io.gatling.mqtt.action.builder.MqttActionPublishBase wrapped;

    Base(io.gatling.mqtt.action.builder.MqttActionPublishBase wrapped) {
      this.wrapped = wrapped;
    }

    /**
     * Provide the message to send
     *
     * @param body a body
     * @return a new ActionBuilder instance
     */
    @Nonnull
    public PublishActionBuilder message(@Nonnull Body body) {
      return new PublishActionBuilder(wrapped.message(body.asScala()));
    }
  }

  public PublishActionBuilder(io.gatling.mqtt.action.builder.PublishBuilder wrapped) {
    this.wrapped = wrapped;
  }

  /**
   * Use an at-most-once QoS
   *
   * @return a new ActionBuilder instance
   */
  @Nonnull
  public PublishActionBuilder qosAtMostOnce() {
    return new PublishActionBuilder(wrapped.qosAtMostOnce());
  }

  /**
   * Use an at-least-once QoS
   *
   * @return a new ActionBuilder instance
   */
  @Nonnull
  public PublishActionBuilder qosAtLeastOnce() {
    return new PublishActionBuilder(wrapped.qosAtLeastOnce());
  }

  /**
   * Use an exactly-once QoS
   *
   * @return a new ActionBuilder instance
   */
  @Nonnull
  public PublishActionBuilder qosExactlyOnce() {
    return new PublishActionBuilder(wrapped.qosExactlyOnce());
  }

  public static class Checkable implements ActionBuilder {

    private final io.gatling.mqtt.action.builder.PublishBuilder wrapped;

    private Checkable(io.gatling.mqtt.action.builder.PublishBuilder wrapped) {
      this.wrapped = wrapped;
    }

    /**
     * Use an at-most-once QoS
     *
     * @return a new ActionBuilder instance
     */
    @Nonnull
    public PublishActionBuilder qosAtMostOnce() {
      return new PublishActionBuilder(wrapped.qosAtMostOnce());
    }

    /**
     * Use an at-least-once QoS
     *
     * @return a new ActionBuilder instance
     */
    @Nonnull
    public PublishActionBuilder qosAtLeastOnce() {
      return new PublishActionBuilder(wrapped.qosAtLeastOnce());
    }

    /**
     * Use an exactly-once QoS
     *
     * @return a new ActionBuilder instance
     */
    @Nonnull
    public PublishActionBuilder qosExactlyOnce() {
      return new PublishActionBuilder(wrapped.qosExactlyOnce());
    }

    /**
     * Apply some checks
     *
     * @param checkBuilders the checks
     * @return the next DSL step
     */
    @Nonnull
    public Checkable check(@Nonnull CheckBuilder... checkBuilders) {
      return check(Arrays.asList(checkBuilders));
    }

    /**
     * Apply some checks
     *
     * @param checkBuilders the checks
     * @return the next DSL step
     */
    @Nonnull
    public Checkable check(@Nonnull List<CheckBuilder> checkBuilders) {
      return new Checkable(
          ((io.gatling.mqtt.action.builder.CheckablePublishBuilder) wrapped)
              .check(MqttChecks.toScalaChecks(checkBuilders)));
    }

    @Override
    public io.gatling.core.action.builder.ActionBuilder asScala() {
      return wrapped;
    }
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout in seconds
   * @return the next DSL step
   */
  @Nonnull
  public Checkable await(long timeout) {
    return await(Duration.ofSeconds(timeout));
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout
   * @return the next DSL step
   */
  @Nonnull
  public Checkable await(@Nonnull Duration timeout) {
    return new Checkable(wrapped.await(toScalaDuration(timeout)));
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout in seconds
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a Gatling Expression Language String
   * @return the next DSL step
   */
  @Nonnull
  public Checkable await(long timeout, @Nonnull String expectedTopic) {
    return new Checkable(
        wrapped.wait(
            toScalaDuration(Duration.ofSeconds(timeout)), toStringExpression(expectedTopic)));
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a Gatling Expression Language String
   * @return the next DSL step
   */
  @Nonnull
  public Checkable await(Duration timeout, @Nonnull String expectedTopic) {
    return new Checkable(wrapped.wait(toScalaDuration(timeout), toStringExpression(expectedTopic)));
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout in seconds
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a function
   * @return the next DSL step
   */
  @Nonnull
  public Checkable wait(long timeout, @Nonnull Function<Session, String> expectedTopic) {
    return new Checkable(
        wrapped.wait(
            toScalaDuration(Duration.ofSeconds(timeout)), javaFunctionToExpression(expectedTopic)));
  }

  /**
   * Wait for the checks to complete
   *
   * @param timeout the check timeout
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a function
   * @return the next DSL step
   */
  @Nonnull
  public Checkable wait(
      @Nonnull Duration timeout, @Nonnull Function<Session, String> expectedTopic) {
    return new Checkable(
        wrapped.wait(toScalaDuration(timeout), javaFunctionToExpression(expectedTopic)));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout in seconds
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(long timeout) {
    return expect(Duration.ofSeconds(timeout));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(@Nonnull Duration timeout) {
    return new Checkable(wrapped.expect(toScalaDuration(timeout)));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout in seconds
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a Gatling Expression Language String
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(long timeout, @Nonnull String expectedTopic) {
    return new Checkable(
        wrapped.expect(
            toScalaDuration(Duration.ofSeconds(timeout)), toStringExpression(expectedTopic)));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a function
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(
      @Nonnull Duration timeout, @Nonnull Function<Session, String> expectedTopic) {
    return new Checkable(
        wrapped.expect(toScalaDuration(timeout), javaFunctionToExpression(expectedTopic)));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout in seconds
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a function
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(long timeout, @Nonnull Function<Session, String> expectedTopic) {
    return new Checkable(
        wrapped.expect(
            toScalaDuration(Duration.ofSeconds(timeout)), javaFunctionToExpression(expectedTopic)));
  }

  /**
   * Perform checks in the background, meaning state will have to be reconciled with {@link
   * MqttDsl#waitForMessages()}
   *
   * @param timeout the check timeout
   * @param expectedTopic the topic where the response message is expected to be published,
   *     expressed as a Gatling Expression Language String
   * @return the next DSL step
   */
  @Nonnull
  public Checkable expect(@Nonnull Duration timeout, @Nonnull String expectedTopic) {
    return new Checkable(
        wrapped.expect(toScalaDuration(timeout), toStringExpression(expectedTopic)));
  }

  @Override
  public io.gatling.core.action.builder.ActionBuilder asScala() {
    return wrapped;
  }
}
