/*
 * Decompiled with CFR 0.152.
 */
package io.grpc;

import com.google.common.base.Preconditions;
import com.google.common.base.VerifyException;
import com.google.common.collect.ImmutableList;
import com.google.common.net.InetAddresses;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import io.grpc.Internal;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.MalformedInputException;
import java.nio.charset.StandardCharsets;
import java.util.BitSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import javax.annotation.Nullable;

@Internal
public final class Uri {
    private final String scheme;
    @Nullable
    private final String userInfo;
    @Nullable
    private final String host;
    @Nullable
    private final String port;
    private final String path;
    @Nullable
    private final String query;
    @Nullable
    private final String fragment;
    static final BitSet digitChars = BitSet.valueOf(new long[]{0x3FF000000000000L});
    static final BitSet alphaChars = BitSet.valueOf(new long[]{0L, 576460743847706622L});
    static final BitSet schemeChars = BitSet.valueOf(new long[]{288063250384289792L, 576460743847706622L});
    static final BitSet unreservedChars = BitSet.valueOf(new long[]{288054454291267584L, 5188146764422578174L});
    static final BitSet genDelimsChars = BitSet.valueOf(new long[]{-8935000888854970368L, 671088641L});
    static final BitSet subDelimsChars = BitSet.valueOf(new long[]{2882338748320710656L});
    static final BitSet reservedChars = BitSet.valueOf(new long[]{-6052662140534259712L, 671088641L});
    static final BitSet regNameChars = BitSet.valueOf(new long[]{3170393202611978240L, 5188146764422578174L});
    static final BitSet userInfoChars = BitSet.valueOf(new long[]{3458623578763689984L, 5188146764422578174L});
    static final BitSet pChars = BitSet.valueOf(new long[]{3458623578763689984L, 5188146764422578175L});
    static final BitSet pCharsAndSlash = BitSet.valueOf(new long[]{3458764316252045312L, 5188146764422578175L});
    static final BitSet queryChars;
    static final BitSet fragmentChars;
    private static final char[] hexDigitsByVal;

    private Uri(Builder builder) {
        this.scheme = (String)Preconditions.checkNotNull((Object)builder.scheme, (Object)"scheme");
        this.userInfo = builder.userInfo;
        this.host = builder.host;
        this.port = builder.port;
        this.path = builder.path;
        this.query = builder.query;
        this.fragment = builder.fragment;
        if (this.hasAuthority()) {
            if (!this.path.isEmpty() && !this.path.startsWith("/")) {
                throw new IllegalArgumentException("Has authority -- Non-empty path must start with '/'");
            }
        } else if (this.path.startsWith("//")) {
            throw new IllegalArgumentException("No authority -- Path cannot start with '//'");
        }
    }

    public static Uri parse(String s) throws URISyntaxException {
        try {
            return Uri.create(s);
        }
        catch (IllegalArgumentException e) {
            throw new URISyntaxException(s, e.getMessage());
        }
    }

    public static Uri create(String s) {
        char c;
        int i;
        Builder builder = new Builder();
        int n = s.length();
        int schemeColon = -1;
        for (i = 0; i < n; ++i) {
            char c2 = s.charAt(i);
            if (c2 == ':') {
                schemeColon = i;
                break;
            }
            if (c2 == '/' || c2 == '?' || c2 == '#') break;
        }
        if (schemeColon < 0) {
            throw new IllegalArgumentException("Missing required scheme.");
        }
        builder.setRawScheme(s.substring(0, schemeColon));
        i = schemeColon + 1;
        if (i + 1 < n && s.charAt(i) == '/' && s.charAt(i + 1) == '/') {
            int hostStart;
            int portStartColon;
            char c3;
            int authorityStart = i += 2;
            while (i < n && (c3 = s.charAt(i)) != '/' && c3 != '?' && c3 != '#') {
                ++i;
            }
            String authority = s.substring(authorityStart, i);
            int userInfoEnd = authority.indexOf(64);
            if (userInfoEnd >= 0) {
                builder.setRawUserInfo(authority.substring(0, userInfoEnd));
            }
            if ((portStartColon = Uri.findPortStartColon(authority, hostStart = userInfoEnd >= 0 ? userInfoEnd + 1 : 0)) < 0) {
                builder.setRawHost(authority.substring(hostStart, authority.length()));
            } else {
                builder.setRawHost(authority.substring(hostStart, portStartColon));
                builder.setRawPort(authority.substring(portStartColon + 1));
            }
        }
        int pathStart = i;
        while (i < n && (c = s.charAt(i)) != '?' && c != '#') {
            ++i;
        }
        builder.setRawPath(s.substring(pathStart, i));
        if (i < n && s.charAt(i) == '?') {
            char c4;
            int queryStart = ++i;
            while (i < n && (c4 = s.charAt(i)) != '#') {
                ++i;
            }
            builder.setRawQuery(s.substring(queryStart, i));
        }
        if (i < n && s.charAt(i) == '#') {
            builder.setRawFragment(s.substring(++i));
        }
        return builder.build();
    }

    private static int findPortStartColon(String authority, int hostStart) {
        for (int i = authority.length() - 1; i >= hostStart; --i) {
            char c = authority.charAt(i);
            if (c == ':') {
                return i;
            }
            if (c == ']' || !digitChars.get(c)) break;
        }
        return -1;
    }

    private static void parseAssumedUtf8PathIntoSegments(String path, ImmutableList.Builder<String> out) {
        int start;
        int i = start = path.startsWith("/") ? 1 : 0;
        while (i < path.length()) {
            String segment;
            int nextSlash = path.indexOf(47, i);
            if (nextSlash >= 0) {
                segment = path.substring(i, nextSlash);
                i = nextSlash + 1;
            } else {
                segment = path.substring(i);
                i = path.length();
            }
            if (out != null) {
                out.add((Object)Uri.percentDecodeAssumedUtf8(segment));
                continue;
            }
            Uri.checkPercentEncodedArg(segment, "path segment", pChars);
        }
        if (path.endsWith("/") && out != null) {
            out.add((Object)"");
        }
    }

    public String getScheme() {
        return this.scheme;
    }

    @Nullable
    public String getAuthority() {
        return Uri.percentDecodeAssumedUtf8(this.getRawAuthority());
    }

    private boolean hasAuthority() {
        return this.host != null;
    }

    @Nullable
    public String getRawAuthority() {
        if (this.hasAuthority()) {
            StringBuilder sb = new StringBuilder();
            this.appendAuthority(sb);
            return sb.toString();
        }
        return null;
    }

    private void appendAuthority(StringBuilder sb) {
        if (this.userInfo != null) {
            sb.append(this.userInfo).append('@');
        }
        if (this.host != null) {
            sb.append(this.host);
        }
        if (this.port != null) {
            sb.append(':').append(this.port);
        }
    }

    @Nullable
    public String getUserInfo() {
        return Uri.percentDecodeAssumedUtf8(this.userInfo);
    }

    @Nullable
    public String getRawUserInfo() {
        return this.userInfo;
    }

    @Nullable
    public String getHost() {
        return Uri.percentDecodeAssumedUtf8(this.host);
    }

    @Nullable
    public String getRawHost() {
        return this.host;
    }

    public int getPort() {
        return this.port != null ? Integer.parseInt(this.port) : -1;
    }

    @Nullable
    public String getRawPort() {
        return this.port;
    }

    public String getPath() {
        return Uri.percentDecodeAssumedUtf8(this.path);
    }

    public List<String> getPathSegments() {
        ImmutableList.Builder segmentsBuilder = ImmutableList.builder();
        Uri.parseAssumedUtf8PathIntoSegments(this.path, (ImmutableList.Builder<String>)segmentsBuilder);
        return segmentsBuilder.build();
    }

    public boolean isPathRootless() {
        return !this.path.isEmpty() && !this.path.startsWith("/");
    }

    public boolean isPathAbsolute() {
        return this.path.startsWith("/");
    }

    public String getRawPath() {
        return this.path;
    }

    @Nullable
    public String getQuery() {
        return Uri.percentDecodeAssumedUtf8(this.query);
    }

    @Nullable
    public String getRawQuery() {
        return this.query;
    }

    @Nullable
    public String getFragment() {
        return Uri.percentDecodeAssumedUtf8(this.fragment);
    }

    @Nullable
    public String getRawFragment() {
        return this.fragment;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.scheme).append(':');
        if (this.hasAuthority()) {
            sb.append("//");
            this.appendAuthority(sb);
        }
        sb.append(this.path);
        if (this.query != null) {
            sb.append('?').append(this.query);
        }
        if (this.fragment != null) {
            sb.append('#').append(this.fragment);
        }
        return sb.toString();
    }

    public boolean isAbsolute() {
        return this.scheme != null && this.fragment == null;
    }

    public boolean equals(Object otherObj) {
        if (!(otherObj instanceof Uri)) {
            return false;
        }
        Uri other = (Uri)otherObj;
        return Objects.equals(this.scheme, other.scheme) && Objects.equals(this.userInfo, other.userInfo) && Objects.equals(this.host, other.host) && Objects.equals(this.port, other.port) && Objects.equals(this.path, other.path) && Objects.equals(this.query, other.query) && Objects.equals(this.fragment, other.fragment);
    }

    public int hashCode() {
        return Objects.hash(this.scheme, this.userInfo, this.host, this.port, this.path, this.query, this.fragment);
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static ByteBuffer percentDecode(CharSequence s) {
        ByteBuffer outBuf = ByteBuffer.allocate(s.length());
        Uri.percentDecode(s, "input", null, outBuf);
        outBuf.flip();
        return outBuf;
    }

    private static void percentDecode(CharSequence s, String what, BitSet allowedChars, ByteBuffer outBuf) {
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (c == '%') {
                if (i + 2 >= s.length()) {
                    throw new IllegalArgumentException("Invalid percent-encoding at index " + i + " of " + what + ": " + s);
                }
                int h1 = Character.digit(s.charAt(i + 1), 16);
                int h2 = Character.digit(s.charAt(i + 2), 16);
                if (h1 == -1 || h2 == -1) {
                    throw new IllegalArgumentException("Invalid hex digit in " + what + " at index " + i + " of: " + s);
                }
                if (outBuf != null) {
                    outBuf.put((byte)(h1 << 4 | h2));
                }
                i += 2;
                continue;
            }
            if (allowedChars == null || allowedChars.get(c)) {
                if (outBuf == null) continue;
                outBuf.put((byte)c);
                continue;
            }
            throw new IllegalArgumentException("Invalid character in " + what + " at index " + i);
        }
    }

    @Nullable
    private static String percentDecodeAssumedUtf8(@Nullable String s) {
        if (s == null || s.indexOf(37) == -1) {
            return s;
        }
        ByteBuffer utf8Bytes = Uri.percentDecode(s);
        try {
            return StandardCharsets.UTF_8.newDecoder().onMalformedInput(CodingErrorAction.REPLACE).onUnmappableCharacter(CodingErrorAction.REPLACE).decode(utf8Bytes).toString();
        }
        catch (CharacterCodingException e) {
            throw new VerifyException((Throwable)e);
        }
    }

    @Nullable
    private static String percentEncode(String s, BitSet allowedCodePoints) {
        ByteBuffer utf8Bytes;
        if (s == null) {
            return null;
        }
        CharsetEncoder encoder = StandardCharsets.UTF_8.newEncoder().onMalformedInput(CodingErrorAction.REPORT).onUnmappableCharacter(CodingErrorAction.REPORT);
        try {
            utf8Bytes = encoder.encode(CharBuffer.wrap(s));
        }
        catch (MalformedInputException e) {
            throw new IllegalArgumentException("Malformed input", e);
        }
        catch (CharacterCodingException e) {
            throw new VerifyException((Throwable)e);
        }
        StringBuilder sb = new StringBuilder();
        while (utf8Bytes.hasRemaining()) {
            int b = 0xFF & utf8Bytes.get();
            if (allowedCodePoints.get(b)) {
                sb.append((char)b);
                continue;
            }
            sb.append('%');
            sb.append(hexDigitsByVal[(b & 0xF0) >> 4]);
            sb.append(hexDigitsByVal[b & 0xF]);
        }
        return sb.toString();
    }

    private static void checkPercentEncodedArg(String s, String what, BitSet allowedChars) {
        Uri.percentDecode(s, what, allowedChars, null);
    }

    static {
        fragmentChars = queryChars = BitSet.valueOf(new long[]{-5764607720602730496L, 5188146764422578175L});
        hexDigitsByVal = "0123456789ABCDEF".toCharArray();
    }

    public static final class Builder {
        private String scheme;
        private String path = "";
        private String query;
        private String fragment;
        private String userInfo;
        private String host;
        private String port;

        private Builder() {
        }

        Builder(Uri prototype) {
            this.scheme = prototype.scheme;
            this.userInfo = prototype.userInfo;
            this.host = prototype.host;
            this.port = prototype.port;
            this.path = prototype.path;
            this.query = prototype.query;
            this.fragment = prototype.fragment;
        }

        @CanIgnoreReturnValue
        public Builder setScheme(String scheme) {
            return this.setRawScheme(scheme.toLowerCase(Locale.ROOT));
        }

        @CanIgnoreReturnValue
        Builder setRawScheme(String scheme) {
            if (scheme.isEmpty() || !alphaChars.get(scheme.charAt(0))) {
                throw new IllegalArgumentException("Scheme must start with an alphabetic char");
            }
            for (int i = 0; i < scheme.length(); ++i) {
                char c = scheme.charAt(i);
                if (schemeChars.get(c)) continue;
                throw new IllegalArgumentException("Invalid character in scheme at index " + i);
            }
            this.scheme = scheme;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setPath(String path) {
            Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"Path can be empty but not null");
            this.path = Uri.percentEncode(path, pCharsAndSlash);
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setRawPath(String path) {
            Preconditions.checkArgument((path != null ? 1 : 0) != 0, (Object)"Path can be empty but not null");
            Uri.parseAssumedUtf8PathIntoSegments(path, (ImmutableList.Builder<String>)null);
            this.path = path;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setQuery(@Nullable String query) {
            this.query = Uri.percentEncode(query, queryChars);
            return this;
        }

        @CanIgnoreReturnValue
        Builder setRawQuery(String query) {
            Uri.checkPercentEncodedArg(query, "query", queryChars);
            this.query = query;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setFragment(@Nullable String fragment) {
            this.fragment = Uri.percentEncode(fragment, fragmentChars);
            return this;
        }

        @CanIgnoreReturnValue
        Builder setRawFragment(String fragment) {
            Uri.checkPercentEncodedArg(fragment, "fragment", fragmentChars);
            this.fragment = fragment;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setUserInfo(@Nullable String userInfo) {
            this.userInfo = Uri.percentEncode(userInfo, userInfoChars);
            return this;
        }

        @CanIgnoreReturnValue
        Builder setRawUserInfo(String userInfo) {
            Uri.checkPercentEncodedArg(userInfo, "userInfo", userInfoChars);
            this.userInfo = userInfo;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setHost(@Nullable String regName) {
            if (regName != null) {
                regName = regName.toLowerCase(Locale.ROOT);
                regName = Uri.percentEncode(regName, regNameChars);
            }
            this.host = regName;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setHost(@Nullable InetAddress addr) {
            this.host = addr != null ? Builder.toUriString(addr) : null;
            return this;
        }

        private static String toUriString(InetAddress addr) {
            String inetAddrStr = InetAddresses.toUriString((InetAddress)addr);
            int percentIndex = inetAddrStr.indexOf(37);
            if (percentIndex < 0) {
                return inetAddrStr;
            }
            String scope = inetAddrStr.substring(percentIndex, inetAddrStr.length() - 1);
            return inetAddrStr.substring(0, percentIndex) + Uri.percentEncode(scope, unreservedChars) + "]";
        }

        @CanIgnoreReturnValue
        Builder setRawHost(String host) {
            int percentIndex;
            if (host.startsWith("[") && host.endsWith("]") && (percentIndex = host.indexOf(37)) > 0) {
                String scope = host.substring(percentIndex, host.length() - 1);
                Uri.checkPercentEncodedArg(scope, "scope", unreservedChars);
            }
            if (!InetAddresses.isUriInetAddress((String)host)) {
                Uri.checkPercentEncodedArg(host, "host", regNameChars);
            }
            this.host = host;
            return this;
        }

        @CanIgnoreReturnValue
        public Builder setPort(int port) {
            this.port = port < 0 ? null : Integer.toString(port);
            return this;
        }

        @CanIgnoreReturnValue
        Builder setRawPort(String port) {
            try {
                Integer.parseInt(port);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid port", e);
            }
            this.port = port;
            return this;
        }

        public Uri build() {
            Preconditions.checkState((this.scheme != null ? 1 : 0) != 0, (Object)"Missing required scheme.");
            if (this.host == null) {
                Preconditions.checkState((this.port == null ? 1 : 0) != 0, (Object)"Cannot set port without host.");
                Preconditions.checkState((this.userInfo == null ? 1 : 0) != 0, (Object)"Cannot set userInfo without host.");
            }
            return new Uri(this);
        }
    }
}

