/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.replay;

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.uber.m3.tally.Scope;
import io.temporal.api.common.v1.WorkflowExecution;
import io.temporal.api.workflowservice.v1.PollWorkflowTaskQueueResponseOrBuilder;
import io.temporal.internal.common.WorkflowExecutionUtils;
import io.temporal.internal.replay.WorkflowRunTaskHandler;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class WorkflowExecutorCache {
    private final Logger log = LoggerFactory.getLogger(WorkflowExecutorCache.class);
    private final Scope metricsScope;
    private final LoadingCache<String, WorkflowRunTaskHandler> cache;
    private final Lock cacheLock = new ReentrantLock();
    private final Set<String> inProcessing = new HashSet<String>();

    public WorkflowExecutorCache(int workflowCacheSize, Scope scope) {
        Preconditions.checkArgument((workflowCacheSize > 0 ? 1 : 0) != 0, (Object)"Max cache size must be greater than 0");
        this.metricsScope = Objects.requireNonNull(scope);
        this.cache = CacheBuilder.newBuilder().maximumSize((long)workflowCacheSize).removalListener(e -> {
            WorkflowRunTaskHandler entry = (WorkflowRunTaskHandler)e.getValue();
            if (entry != null) {
                try {
                    this.log.trace("Closing workflow execution for runId {}, cause {}", e.getKey(), (Object)e.getCause());
                    entry.close();
                    this.log.trace("Workflow execution for runId {} closed", (Object)e);
                }
                catch (Throwable t) {
                    this.log.error("Workflow execution closure failed with an exception", t);
                    throw t;
                }
            }
        }).build((CacheLoader)new CacheLoader<String, WorkflowRunTaskHandler>(){

            public WorkflowRunTaskHandler load(String key) {
                return null;
            }
        });
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }

    public WorkflowRunTaskHandler getOrCreate(PollWorkflowTaskQueueResponseOrBuilder workflowTask, Scope metricsScope, Callable<WorkflowRunTaskHandler> workflowExecutorFn) throws Exception {
        WorkflowExecution execution = workflowTask.getWorkflowExecution();
        if (WorkflowExecutionUtils.isFullHistory(workflowTask)) {
            this.invalidate(execution, metricsScope, "full history", null);
            this.log.trace("New Workflow Executor {}-{} has been created for a full history run", (Object)execution.getWorkflowId(), (Object)execution.getRunId());
            return workflowExecutorFn.call();
        }
        WorkflowRunTaskHandler workflowRunTaskHandler = this.getForProcessing(execution, metricsScope);
        if (workflowRunTaskHandler != null) {
            return workflowRunTaskHandler;
        }
        this.log.trace("Workflow Executor {}-{} wasn't found in cache and a new executor has been created", (Object)execution.getWorkflowId(), (Object)execution.getRunId());
        return workflowExecutorFn.call();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private WorkflowRunTaskHandler getForProcessing(WorkflowExecution workflowExecution, Scope metricsScope) throws ExecutionException {
        String runId = workflowExecution.getRunId();
        this.cacheLock.lock();
        try {
            WorkflowRunTaskHandler workflowRunTaskHandler = (WorkflowRunTaskHandler)this.cache.get((Object)runId);
            this.inProcessing.add(runId);
            this.log.trace("Workflow Execution {}-{} has been marked as in-progress", (Object)workflowExecution.getWorkflowId(), (Object)workflowExecution.getRunId());
            metricsScope.counter("temporal_sticky_cache_hit").inc(1L);
            WorkflowRunTaskHandler workflowRunTaskHandler2 = workflowRunTaskHandler;
            return workflowRunTaskHandler2;
        }
        catch (CacheLoader.InvalidCacheLoadException e) {
            metricsScope.counter("temporal_sticky_cache_miss").inc(1L);
            WorkflowRunTaskHandler workflowRunTaskHandler = null;
            return workflowRunTaskHandler;
        }
        finally {
            this.cacheLock.unlock();
        }
    }

    void markProcessingDone(WorkflowExecution workflowExecution) {
        this.cacheLock.lock();
        try {
            this.inProcessing.remove(workflowExecution.getRunId());
            this.log.trace("Workflow Execution {}-{} has been marked as not in-progress", (Object)workflowExecution.getWorkflowId(), (Object)workflowExecution.getRunId());
        }
        finally {
            this.cacheLock.unlock();
        }
    }

    public void addToCache(WorkflowExecution workflowExecution, WorkflowRunTaskHandler workflowRunTaskHandler) {
        this.cache.put((Object)workflowExecution.getRunId(), (Object)workflowRunTaskHandler);
        this.log.trace("Workflow Execution {}-{} has been added to cache", (Object)workflowExecution.getWorkflowId(), (Object)workflowExecution.getRunId());
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public boolean evictAnyNotInProcessing(WorkflowExecution inFavorOfExecution, Scope metricsScope) {
        this.cacheLock.lock();
        try {
            String inFavorOfRunId = inFavorOfExecution.getRunId();
            for (String key : this.cache.asMap().keySet()) {
                if (key.equals(inFavorOfRunId) || this.inProcessing.contains(key)) continue;
                this.log.trace("Workflow Execution {}-{} caused eviction of Workflow Execution with runId {}", new Object[]{inFavorOfExecution.getWorkflowId(), inFavorOfRunId, key});
                this.cache.invalidate((Object)key);
                metricsScope.counter("temporal_sticky_cache_thread_forced_eviction").inc(1L);
                boolean bl = true;
                return bl;
            }
            this.log.trace("Failed to evict from Workflow Execution cache, cache size is {}, inProcessing collection size is {}", (Object)this.cache.size(), (Object)this.inProcessing.size());
            boolean bl = false;
            return bl;
        }
        finally {
            this.cacheLock.unlock();
            this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void invalidate(WorkflowExecution execution, Scope metricsScope, String reason, Throwable cause) {
        this.cacheLock.lock();
        try {
            String runId = execution.getRunId();
            this.log.trace("Invalidating {}-{} because of {}, value is present in the cache: {}", new Object[]{execution.getWorkflowId(), runId, reason, this.cache.getIfPresent((Object)runId), cause});
            this.cache.invalidate((Object)runId);
            this.inProcessing.remove(runId);
            metricsScope.counter("temporal_sticky_cache_total_forced_eviction").inc(1L);
        }
        finally {
            this.cacheLock.unlock();
            this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
        }
    }

    public long size() {
        return this.cache.size();
    }

    public void invalidateAll() {
        this.cache.invalidateAll();
        this.metricsScope.gauge("temporal_sticky_cache_size").update((double)this.size());
    }
}

