/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq;

import jakarta.jms.Connection;
import jakarta.jms.ConnectionFactory;
import jakarta.jms.Destination;
import jakarta.jms.JMSException;
import jakarta.jms.Message;
import jakarta.jms.MessageConsumer;
import jakarta.jms.MessageListener;
import jakarta.jms.MessageProducer;
import jakarta.jms.Session;
import jakarta.jms.Topic;
import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.activemq.ActiveMQConnectionFactory;
import org.apache.activemq.TestSupport;
import org.apache.activemq.advisory.AdvisorySupport;
import org.apache.activemq.broker.BrokerService;
import org.apache.activemq.broker.TransportConnector;
import org.apache.activemq.broker.region.policy.ConstantPendingMessageLimitStrategy;
import org.apache.activemq.broker.region.policy.FilePendingSubscriberMessageStoragePolicy;
import org.apache.activemq.broker.region.policy.MessageEvictionStrategy;
import org.apache.activemq.broker.region.policy.OldestMessageEvictionStrategy;
import org.apache.activemq.broker.region.policy.PendingMessageLimitStrategy;
import org.apache.activemq.broker.region.policy.PendingSubscriberMessageStoragePolicy;
import org.apache.activemq.broker.region.policy.PolicyEntry;
import org.apache.activemq.broker.region.policy.PolicyMap;
import org.apache.activemq.broker.region.policy.VMPendingSubscriberMessageStoragePolicy;
import org.apache.activemq.command.ActiveMQDestination;
import org.apache.activemq.command.ActiveMQMessage;
import org.apache.activemq.command.ActiveMQTopic;
import org.apache.activemq.util.Wait;
import org.junit.After;
import org.junit.Assert;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageEvictionTest {
    static final Logger LOG = LoggerFactory.getLogger(MessageEvictionTest.class);
    private BrokerService broker;
    private ConnectionFactory connectionFactory;
    Connection connection;
    private Session session;
    private Topic destination;
    private final String destinationName = "verifyEvection";
    protected int numMessages = 2000;
    protected String payload = new String(new byte[2048]);

    public void setUp(PendingSubscriberMessageStoragePolicy pendingSubscriberPolicy) throws Exception {
        this.broker = this.createBroker(pendingSubscriberPolicy);
        this.broker.start();
        this.connectionFactory = this.createConnectionFactory();
        this.connection = this.connectionFactory.createConnection();
        this.connection.start();
        this.session = this.connection.createSession(false, 2);
        this.destination = this.session.createTopic("verifyEvection");
    }

    @After
    public void tearDown() throws Exception {
        this.connection.stop();
        this.broker.stop();
    }

    @Test
    public void testMessageEvictionMemoryUsageFileCursor() throws Exception {
        this.setUp((PendingSubscriberMessageStoragePolicy)new FilePendingSubscriberMessageStoragePolicy());
        this.doTestMessageEvictionMemoryUsage();
    }

    @Test
    public void testMessageEvictionMemoryUsageVmCursor() throws Exception {
        this.setUp((PendingSubscriberMessageStoragePolicy)new VMPendingSubscriberMessageStoragePolicy());
        this.doTestMessageEvictionMemoryUsage();
    }

    @Test
    public void testMessageEvictionDiscardedAdvisory() throws Exception {
        this.setUp((PendingSubscriberMessageStoragePolicy)new VMPendingSubscriberMessageStoragePolicy());
        ExecutorService executor = Executors.newSingleThreadExecutor();
        final CountDownLatch consumerRegistered = new CountDownLatch(1);
        final CountDownLatch gotAdvisory = new CountDownLatch(1);
        final CountDownLatch advisoryIsGood = new CountDownLatch(1);
        executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    ActiveMQTopic discardedAdvisoryDestination = AdvisorySupport.getMessageDiscardedAdvisoryTopic((Destination)MessageEvictionTest.this.destination);
                    Session advisorySession = MessageEvictionTest.this.connection.createSession(false, 1);
                    MessageConsumer consumer = advisorySession.createConsumer((Destination)discardedAdvisoryDestination);
                    consumer.setMessageListener(new MessageListener(){
                        int advisoriesReceived = 0;

                        public void onMessage(Message message) {
                            try {
                                LOG.info("advisory:" + String.valueOf(message));
                                ActiveMQMessage activeMQMessage = (ActiveMQMessage)message;
                                Assert.assertNotNull((Object)activeMQMessage.getStringProperty("consumerId"));
                                Assert.assertEquals((long)(++this.advisoriesReceived), (long)activeMQMessage.getIntProperty("discardedCount"));
                                message.acknowledge();
                                advisoryIsGood.countDown();
                            }
                            catch (JMSException e) {
                                e.printStackTrace();
                                Assert.fail((String)e.toString());
                            }
                            finally {
                                gotAdvisory.countDown();
                            }
                        }
                    });
                    consumerRegistered.countDown();
                    gotAdvisory.await(120L, TimeUnit.SECONDS);
                    consumer.close();
                    advisorySession.close();
                }
                catch (Exception e) {
                    e.printStackTrace();
                    Assert.fail((String)e.toString());
                }
            }
        });
        Assert.assertTrue((String)"we have an advisory consumer", (boolean)consumerRegistered.await(60L, TimeUnit.SECONDS));
        this.doTestMessageEvictionMemoryUsage();
        Assert.assertTrue((String)"got an advisory for discarded", (boolean)gotAdvisory.await(0L, TimeUnit.SECONDS));
        Assert.assertTrue((String)"advisory is good", (boolean)advisoryIsGood.await(0L, TimeUnit.SECONDS));
    }

    public void doTestMessageEvictionMemoryUsage() throws Exception {
        ExecutorService executor = Executors.newCachedThreadPool();
        final CountDownLatch doAck = new CountDownLatch(1);
        final CountDownLatch ackDone = new CountDownLatch(1);
        final CountDownLatch consumerRegistered = new CountDownLatch(1);
        executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    MessageConsumer consumer = MessageEvictionTest.this.session.createConsumer((Destination)MessageEvictionTest.this.destination);
                    consumer.setMessageListener(new MessageListener(){

                        public void onMessage(Message message) {
                            try {
                                doAck.await(60L, TimeUnit.SECONDS);
                                LOG.info("acking: " + message.getJMSMessageID());
                                message.acknowledge();
                                ackDone.countDown();
                            }
                            catch (Exception e) {
                                e.printStackTrace();
                                Assert.fail((String)e.toString());
                            }
                            finally {
                                consumerRegistered.countDown();
                                ackDone.countDown();
                            }
                        }
                    });
                    consumerRegistered.countDown();
                    ackDone.await(60L, TimeUnit.SECONDS);
                    consumer.close();
                }
                catch (Exception e) {
                    e.printStackTrace();
                    Assert.fail((String)e.toString());
                }
            }
        });
        Assert.assertTrue((String)"we have a consumer", (boolean)consumerRegistered.await(10L, TimeUnit.SECONDS));
        final AtomicInteger sent = new AtomicInteger(0);
        final CountDownLatch sendDone = new CountDownLatch(1);
        executor.execute(new Runnable(){

            @Override
            public void run() {
                try {
                    MessageProducer producer = MessageEvictionTest.this.session.createProducer((Destination)MessageEvictionTest.this.destination);
                    for (int i = 0; i < MessageEvictionTest.this.numMessages; ++i) {
                        producer.send((Message)MessageEvictionTest.this.session.createTextMessage(MessageEvictionTest.this.payload));
                        sent.incrementAndGet();
                    }
                    producer.close();
                    sendDone.countDown();
                }
                catch (Exception e) {
                    sendDone.countDown();
                    e.printStackTrace();
                    Assert.fail((String)e.toString());
                }
            }
        });
        Assert.assertTrue((String)"messages sending done", (boolean)sendDone.await(180L, TimeUnit.SECONDS));
        Assert.assertEquals((String)"all message were sent", (long)this.numMessages, (long)sent.get());
        doAck.countDown();
        executor.shutdown();
        executor.awaitTermination(30L, TimeUnit.SECONDS);
        Assert.assertTrue((String)"usage goes to 0 once consumer goes away", (boolean)Wait.waitFor((Wait.Condition)new Wait.Condition(){

            public boolean isSatisified() throws Exception {
                return 0 == TestSupport.getDestination(MessageEvictionTest.this.broker, ActiveMQDestination.transform((Destination)MessageEvictionTest.this.destination)).getMemoryUsage().getPercentUsage();
            }
        }));
    }

    BrokerService createBroker(PendingSubscriberMessageStoragePolicy pendingSubscriberPolicy) throws Exception {
        BrokerService brokerService = new BrokerService();
        brokerService.addConnector("tcp://localhost:0");
        brokerService.setUseJmx(false);
        brokerService.setDeleteAllMessagesOnStartup(true);
        brokerService.getSystemUsage().getMemoryUsage().setLimit(512000L);
        ArrayList<PolicyEntry> policyEntries = new ArrayList<PolicyEntry>();
        PolicyEntry entry = new PolicyEntry();
        entry.setTopic(">");
        entry.setAdvisoryForDiscardingMessages(true);
        entry.setTopicPrefetch(50);
        entry.setPendingSubscriberPolicy(pendingSubscriberPolicy);
        int pendingMessageLimit = 50;
        if (pendingSubscriberPolicy instanceof FilePendingSubscriberMessageStoragePolicy) {
            pendingMessageLimit = 500;
        }
        ConstantPendingMessageLimitStrategy pendingMessageLimitStrategy = new ConstantPendingMessageLimitStrategy();
        pendingMessageLimitStrategy.setLimit(pendingMessageLimit);
        entry.setPendingMessageLimitStrategy((PendingMessageLimitStrategy)pendingMessageLimitStrategy);
        OldestMessageEvictionStrategy messageEvictionStrategy = new OldestMessageEvictionStrategy();
        entry.setMessageEvictionStrategy((MessageEvictionStrategy)messageEvictionStrategy);
        entry.setDeadLetterStrategy(null);
        policyEntries.add(entry);
        PolicyMap policyMap = new PolicyMap();
        policyMap.setPolicyEntries(policyEntries);
        brokerService.setDestinationPolicy(policyMap);
        return brokerService;
    }

    ConnectionFactory createConnectionFactory() throws Exception {
        String url = ((TransportConnector)this.broker.getTransportConnectors().get(0)).getServer().getConnectURI().toString();
        ActiveMQConnectionFactory factory = new ActiveMQConnectionFactory(url);
        factory.setWatchTopicAdvisories(false);
        return factory;
    }
}

