/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.any23.extractor.html;

import org.apache.any23.extractor.ExtractionContext;
import org.apache.any23.extractor.ExtractionException;
import org.apache.any23.extractor.ExtractionParameters;
import org.apache.any23.extractor.ExtractionResult;
import org.apache.any23.extractor.ExtractorDescription;
import org.apache.any23.extractor.IssueReport;
import org.apache.any23.vocab.XHTML;
import org.apache.any23.extractor.Extractor.TagSoupDOMExtractor;
import org.openrdf.model.URI;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

import java.io.IOException;

/**
 * Extractor for the <a href="http://microformats.org/wiki/rel-license">rel-license</a>
 * microformat.
 * <p/>
 *
 * @author Gabriele Renzi
 * @author Richard Cyganiak
 */
public class LicenseExtractor implements TagSoupDOMExtractor {

    private static final XHTML vXHTML = XHTML.getInstance();

    @Override
    public void run(
            ExtractionParameters extractionParameters,
            ExtractionContext extractionContext,
            Document in,
            ExtractionResult out
    ) throws IOException, ExtractionException {
        HTMLDocument document = new HTMLDocument(in);
        final URI documentURI = extractionContext.getDocumentURI();
        for (Node node : DomUtils.findAll(in, "//A[@rel='license']/@href")) {
            String link = node.getNodeValue();
            if ("".equals(link)) {
                out.notifyIssue(
                        IssueReport.IssueLevel.Warning,
                        String.format(
                                "Invalid license link detected within document %s.",
                                documentURI.toString()
                        ),
                        0, 0
                );
                continue;
            }
            out.writeTriple(documentURI, vXHTML.license, document.resolveURI(link));
        }
    }

    @Override
    public ExtractorDescription getDescription() {
        return LicenseExtractorFactory.getDescriptionInstance();
    }
    
}
