

/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.arrow.vector.complex;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import com.google.flatbuffers.FlatBufferBuilder;

import com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.arrow.memory.*;
import org.apache.arrow.vector.types.Types;
import org.apache.arrow.vector.types.Types.*;
import org.apache.arrow.vector.types.pojo.*;
import org.apache.arrow.vector.types.pojo.ArrowType.*;
import org.apache.arrow.vector.types.*;
import org.apache.arrow.vector.*;
import org.apache.arrow.vector.holders.*;
import org.apache.arrow.vector.util.*;
import org.apache.arrow.vector.complex.*;
import org.apache.arrow.vector.complex.reader.*;
import org.apache.arrow.vector.complex.impl.*;
import org.apache.arrow.vector.complex.writer.*;
import org.apache.arrow.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.arrow.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.arrow.vector.util.JsonStringArrayList;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;






import com.google.common.collect.ImmutableList;
import java.util.ArrayList;
import java.util.Iterator;
import org.apache.arrow.vector.complex.impl.ComplexCopier;
import org.apache.arrow.vector.util.CallBack;
import org.apache.arrow.vector.schema.ArrowFieldNode;

import static org.apache.arrow.flatbuf.UnionMode.Sparse;


/*
 * This class is generated using freemarker and the UnionVector.java template.
 */
@SuppressWarnings("unused")


/**
 * A vector which can hold values of different types. It does so by using a MapVector which contains a vector for each
 * primitive type that is stored. MapVector is used in order to take advantage of its serialization/deserialization methods,
 * as well as the addOrGet method.
 *
 * For performance reasons, UnionVector stores a cached reference to each subtype vector, to avoid having to do the map lookup
 * each time the vector is accessed.
 */
public class UnionVector implements FieldVector {

  private String name;
  private BufferAllocator allocator;
  private Accessor accessor = new Accessor();
  private Mutator mutator = new Mutator();
  int valueCount;

  MapVector internalMap;
  UInt1Vector typeVector;

  private NullableMapVector mapVector;
  private ListVector listVector;

  private FieldReader reader;

  private int singleType = 0;
  private ValueVector singleVector;

  private final CallBack callBack;

  public UnionVector(String name, BufferAllocator allocator, CallBack callBack) {
    this.name = name;
    this.allocator = allocator;
    this.internalMap = new MapVector("internal", allocator, callBack);
    this.typeVector = new UInt1Vector("types", allocator);
    this.callBack = callBack;
  }

  public BufferAllocator getAllocator() {
    return allocator;
  }

  @Override
  public MinorType getMinorType() {
    return MinorType.UNION;
  }

  @Override
  public void initializeChildrenFromFields(List<Field> children) {
    getMap().initializeChildrenFromFields(children);
  }

  @Override
  public List<FieldVector> getChildrenFromFields() {
    return getMap().getChildrenFromFields();
  }

  @Override
  public void loadFieldBuffers(ArrowFieldNode fieldNode, List<ArrowBuf> ownBuffers) {
    // TODO
    throw new UnsupportedOperationException();
  }

  @Override
  public List<ArrowBuf> getFieldBuffers() {
    // TODO
    throw new UnsupportedOperationException();
  }

  @Override
  public List<BufferBacked> getFieldInnerVectors() {
    // TODO
    throw new UnsupportedOperationException();
  }
  
  public NullableMapVector getMap() {
    if (mapVector == null) {
      int vectorCount = internalMap.size();
      mapVector = internalMap.addOrGet("map", MinorType.MAP, NullableMapVector.class);
      if (internalMap.size() > vectorCount) {
        mapVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return mapVector;
  }


  private NullableTinyIntVector tinyIntVector;

  public NullableTinyIntVector getTinyIntVector() {
    if (tinyIntVector == null) {
      int vectorCount = internalMap.size();
      tinyIntVector = internalMap.addOrGet("tinyInt", MinorType.TINYINT, NullableTinyIntVector.class);
      if (internalMap.size() > vectorCount) {
        tinyIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return tinyIntVector;
  }



  private NullableUInt1Vector uInt1Vector;

  public NullableUInt1Vector getUInt1Vector() {
    if (uInt1Vector == null) {
      int vectorCount = internalMap.size();
      uInt1Vector = internalMap.addOrGet("uInt1", MinorType.UINT1, NullableUInt1Vector.class);
      if (internalMap.size() > vectorCount) {
        uInt1Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt1Vector;
  }



  private NullableUInt2Vector uInt2Vector;

  public NullableUInt2Vector getUInt2Vector() {
    if (uInt2Vector == null) {
      int vectorCount = internalMap.size();
      uInt2Vector = internalMap.addOrGet("uInt2", MinorType.UINT2, NullableUInt2Vector.class);
      if (internalMap.size() > vectorCount) {
        uInt2Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt2Vector;
  }



  private NullableSmallIntVector smallIntVector;

  public NullableSmallIntVector getSmallIntVector() {
    if (smallIntVector == null) {
      int vectorCount = internalMap.size();
      smallIntVector = internalMap.addOrGet("smallInt", MinorType.SMALLINT, NullableSmallIntVector.class);
      if (internalMap.size() > vectorCount) {
        smallIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return smallIntVector;
  }



  private NullableIntVector intVector;

  public NullableIntVector getIntVector() {
    if (intVector == null) {
      int vectorCount = internalMap.size();
      intVector = internalMap.addOrGet("int", MinorType.INT, NullableIntVector.class);
      if (internalMap.size() > vectorCount) {
        intVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intVector;
  }



  private NullableUInt4Vector uInt4Vector;

  public NullableUInt4Vector getUInt4Vector() {
    if (uInt4Vector == null) {
      int vectorCount = internalMap.size();
      uInt4Vector = internalMap.addOrGet("uInt4", MinorType.UINT4, NullableUInt4Vector.class);
      if (internalMap.size() > vectorCount) {
        uInt4Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt4Vector;
  }



  private NullableFloat4Vector float4Vector;

  public NullableFloat4Vector getFloat4Vector() {
    if (float4Vector == null) {
      int vectorCount = internalMap.size();
      float4Vector = internalMap.addOrGet("float4", MinorType.FLOAT4, NullableFloat4Vector.class);
      if (internalMap.size() > vectorCount) {
        float4Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return float4Vector;
  }



  private NullableIntervalYearVector intervalYearVector;

  public NullableIntervalYearVector getIntervalYearVector() {
    if (intervalYearVector == null) {
      int vectorCount = internalMap.size();
      intervalYearVector = internalMap.addOrGet("intervalYear", MinorType.INTERVALYEAR, NullableIntervalYearVector.class);
      if (internalMap.size() > vectorCount) {
        intervalYearVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intervalYearVector;
  }



  private NullableTimeVector timeVector;

  public NullableTimeVector getTimeVector() {
    if (timeVector == null) {
      int vectorCount = internalMap.size();
      timeVector = internalMap.addOrGet("time", MinorType.TIME, NullableTimeVector.class);
      if (internalMap.size() > vectorCount) {
        timeVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeVector;
  }



  private NullableBigIntVector bigIntVector;

  public NullableBigIntVector getBigIntVector() {
    if (bigIntVector == null) {
      int vectorCount = internalMap.size();
      bigIntVector = internalMap.addOrGet("bigInt", MinorType.BIGINT, NullableBigIntVector.class);
      if (internalMap.size() > vectorCount) {
        bigIntVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return bigIntVector;
  }



  private NullableUInt8Vector uInt8Vector;

  public NullableUInt8Vector getUInt8Vector() {
    if (uInt8Vector == null) {
      int vectorCount = internalMap.size();
      uInt8Vector = internalMap.addOrGet("uInt8", MinorType.UINT8, NullableUInt8Vector.class);
      if (internalMap.size() > vectorCount) {
        uInt8Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return uInt8Vector;
  }



  private NullableFloat8Vector float8Vector;

  public NullableFloat8Vector getFloat8Vector() {
    if (float8Vector == null) {
      int vectorCount = internalMap.size();
      float8Vector = internalMap.addOrGet("float8", MinorType.FLOAT8, NullableFloat8Vector.class);
      if (internalMap.size() > vectorCount) {
        float8Vector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return float8Vector;
  }



  private NullableDateVector dateVector;

  public NullableDateVector getDateVector() {
    if (dateVector == null) {
      int vectorCount = internalMap.size();
      dateVector = internalMap.addOrGet("date", MinorType.DATE, NullableDateVector.class);
      if (internalMap.size() > vectorCount) {
        dateVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return dateVector;
  }



  private NullableTimeStampVector timeStampVector;

  public NullableTimeStampVector getTimeStampVector() {
    if (timeStampVector == null) {
      int vectorCount = internalMap.size();
      timeStampVector = internalMap.addOrGet("timeStamp", MinorType.TIMESTAMP, NullableTimeStampVector.class);
      if (internalMap.size() > vectorCount) {
        timeStampVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return timeStampVector;
  }



  private NullableIntervalDayVector intervalDayVector;

  public NullableIntervalDayVector getIntervalDayVector() {
    if (intervalDayVector == null) {
      int vectorCount = internalMap.size();
      intervalDayVector = internalMap.addOrGet("intervalDay", MinorType.INTERVALDAY, NullableIntervalDayVector.class);
      if (internalMap.size() > vectorCount) {
        intervalDayVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return intervalDayVector;
  }




  private NullableVarBinaryVector varBinaryVector;

  public NullableVarBinaryVector getVarBinaryVector() {
    if (varBinaryVector == null) {
      int vectorCount = internalMap.size();
      varBinaryVector = internalMap.addOrGet("varBinary", MinorType.VARBINARY, NullableVarBinaryVector.class);
      if (internalMap.size() > vectorCount) {
        varBinaryVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return varBinaryVector;
  }



  private NullableVarCharVector varCharVector;

  public NullableVarCharVector getVarCharVector() {
    if (varCharVector == null) {
      int vectorCount = internalMap.size();
      varCharVector = internalMap.addOrGet("varChar", MinorType.VARCHAR, NullableVarCharVector.class);
      if (internalMap.size() > vectorCount) {
        varCharVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return varCharVector;
  }



  private NullableBitVector bitVector;

  public NullableBitVector getBitVector() {
    if (bitVector == null) {
      int vectorCount = internalMap.size();
      bitVector = internalMap.addOrGet("bit", MinorType.BIT, NullableBitVector.class);
      if (internalMap.size() > vectorCount) {
        bitVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return bitVector;
  }



  public ListVector getList() {
    if (listVector == null) {
      int vectorCount = internalMap.size();
      listVector = internalMap.addOrGet("list", MinorType.LIST, ListVector.class);
      if (internalMap.size() > vectorCount) {
        listVector.allocateNew();
        if (callBack != null) {
          callBack.doWork();
        }
      }
    }
    return listVector;
  }

  public int getTypeValue(int index) {
    return typeVector.getAccessor().get(index);
  }

  public UInt1Vector getTypeVector() {
    return typeVector;
  }

  @Override
  public void allocateNew() throws OutOfMemoryException {
    internalMap.allocateNew();
    typeVector.allocateNew();
    if (typeVector != null) {
      typeVector.zeroVector();
    }
  }

  @Override
  public boolean allocateNewSafe() {
    boolean safe = internalMap.allocateNewSafe();
    safe = safe && typeVector.allocateNewSafe();
    if (safe) {
      if (typeVector != null) {
        typeVector.zeroVector();
      }
    }
    return safe;
  }

  @Override
  public void setInitialCapacity(int numRecords) {
  }

  @Override
  public int getValueCapacity() {
    return Math.min(typeVector.getValueCapacity(), internalMap.getValueCapacity());
  }

  @Override
  public void close() {
    clear();
  }

  @Override
  public void clear() {
    typeVector.clear();
    internalMap.clear();
  }

  @Override
  public Field getField() {
    List<org.apache.arrow.vector.types.pojo.Field> childFields = new ArrayList<>();
    List<FieldVector> children = internalMap.getChildren();
    int[] typeIds = new int[children.size()];
    for (ValueVector v : children) {
      typeIds[childFields.size()] = v.getMinorType().ordinal();
      childFields.add(v.getField());
    }
    return new Field(name, true, new ArrowType.Union(Sparse, typeIds), childFields);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator) {
    return new TransferImpl(name, allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator) {
    return new TransferImpl(ref, allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector target) {
    return new TransferImpl((UnionVector) target);
  }

  public void transferTo(org.apache.arrow.vector.complex.UnionVector target) {
    typeVector.makeTransferPair(target.typeVector).transfer();
    internalMap.makeTransferPair(target.internalMap).transfer();
    target.valueCount = valueCount;
  }

  public void copyFrom(int inIndex, int outIndex, UnionVector from) {
    from.getReader().setPosition(inIndex);
    getWriter().setPosition(outIndex);
    ComplexCopier.copy(from.reader, mutator.writer);
  }

  public void copyFromSafe(int inIndex, int outIndex, UnionVector from) {
    copyFrom(inIndex, outIndex, from);
  }

  public FieldVector addVector(FieldVector v) {
    String name = v.getMinorType().name().toLowerCase();
    Preconditions.checkState(internalMap.getChild(name) == null, String.format("%s vector already exists", name));
    final FieldVector newVector = internalMap.addOrGet(name, v.getMinorType(), v.getClass());
    v.makeTransferPair(newVector).transfer();
    internalMap.putChild(name, newVector);
    if (callBack != null) {
      callBack.doWork();
    }
    return newVector;
  }

  private class TransferImpl implements TransferPair {

    UnionVector to;

    public TransferImpl(String name, BufferAllocator allocator) {
      to = new UnionVector(name, allocator, null);
    }

    public TransferImpl(UnionVector to) {
      this.to = to;
    }

    @Override
    public void transfer() {
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      to.allocateNew();
      for (int i = 0; i < length; i++) {
        to.copyFromSafe(startIndex + i, i, org.apache.arrow.vector.complex.UnionVector.this);
      }
      to.getMutator().setValueCount(length);
    }

    @Override
    public ValueVector getTo() {
      return to;
    }

    @Override
    public void copyValueSafe(int from, int to) {
      this.to.copyFrom(from, to, UnionVector.this);
    }
  }

  @Override
  public Accessor getAccessor() {
    return accessor;
  }

  @Override
  public Mutator getMutator() {
    return mutator;
  }

  @Override
  public FieldReader getReader() {
    if (reader == null) {
      reader = new UnionReader(this);
    }
    return reader;
  }

  public FieldWriter getWriter() {
    if (mutator.writer == null) {
      mutator.writer = new UnionWriter(this);
    }
    return mutator.writer;
  }

  @Override
  public int getBufferSize() {
    return typeVector.getBufferSize() + internalMap.getBufferSize();
  }

  @Override
  public int getBufferSizeFor(final int valueCount) {
    if (valueCount == 0) {
      return 0;
    }

    long bufferSize = 0;
    for (final ValueVector v : (Iterable<ValueVector>) this) {
      bufferSize += v.getBufferSizeFor(valueCount);
    }

    return (int) bufferSize;
  }

  @Override
  public ArrowBuf[] getBuffers(boolean clear) {
    ImmutableList.Builder<ArrowBuf> builder = ImmutableList.builder();
    builder.add(typeVector.getBuffers(clear));
    builder.add(internalMap.getBuffers(clear));
    List<ArrowBuf> list = builder.build();
    return list.toArray(new ArrowBuf[list.size()]);
  }

  @Override
  public Iterator<ValueVector> iterator() {
    List<ValueVector> vectors = Lists.newArrayList(internalMap.iterator());
    vectors.add(typeVector);
    return vectors.iterator();
  }

  public class Accessor extends BaseValueVector.BaseAccessor {


    @Override
    public Object getObject(int index) {
      int type = typeVector.getAccessor().get(index);
      switch (MinorType.values()[type]) {
      case NULL:
        return null;
      case TINYINT:
        return getTinyIntVector().getAccessor().getObject(index);

      case UINT1:
        return getUInt1Vector().getAccessor().getObject(index);

      case UINT2:
        return getUInt2Vector().getAccessor().getObject(index);

      case SMALLINT:
        return getSmallIntVector().getAccessor().getObject(index);

      case INT:
        return getIntVector().getAccessor().getObject(index);

      case UINT4:
        return getUInt4Vector().getAccessor().getObject(index);

      case FLOAT4:
        return getFloat4Vector().getAccessor().getObject(index);

      case INTERVALYEAR:
        return getIntervalYearVector().getAccessor().getObject(index);

      case TIME:
        return getTimeVector().getAccessor().getObject(index);

      case BIGINT:
        return getBigIntVector().getAccessor().getObject(index);

      case UINT8:
        return getUInt8Vector().getAccessor().getObject(index);

      case FLOAT8:
        return getFloat8Vector().getAccessor().getObject(index);

      case DATE:
        return getDateVector().getAccessor().getObject(index);

      case TIMESTAMP:
        return getTimeStampVector().getAccessor().getObject(index);

      case INTERVALDAY:
        return getIntervalDayVector().getAccessor().getObject(index);


      case VARBINARY:
        return getVarBinaryVector().getAccessor().getObject(index);

      case VARCHAR:
        return getVarCharVector().getAccessor().getObject(index);

      case BIT:
        return getBitVector().getAccessor().getObject(index);

      case MAP:
        return getMap().getAccessor().getObject(index);
      case LIST:
        return getList().getAccessor().getObject(index);
      default:
        throw new UnsupportedOperationException("Cannot support type: " + MinorType.values()[type]);
      }
    }

    public byte[] get(int index) {
      return null;
    }

    public void get(int index, ComplexHolder holder) {
    }

    public void get(int index, UnionHolder holder) {
      FieldReader reader = new UnionReader(UnionVector.this);
      reader.setPosition(index);
      holder.reader = reader;
    }

    @Override
    public int getValueCount() {
      return valueCount;
    }

    @Override
    public boolean isNull(int index) {
      return typeVector.getAccessor().get(index) == 0;
    }

    public int isSet(int index) {
      return isNull(index) ? 0 : 1;
    }
  }

  public class Mutator extends BaseValueVector.BaseMutator {

    UnionWriter writer;

    @Override
    public void setValueCount(int valueCount) {
      UnionVector.this.valueCount = valueCount;
      typeVector.getMutator().setValueCount(valueCount);
      internalMap.getMutator().setValueCount(valueCount);
    }

    public void setSafe(int index, UnionHolder holder) {
      FieldReader reader = holder.reader;
      if (writer == null) {
        writer = new UnionWriter(UnionVector.this);
      }
      writer.setPosition(index);
      MinorType type = reader.getMinorType();
      switch (type) {
      case TINYINT:
        NullableTinyIntHolder tinyIntHolder = new NullableTinyIntHolder();
        reader.read(tinyIntHolder);
        setSafe(index, tinyIntHolder);
        break;
      case UINT1:
        NullableUInt1Holder uInt1Holder = new NullableUInt1Holder();
        reader.read(uInt1Holder);
        setSafe(index, uInt1Holder);
        break;
      case UINT2:
        NullableUInt2Holder uInt2Holder = new NullableUInt2Holder();
        reader.read(uInt2Holder);
        setSafe(index, uInt2Holder);
        break;
      case SMALLINT:
        NullableSmallIntHolder smallIntHolder = new NullableSmallIntHolder();
        reader.read(smallIntHolder);
        setSafe(index, smallIntHolder);
        break;
      case INT:
        NullableIntHolder intHolder = new NullableIntHolder();
        reader.read(intHolder);
        setSafe(index, intHolder);
        break;
      case UINT4:
        NullableUInt4Holder uInt4Holder = new NullableUInt4Holder();
        reader.read(uInt4Holder);
        setSafe(index, uInt4Holder);
        break;
      case FLOAT4:
        NullableFloat4Holder float4Holder = new NullableFloat4Holder();
        reader.read(float4Holder);
        setSafe(index, float4Holder);
        break;
      case INTERVALYEAR:
        NullableIntervalYearHolder intervalYearHolder = new NullableIntervalYearHolder();
        reader.read(intervalYearHolder);
        setSafe(index, intervalYearHolder);
        break;
      case TIME:
        NullableTimeHolder timeHolder = new NullableTimeHolder();
        reader.read(timeHolder);
        setSafe(index, timeHolder);
        break;
      case BIGINT:
        NullableBigIntHolder bigIntHolder = new NullableBigIntHolder();
        reader.read(bigIntHolder);
        setSafe(index, bigIntHolder);
        break;
      case UINT8:
        NullableUInt8Holder uInt8Holder = new NullableUInt8Holder();
        reader.read(uInt8Holder);
        setSafe(index, uInt8Holder);
        break;
      case FLOAT8:
        NullableFloat8Holder float8Holder = new NullableFloat8Holder();
        reader.read(float8Holder);
        setSafe(index, float8Holder);
        break;
      case DATE:
        NullableDateHolder dateHolder = new NullableDateHolder();
        reader.read(dateHolder);
        setSafe(index, dateHolder);
        break;
      case TIMESTAMP:
        NullableTimeStampHolder timeStampHolder = new NullableTimeStampHolder();
        reader.read(timeStampHolder);
        setSafe(index, timeStampHolder);
        break;
      case INTERVALDAY:
        NullableIntervalDayHolder intervalDayHolder = new NullableIntervalDayHolder();
        reader.read(intervalDayHolder);
        setSafe(index, intervalDayHolder);
        break;
      case VARBINARY:
        NullableVarBinaryHolder varBinaryHolder = new NullableVarBinaryHolder();
        reader.read(varBinaryHolder);
        setSafe(index, varBinaryHolder);
        break;
      case VARCHAR:
        NullableVarCharHolder varCharHolder = new NullableVarCharHolder();
        reader.read(varCharHolder);
        setSafe(index, varCharHolder);
        break;
      case BIT:
        NullableBitHolder bitHolder = new NullableBitHolder();
        reader.read(bitHolder);
        setSafe(index, bitHolder);
        break;
      case MAP: {
        ComplexCopier.copy(reader, writer);
        break;
      }
      case LIST: {
        ComplexCopier.copy(reader, writer);
        break;
      }
      default:
        throw new UnsupportedOperationException();
      }
    }

    public void setSafe(int index, NullableTinyIntHolder holder) {
      setType(index, MinorType.TINYINT);
      getTinyIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt1Holder holder) {
      setType(index, MinorType.UINT1);
      getUInt1Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt2Holder holder) {
      setType(index, MinorType.UINT2);
      getUInt2Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableSmallIntHolder holder) {
      setType(index, MinorType.SMALLINT);
      getSmallIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntHolder holder) {
      setType(index, MinorType.INT);
      getIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt4Holder holder) {
      setType(index, MinorType.UINT4);
      getUInt4Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat4Holder holder) {
      setType(index, MinorType.FLOAT4);
      getFloat4Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalYearHolder holder) {
      setType(index, MinorType.INTERVALYEAR);
      getIntervalYearVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeHolder holder) {
      setType(index, MinorType.TIME);
      getTimeVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBigIntHolder holder) {
      setType(index, MinorType.BIGINT);
      getBigIntVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableUInt8Holder holder) {
      setType(index, MinorType.UINT8);
      getUInt8Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableFloat8Holder holder) {
      setType(index, MinorType.FLOAT8);
      getFloat8Vector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableDateHolder holder) {
      setType(index, MinorType.DATE);
      getDateVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableTimeStampHolder holder) {
      setType(index, MinorType.TIMESTAMP);
      getTimeStampVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableIntervalDayHolder holder) {
      setType(index, MinorType.INTERVALDAY);
      getIntervalDayVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarBinaryHolder holder) {
      setType(index, MinorType.VARBINARY);
      getVarBinaryVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableVarCharHolder holder) {
      setType(index, MinorType.VARCHAR);
      getVarCharVector().getMutator().setSafe(index, holder);
    }

    public void setSafe(int index, NullableBitHolder holder) {
      setType(index, MinorType.BIT);
      getBitVector().getMutator().setSafe(index, holder);
    }


    public void setType(int index, MinorType type) {
      typeVector.getMutator().setSafe(index, (byte) type.ordinal());
    }

    @Override
    public void reset() { }

    @Override
    public void generateTestData(int values) { }
  }
}
