/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.main;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import org.apache.camel.CamelContext;
import org.apache.camel.main.MainShutdownStrategy;
import org.apache.camel.spi.CamelEvent;
import org.apache.camel.support.EventNotifierSupport;
import org.apache.camel.util.StopWatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MainDurationEventNotifier
extends EventNotifierSupport {
    private static final Logger LOG = LoggerFactory.getLogger(MainDurationEventNotifier.class);
    private final CamelContext camelContext;
    private final int maxMessages;
    private final long maxIdleSeconds;
    private final MainShutdownStrategy shutdownStrategy;
    private final boolean stopCamelContext;
    private final boolean restartDuration;
    private final Action action;
    private final LongAdder doneMessages;
    private volatile StopWatch watch;
    private volatile ScheduledExecutorService idleExecutorService;

    public MainDurationEventNotifier(CamelContext camelContext, int maxMessages, long maxIdleSeconds, MainShutdownStrategy shutdownStrategy, boolean stopCamelContext, boolean restartDuration, String action) {
        this.camelContext = camelContext;
        this.maxMessages = maxMessages;
        this.maxIdleSeconds = maxIdleSeconds;
        this.shutdownStrategy = shutdownStrategy;
        this.stopCamelContext = stopCamelContext;
        this.restartDuration = restartDuration;
        this.action = Action.toAction(action);
        this.doneMessages = new LongAdder();
        if (maxMessages == 0 && maxIdleSeconds == 0L) {
            this.setIgnoreExchangeEvents(true);
        }
    }

    public void notify(CamelEvent event) throws Exception {
        try {
            this.doNotify(event);
        }
        catch (Exception e) {
            LOG.warn("Error during processing CamelEvent: {}. This exception is ignored.", (Object)event, (Object)e);
        }
    }

    protected void doNotify(CamelEvent event) {
        if (!this.shutdownStrategy.isRunAllowed()) {
            return;
        }
        if (event.getType() == CamelEvent.Type.RouteReloaded) {
            this.resetOnReload();
            return;
        }
        boolean complete = false;
        if (this.maxMessages > 0) {
            boolean bl = complete = event.getType() == CamelEvent.Type.ExchangeCompleted || event.getType() == CamelEvent.Type.ExchangeFailed;
            if (complete) {
                boolean result;
                this.doneMessages.increment();
                int doneCount = this.doneMessages.intValue();
                boolean bl2 = result = doneCount >= this.maxMessages;
                if (LOG.isTraceEnabled()) {
                    LOG.trace("Duration max messages check {} >= {} -> {}", new Object[]{doneCount, this.maxMessages, result});
                }
                if (result && this.shutdownStrategy.isRunAllowed()) {
                    this.triggerDoneEvent();
                }
            }
        }
        if (this.maxIdleSeconds > 0L) {
            this.resetOnActivity(event, complete);
        }
    }

    private void triggerDoneEvent() {
        if (this.action == Action.SHUTDOWN) {
            LOG.info("Duration max messages triggering shutdown of the JVM");
            this.camelContext.getExecutorServiceManager().newThread("CamelMainShutdownCamelContext", this::shutdownTask).start();
        } else if (this.action == Action.STOP) {
            LOG.info("Duration max messages triggering stopping all routes");
            this.camelContext.getExecutorServiceManager().newThread("CamelMainShutdownCamelContext", this::stopTask).start();
        }
    }

    private void resetOnActivity(CamelEvent event, boolean complete) {
        boolean created;
        boolean bl = created = event.getType() == CamelEvent.Type.ExchangeCreated;
        if ((created || complete) && this.watch != null) {
            LOG.trace("Message activity so restarting stop watch");
            this.watch.restart();
        }
    }

    private void resetOnReload() {
        if (this.restartDuration) {
            LOG.debug("Routes reloaded. Resetting maxMessages/maxIdleSeconds/maxSeconds");
            this.shutdownStrategy.restartAwait();
            this.doneMessages.reset();
            if (this.watch != null) {
                this.watch.restart();
            }
        }
    }

    public boolean isEnabled(CamelEvent event) {
        return event.getType() == CamelEvent.Type.ExchangeCreated || event.getType() == CamelEvent.Type.ExchangeCompleted || event.getType() == CamelEvent.Type.ExchangeFailed || event.getType() == CamelEvent.Type.RouteReloaded;
    }

    public String toString() {
        return "MainDurationEventNotifier[" + this.maxMessages + " max messages]";
    }

    protected void doInit() throws Exception {
        super.doInit();
        if (this.action == null) {
            throw new IllegalArgumentException("Unknown action: " + this.action);
        }
    }

    protected void doStart() throws Exception {
        if (this.maxIdleSeconds > 0L) {
            this.camelContext.addStartupListener((context, alreadyStarted) -> {
                this.watch = new StopWatch();
            });
            this.idleExecutorService = Executors.newSingleThreadScheduledExecutor();
            this.idleExecutorService.scheduleAtFixedRate(this::idleTask, 1L, 1L, TimeUnit.SECONDS);
        }
    }

    private void stopTask() {
        if (!this.shutdownStrategy.isRunAllowed()) {
            return;
        }
        try {
            this.camelContext.getRouteController().stopAllRoutes();
        }
        catch (Exception e) {
            LOG.warn("Error during stopping all routes. This exception is ignored.", (Throwable)e);
        }
    }

    private void shutdownTask() {
        if (!this.shutdownStrategy.isRunAllowed()) {
            return;
        }
        if (this.idleExecutorService != null) {
            this.idleExecutorService.shutdownNow();
        }
        try {
            if (this.stopCamelContext) {
                this.camelContext.stop();
            }
        }
        catch (Exception e) {
            LOG.warn("Error during stopping CamelContext. This exception is ignored.", (Throwable)e);
        }
        finally {
            this.shutdownStrategy.shutdown();
        }
    }

    private void idleTask() {
        if (!this.shutdownStrategy.isRunAllowed()) {
            return;
        }
        if (this.watch == null) {
            return;
        }
        int inflight = this.camelContext.getInflightRepository().size();
        if (inflight > 0) {
            LOG.trace("Duration max idle check is skipped due {} inflight messages", (Object)inflight);
            return;
        }
        long seconds = this.watch.taken() / 1000L;
        boolean result = seconds >= this.maxIdleSeconds;
        LOG.trace("Duration max idle check {} >= {} -> {}", new Object[]{seconds, this.maxIdleSeconds, result});
        if (result && this.shutdownStrategy.isRunAllowed()) {
            this.triggerIdleEvent();
        }
    }

    private void triggerIdleEvent() {
        if (this.action == Action.SHUTDOWN) {
            LOG.info("Duration max idle triggering shutdown of the JVM");
            this.camelContext.getExecutorServiceManager().newThread("CamelMainShutdownCamelContext", this::shutdownTask).start();
        } else if (this.action == Action.STOP) {
            LOG.info("Duration max idle triggering stopping all routes");
            this.camelContext.getExecutorServiceManager().newThread("CamelMainShutdownCamelContext", this::stopTask).start();
        }
    }

    private static enum Action {
        SHUTDOWN,
        STOP;


        static Action toAction(String action) {
            if ("shutdown".equals(action)) {
                return SHUTDOWN;
            }
            if ("stop".equals(action)) {
                return STOP;
            }
            LOG.warn("Invalid action: {}. Main execution will be aborted during initialization", (Object)action);
            return null;
        }
    }
}

