/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.salesforce.internal;

import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.PrivateKey;
import java.security.Signature;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.security.auth.DestroyFailedException;
import org.apache.camel.CamelContext;
import org.apache.camel.component.salesforce.AuthenticationType;
import org.apache.camel.component.salesforce.SalesforceHttpClient;
import org.apache.camel.component.salesforce.SalesforceLoginConfig;
import org.apache.camel.component.salesforce.api.SalesforceException;
import org.apache.camel.component.salesforce.api.dto.RestError;
import org.apache.camel.component.salesforce.api.utils.JsonUtils;
import org.apache.camel.component.salesforce.internal.dto.LoginError;
import org.apache.camel.component.salesforce.internal.dto.LoginToken;
import org.apache.camel.support.jsse.KeyStoreParameters;
import org.apache.camel.support.service.ServiceSupport;
import org.apache.camel.util.ObjectHelper;
import org.eclipse.jetty.client.ContentResponse;
import org.eclipse.jetty.client.FormRequestContent;
import org.eclipse.jetty.client.Request;
import org.eclipse.jetty.client.transport.HttpConversation;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.util.Fields;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SalesforceSession
extends ServiceSupport {
    private static final String JWT_SIGNATURE_ALGORITHM = "SHA256withRSA";
    private static final int JWT_CLAIM_WINDOW = 270;
    private static final String JWT_HEADER = Base64.getUrlEncoder().encodeToString("{\"alg\":\"RS256\"}".getBytes(StandardCharsets.UTF_8));
    private static final String OAUTH2_REVOKE_PATH = "/services/oauth2/revoke?token=";
    private static final String OAUTH2_TOKEN_PATH = "/services/oauth2/token";
    private static final Logger LOG = LoggerFactory.getLogger(SalesforceSession.class);
    private final SalesforceHttpClient httpClient;
    private final long timeout;
    private final SalesforceLoginConfig config;
    private final ObjectMapper objectMapper;
    private final Set<SalesforceSessionListener> listeners;
    private volatile String accessToken;
    private volatile String instanceUrl;
    private volatile String id;
    private volatile String orgId;
    private final CamelContext camelContext;
    private final AtomicBoolean loggingIn = new AtomicBoolean();
    private CountDownLatch latch = new CountDownLatch(1);

    public SalesforceSession(CamelContext camelContext, SalesforceHttpClient httpClient, long timeout, SalesforceLoginConfig config) {
        this.camelContext = camelContext;
        ObjectHelper.notNull((Object)((Object)httpClient), (String)"httpClient");
        ObjectHelper.notNull((Object)config, (String)"SalesforceLoginConfig");
        config.validate();
        this.httpClient = httpClient;
        this.timeout = timeout;
        this.config = config;
        this.objectMapper = JsonUtils.createObjectMapper();
        this.listeners = new CopyOnWriteArraySet<SalesforceSessionListener>();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public void attemptLoginUntilSuccessful(long backoffIncrement, long maxBackoff) {
        if (!this.loggingIn.compareAndSet(false, true)) {
            LOG.debug("waiting on login from another thread");
            try {
                while (this.latch == null) {
                    Thread.sleep(100L);
                }
                this.latch.await();
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Failed to login.", ex);
            }
            LOG.debug("done waiting");
            return;
        }
        LOG.debug("Attempting to login, no other threads logging in");
        this.latch = new CountDownLatch(1);
        long backoff = 0L;
        while (true) {
            try {
                if (this.isStoppingOrStopped()) {
                    this.loggingIn.set(false);
                    this.latch.countDown();
                    return;
                }
                this.login(this.getAccessToken());
                break;
            }
            catch (SalesforceException e) {
                if ((backoff += backoffIncrement) > maxBackoff) {
                    backoff = maxBackoff;
                }
                LOG.warn(String.format("Salesforce login failed. Pausing for %d milliseconds", backoff), (Throwable)((Object)e));
                try {
                    Thread.sleep(backoff);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Failed to login.", ex);
                }
            }
        }
        this.loggingIn.set(false);
        this.latch.countDown();
        return;
        catch (Throwable throwable) {
            this.loggingIn.set(false);
            this.latch.countDown();
            throw throwable;
        }
    }

    public synchronized String login(String oldToken) throws SalesforceException {
        if (this.accessToken == null || this.accessToken.equals(oldToken)) {
            this.accessToken = oldToken;
            if (this.accessToken != null) {
                try {
                    this.logout();
                }
                catch (SalesforceException e) {
                    LOG.warn("Error revoking old access token: {}", (Object)e.getMessage(), (Object)e);
                }
                this.accessToken = null;
            }
            Request loginPost = this.getLoginRequest(null);
            try {
                ContentResponse loginResponse = loginPost.send();
                this.parseLoginResponse(loginResponse, loginResponse.getContentAsString());
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new SalesforceException("Login error: interrupted", e);
            }
            catch (TimeoutException e) {
                throw new SalesforceException("Login request timeout: " + e.getMessage(), e);
            }
            catch (ExecutionException e) {
                throw new SalesforceException("Unexpected login error: " + e.getCause().getMessage(), e.getCause());
            }
        }
        return this.accessToken;
    }

    public Request getLoginRequest(HttpConversation conversation) {
        String loginUrl = (this.instanceUrl == null ? this.config.getLoginUrl() : this.instanceUrl) + OAUTH2_TOKEN_PATH;
        LOG.info("Login at Salesforce loginUrl: {}", (Object)loginUrl);
        Fields fields = new Fields(true);
        fields.put("client_id", this.config.getClientId());
        fields.put("format", "json");
        AuthenticationType type = this.config.getType();
        switch (type) {
            case USERNAME_PASSWORD: {
                fields.put("client_secret", this.config.getClientSecret());
                fields.put("grant_type", "password");
                fields.put("username", this.config.getUserName());
                fields.put("password", this.config.getPassword());
                break;
            }
            case REFRESH_TOKEN: {
                fields.put("client_secret", this.config.getClientSecret());
                fields.put("grant_type", "refresh_token");
                fields.put("refresh_token", this.config.getRefreshToken());
                break;
            }
            case JWT: {
                fields.put("grant_type", "urn:ietf:params:oauth:grant-type:jwt-bearer");
                fields.put("assertion", this.generateJwtAssertion());
                break;
            }
            case CLIENT_CREDENTIALS: {
                fields.put("grant_type", "client_credentials");
                fields.put("client_secret", this.config.getClientSecret());
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported login configuration type: " + type);
            }
        }
        Request post = conversation == null ? this.httpClient.POST(loginUrl) : this.httpClient.newHttpRequest(conversation, URI.create(loginUrl)).method(HttpMethod.POST);
        return post.body((Request.Content)new FormRequestContent(fields)).timeout(this.timeout, TimeUnit.MILLISECONDS);
    }

    String generateJwtAssertion() {
        long utcPlusWindow = Clock.systemUTC().millis() / 1000L + 270L;
        String audience = this.config.getJwtAudience() != null ? this.config.getJwtAudience() : this.config.getLoginUrl();
        StringBuilder claim = new StringBuilder().append("{\"iss\":\"").append(this.config.getClientId()).append("\",\"sub\":\"").append(this.config.getUserName()).append("\",\"aud\":\"").append(audience).append("\",\"exp\":\"").append(utcPlusWindow).append("\"}");
        StringBuilder token = new StringBuilder(JWT_HEADER).append('.').append(Base64.getUrlEncoder().encodeToString(claim.toString().getBytes(StandardCharsets.UTF_8)));
        KeyStoreParameters keyStoreParameters = this.config.getKeystore();
        keyStoreParameters.setCamelContext(this.camelContext);
        try {
            KeyStore keystore = keyStoreParameters.createKeyStore();
            Enumeration<String> aliases = keystore.aliases();
            String alias = null;
            while (aliases.hasMoreElements()) {
                String tmp = aliases.nextElement();
                if (!keystore.isKeyEntry(tmp)) continue;
                if (alias == null) {
                    alias = tmp;
                    continue;
                }
                throw new IllegalArgumentException("The given keystore `" + keyStoreParameters.getResource() + "` contains more than one key entry, expecting only one");
            }
            PrivateKey key = (PrivateKey)keystore.getKey(alias, keyStoreParameters.getPassword().toCharArray());
            Signature signature = Signature.getInstance(JWT_SIGNATURE_ALGORITHM);
            signature.initSign(key);
            signature.update(token.toString().getBytes(StandardCharsets.UTF_8));
            byte[] signed = signature.sign();
            token.append('.').append(Base64.getUrlEncoder().encodeToString(signed));
            try {
                key.destroy();
            }
            catch (DestroyFailedException ex) {
                LOG.debug("Error destroying private key: {}", (Object)ex.getMessage());
            }
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IllegalStateException(e);
        }
        return token.toString();
    }

    public synchronized void parseLoginResponse(ContentResponse loginResponse, String responseContent) throws SalesforceException {
        int responseStatus = loginResponse.getStatus();
        try {
            switch (responseStatus) {
                case 200: {
                    LoginToken token = (LoginToken)this.objectMapper.readValue(responseContent, LoginToken.class);
                    LOG.info("Login successful");
                    this.accessToken = token.getAccessToken();
                    this.instanceUrl = Optional.ofNullable(this.config.getInstanceUrl()).orElse(token.getInstanceUrl());
                    this.id = token.getId();
                    this.orgId = this.id.substring(this.id.indexOf("id/") + 3, this.id.indexOf("id/") + 21);
                    int lastChar = this.instanceUrl.length() - 1;
                    if (this.instanceUrl.charAt(lastChar) == '/') {
                        this.instanceUrl = this.instanceUrl.substring(0, lastChar);
                    }
                    for (SalesforceSessionListener listener : this.listeners) {
                        try {
                            listener.onLogin(this.accessToken, this.instanceUrl);
                        }
                        catch (Exception t) {
                            LOG.warn("Unexpected error from listener {}: {}", (Object)listener, (Object)t.getMessage());
                        }
                    }
                    break;
                }
                case 400: {
                    LoginError error = (LoginError)this.objectMapper.readValue(responseContent, LoginError.class);
                    String errorCode = error.getError();
                    String msg = String.format("Login error code:[%s] description:[%s]", error.getError(), error.getErrorDescription());
                    ArrayList<RestError> errors = new ArrayList<RestError>();
                    errors.add(new RestError(errorCode, msg));
                    throw new SalesforceException(errors, 400);
                }
                default: {
                    throw new SalesforceException(String.format("Login error status:[%s] reason:[%s]", responseStatus, loginResponse.getReason()), responseStatus);
                }
            }
        }
        catch (IOException e) {
            String msg = "Login error: response parse exception " + e.getMessage();
            throw new SalesforceException(msg, e);
        }
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public synchronized void logout() throws SalesforceException {
        Object logoutUrl;
        if (this.accessToken == null) {
            return;
        }
        try {
            logoutUrl = (this.instanceUrl == null ? this.config.getLoginUrl() : this.instanceUrl) + OAUTH2_REVOKE_PATH + this.accessToken;
            Request logoutGet = this.httpClient.newRequest((String)logoutUrl).timeout(this.timeout, TimeUnit.MILLISECONDS);
            ContentResponse logoutResponse = logoutGet.send();
            int statusCode = logoutResponse.getStatus();
            if (statusCode == 200) {
                LOG.debug("Logout successful");
            } else {
                LOG.debug("Failed to revoke OAuth token. This is expected if the token is invalid or already expired");
            }
            this.accessToken = null;
            this.instanceUrl = null;
            logoutUrl = this.listeners.iterator();
        }
        catch (InterruptedException e) {
            try {
                Thread.currentThread().interrupt();
                throw new SalesforceException("Interrupted while logging out", e);
                catch (ExecutionException e2) {
                    Throwable ex = e2.getCause();
                    throw new SalesforceException("Unexpected logout exception: " + ex.getMessage(), ex);
                }
                catch (TimeoutException e3) {
                    throw new SalesforceException("Logout request TIMEOUT!", e3);
                }
            }
            catch (Throwable throwable) {
                this.accessToken = null;
                this.instanceUrl = null;
                Iterator<SalesforceSessionListener> iterator = this.listeners.iterator();
                while (true) {
                    if (!iterator.hasNext()) {
                        throw throwable;
                    }
                    SalesforceSessionListener listener = iterator.next();
                    try {
                        listener.onLogout();
                    }
                    catch (Exception t) {
                        LOG.warn("Unexpected error from listener {}: {}", (Object)listener, (Object)t.getMessage());
                    }
                }
            }
        }
        while (logoutUrl.hasNext()) {
            SalesforceSessionListener listener = (SalesforceSessionListener)logoutUrl.next();
            try {
                listener.onLogout();
            }
            catch (Exception t) {
                LOG.warn("Unexpected error from listener {}: {}", (Object)listener, (Object)t.getMessage());
                continue;
            }
            break;
        }
        return;
    }

    public String getAccessToken() {
        return this.accessToken;
    }

    public String getInstanceUrl() {
        return this.instanceUrl;
    }

    public String getId() {
        return this.id;
    }

    public String getOrgId() {
        return this.orgId;
    }

    public boolean addListener(SalesforceSessionListener listener) {
        return this.listeners.add(listener);
    }

    public boolean removeListener(SalesforceSessionListener listener) {
        return this.listeners.remove(listener);
    }

    public void doStart() throws Exception {
        this.login(this.accessToken);
    }

    public void doStop() throws Exception {
        this.logout();
    }

    public long getTimeout() {
        return this.timeout;
    }

    public static interface SalesforceSessionListener {
        public void onLogin(String var1, String var2);

        public void onLogout();
    }
}

