/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.converter.stream;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FilterInputStream;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.security.GeneralSecurityException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import org.apache.camel.Exchange;
import org.apache.camel.ExchangePropertyKey;
import org.apache.camel.RuntimeCamelException;
import org.apache.camel.StreamCache;
import org.apache.camel.converter.stream.CipherPair;
import org.apache.camel.spi.StreamCachingStrategy;
import org.apache.camel.spi.Synchronization;
import org.apache.camel.spi.UnitOfWork;
import org.apache.camel.support.SynchronizationAdapter;
import org.apache.camel.util.FileUtil;
import org.apache.camel.util.IOHelper;
import org.apache.camel.util.ObjectHelper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class FileInputStreamCache
extends InputStream
implements StreamCache {
    private final Lock lock = new ReentrantLock();
    private InputStream stream;
    private final long length;
    private final TempFileManager tempFileManager;
    private final File file;
    private final CipherPair ciphers;

    public FileInputStreamCache(File file) {
        this(new TempFileManager(file, true));
    }

    FileInputStreamCache(TempFileManager closer) {
        this.file = closer.getTempFile();
        this.stream = null;
        this.ciphers = closer.getCiphers();
        this.length = this.file.length();
        this.tempFileManager = closer;
        this.tempFileManager.add(this);
    }

    @Override
    public void close() {
        if (this.stream != null) {
            IOHelper.close((Closeable)this.stream);
        }
    }

    @Override
    public void reset() {
        this.lock.lock();
        try {
            this.close();
            this.stream = null;
            if (!this.file.exists()) {
                throw new RuntimeCamelException("Cannot reset stream from file " + this.file);
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public void writeTo(OutputStream os) throws IOException {
        if (this.stream == null && this.ciphers == null) {
            Files.copy(this.file.toPath(), os);
        } else {
            IOHelper.copy((InputStream)this.getInputStream(), (OutputStream)os);
        }
    }

    public StreamCache copy(Exchange exchange) throws IOException {
        this.tempFileManager.addExchange(exchange);
        return new FileInputStreamCache(this.tempFileManager);
    }

    public boolean inMemory() {
        return false;
    }

    public long length() {
        return this.length;
    }

    public long position() {
        return -1L;
    }

    @Override
    public int available() throws IOException {
        return this.getInputStream().available();
    }

    @Override
    public int read() throws IOException {
        return this.getInputStream().read();
    }

    @Override
    public int read(byte[] b) throws IOException {
        return this.getInputStream().read(b);
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        return this.getInputStream().read(b, off, len);
    }

    @Override
    public long skip(long n) throws IOException {
        return this.getInputStream().skip(n);
    }

    @Override
    public byte[] readAllBytes() throws IOException {
        return this.getInputStream().readAllBytes();
    }

    @Override
    public byte[] readNBytes(int len) throws IOException {
        return this.getInputStream().readNBytes(len);
    }

    @Override
    public int readNBytes(byte[] b, int off, int len) throws IOException {
        return this.getInputStream().readNBytes(b, off, len);
    }

    @Override
    public long transferTo(OutputStream out) throws IOException {
        return this.getInputStream().transferTo(out);
    }

    private InputStream getInputStream() throws IOException {
        if (this.stream == null) {
            this.stream = this.createInputStream(this.file);
        }
        return this.stream;
    }

    private InputStream createInputStream(File file) throws IOException {
        FilterInputStream in = new BufferedInputStream(Files.newInputStream(file.toPath(), StandardOpenOption.READ));
        if (this.ciphers != null) {
            in = new CipherInputStream(in, this.ciphers.createDecryptor()){
                boolean closed;

                @Override
                public void close() throws IOException {
                    if (!this.closed) {
                        super.close();
                        this.closed = true;
                    }
                }
            };
        }
        return in;
    }

    static class TempFileManager {
        private static final Logger LOG = LoggerFactory.getLogger(TempFileManager.class);
        private final boolean closedOnCompletion;
        private final Lock lock = new ReentrantLock();
        private final AtomicInteger exchangeCounter = new AtomicInteger();
        private File tempFile;
        private OutputStream outputStream;
        private CipherPair ciphers;
        private List<FileInputStreamCache> fileInputStreamCaches;

        private TempFileManager(File file, boolean closedOnCompletion) {
            this(closedOnCompletion);
            this.tempFile = file;
        }

        TempFileManager(boolean closedOnCompletion) {
            this.closedOnCompletion = closedOnCompletion;
        }

        void add(FileInputStreamCache fileInputStreamCache) {
            this.lock.lock();
            try {
                if (this.fileInputStreamCaches == null) {
                    this.fileInputStreamCaches = new ArrayList<FileInputStreamCache>(3);
                }
                this.fileInputStreamCaches.add(fileInputStreamCache);
            }
            finally {
                this.lock.unlock();
            }
        }

        void addExchange(Exchange exchange) {
            if (this.closedOnCompletion) {
                this.exchangeCounter.incrementAndGet();
                SynchronizationAdapter onCompletion = new SynchronizationAdapter(){

                    @Override
                    public void onDone(Exchange exchange) {
                        int actualExchanges = exchangeCounter.decrementAndGet();
                        if (actualExchanges == 0) {
                            try {
                                this.closeFileInputStreams();
                                if (outputStream != null) {
                                    outputStream.close();
                                }
                            }
                            catch (Exception e) {
                                LOG.warn("Error closing streams. This exception will be ignored.", (Throwable)e);
                            }
                            try {
                                this.cleanUpTempFile();
                            }
                            catch (Exception e) {
                                LOG.warn("Error deleting temporary cache file: {}. This exception will be ignored.", (Object)tempFile, (Object)e);
                            }
                        }
                    }

                    public String toString() {
                        return "OnCompletion[CachedOutputStream]";
                    }
                };
                UnitOfWork streamCacheUnitOfWork = (UnitOfWork)exchange.getProperty(ExchangePropertyKey.STREAM_CACHE_UNIT_OF_WORK, UnitOfWork.class);
                if (streamCacheUnitOfWork != null && streamCacheUnitOfWork.getRoute() != null) {
                    streamCacheUnitOfWork.addSynchronization((Synchronization)onCompletion);
                } else {
                    exchange.getExchangeExtension().addOnCompletion((Synchronization)onCompletion);
                }
            }
        }

        OutputStream createOutputStream(StreamCachingStrategy strategy) throws IOException {
            if (this.tempFile != null) {
                throw new IllegalStateException("The method 'createOutputStream' can only be called once!");
            }
            if (this.closedOnCompletion && this.exchangeCounter.get() == 0) {
                String error = "Cannot create a FileOutputStream for Stream Caching, because this FileOutputStream would never be removed from the file system. This situation can happen with a Splitter or Multi Cast in parallel processing if there is a timeout set on the Splitter or Multi Cast,  and the processing in a sub-branch takes longer than the timeout. Consider to increase the timeout.";
                LOG.error(error);
                throw new IOException(error);
            }
            this.tempFile = FileUtil.createTempFile((String)"cos", (String)".tmp", (File)strategy.getSpoolDirectory());
            LOG.trace("Creating temporary stream cache file: {}", (Object)this.tempFile);
            FilterOutputStream out = new BufferedOutputStream(Files.newOutputStream(this.tempFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.WRITE));
            if (ObjectHelper.isNotEmpty((String)strategy.getSpoolCipher())) {
                try {
                    if (this.ciphers == null) {
                        this.ciphers = new CipherPair(strategy.getSpoolCipher());
                    }
                }
                catch (GeneralSecurityException e) {
                    throw new IOException(e.getMessage(), e);
                }
                out = new CipherOutputStream(out, this.ciphers.getEncryptor()){
                    boolean closed;

                    @Override
                    public void close() throws IOException {
                        if (!this.closed) {
                            super.close();
                            this.closed = true;
                        }
                    }
                };
            }
            this.outputStream = out;
            return out;
        }

        FileInputStreamCache newStreamCache() {
            return new FileInputStreamCache(this);
        }

        void closeFileInputStreams() {
            if (this.fileInputStreamCaches != null) {
                for (FileInputStreamCache fileInputStreamCache : this.fileInputStreamCaches) {
                    fileInputStreamCache.close();
                }
                this.fileInputStreamCaches.clear();
            }
        }

        void cleanUpTempFile() {
            try {
                if (this.tempFile != null) {
                    FileUtil.deleteFile((File)this.tempFile);
                    this.tempFile = null;
                }
            }
            catch (Exception e) {
                LOG.warn("Error deleting temporary cache file: {}. This exception will be ignored.", (Object)this.tempFile, (Object)e);
            }
        }

        File getTempFile() {
            return this.tempFile;
        }

        CipherPair getCiphers() {
            return this.ciphers;
        }
    }
}

