/*
 * Decompiled with CFR 0.152.
 */
package org.apache.drill.exec.store.parquet.columnreaders.batchsizing;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.apache.drill.common.map.CaseInsensitiveMap;
import org.apache.drill.common.types.TypeProtos;
import org.apache.drill.exec.exception.OutOfMemoryException;
import org.apache.drill.exec.record.MaterializedField;
import org.apache.drill.exec.server.options.OptionManager;
import org.apache.drill.exec.store.parquet.columnreaders.ParquetColumnMetadata;
import org.apache.drill.exec.store.parquet.columnreaders.ParquetSchema;
import org.apache.drill.exec.store.parquet.columnreaders.VarLenColumnBulkInput;
import org.apache.drill.exec.store.parquet.columnreaders.batchsizing.BatchOverflowOptimizer;
import org.apache.drill.exec.store.parquet.columnreaders.batchsizing.BatchSizingMemoryUtil;
import org.apache.drill.exec.store.parquet.columnreaders.batchsizing.RecordBatchOverflow;
import org.apache.drill.exec.util.record.RecordBatchStats;
import org.apache.drill.exec.vector.AllocationHelper;
import org.apache.drill.exec.vector.ValueVector;
import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class RecordBatchSizerManager {
    private static final Logger logger = LoggerFactory.getLogger(RecordBatchSizerManager.class);
    private static final int MIN_COLUMN_MEMORY_SZ = VarLenColumnBulkInput.getMinVLColumnMemorySize();
    private final ParquetSchema schema;
    private final long totalRecordsToRead;
    private final BatchOverflowOptimizer overflowOptimizer;
    private final int configRecordsPerBatch;
    private final long configMemorySizePerBatch;
    private int maxRecordsPerBatch;
    private long maxMemorySizePerBatch;
    private int recordsPerBatch;
    private final List<ColumnMemoryInfo> fixedLengthColumns = new ArrayList<ColumnMemoryInfo>();
    private final List<ColumnMemoryInfo> variableLengthColumns = new ArrayList<ColumnMemoryInfo>();
    private final Map<String, ColumnMemoryInfo> columnMemoryInfoMap = CaseInsensitiveMap.newHashMap();
    private boolean columnPrecisionChanged;
    private Map<String, FieldOverflowStateContainer> fieldOverflowMap = CaseInsensitiveMap.newHashMap();
    private final RecordBatchStats.RecordBatchStatsContext batchStatsContext;

    public RecordBatchSizerManager(OptionManager options, ParquetSchema schema, long totalRecordsToRead, RecordBatchStats.RecordBatchStatsContext batchStatsContext) {
        this.schema = schema;
        this.totalRecordsToRead = totalRecordsToRead;
        this.configRecordsPerBatch = (int)options.getLong("store.parquet.flat.batch.num_records");
        this.maxMemorySizePerBatch = this.configMemorySizePerBatch = this.getConfiguredMaxBatchMemory(options);
        this.maxRecordsPerBatch = this.configRecordsPerBatch;
        this.recordsPerBatch = this.configRecordsPerBatch;
        this.overflowOptimizer = new BatchOverflowOptimizer(this.columnMemoryInfoMap);
        this.batchStatsContext = batchStatsContext;
    }

    public void setup() {
        this.maxMemorySizePerBatch = this.normalizeMemorySizePerBatch();
        this.maxRecordsPerBatch = this.normalizeNumRecordsPerBatch();
        this.loadColumnsPrecisionInfo();
        if (this.getNumColumns() == 0) {
            return;
        }
        this.assignColumnsBatchMemory();
        this.overflowOptimizer.setup();
    }

    public ParquetSchema getSchema() {
        return this.schema;
    }

    public RecordBatchStats.RecordBatchStatsContext getBatchStatsContext() {
        return this.batchStatsContext;
    }

    public void allocate(Map<String, ValueVector> vectorMap) throws OutOfMemoryException {
        if (this.columnPrecisionChanged) {
            this.assignColumnsBatchMemory();
        }
        try {
            for (ValueVector v : vectorMap.values()) {
                ColumnMemoryInfo columnMemoryInfo = this.columnMemoryInfoMap.get(v.getField().getName());
                if (columnMemoryInfo != null) {
                    Preconditions.checkState(columnMemoryInfo.columnPrecision <= Integer.MAX_VALUE, "Column precision cannot exceed 2GB");
                    AllocationHelper.allocate(v, this.recordsPerBatch, (int)columnMemoryInfo.columnPrecision, 0);
                    continue;
                }
                AllocationHelper.allocate(v, this.recordsPerBatch, 0, 0);
            }
        }
        catch (NullPointerException e) {
            throw new OutOfMemoryException();
        }
    }

    public Map<String, FieldOverflowStateContainer> getFieldOverflowMap() {
        return this.fieldOverflowMap;
    }

    public FieldOverflowStateContainer getFieldOverflowContainer(String field) {
        return this.fieldOverflowMap.get(field);
    }

    public boolean releaseFieldOverflowContainer(String field) {
        return this.releaseFieldOverflowContainer(field, true);
    }

    public ColumnMemoryQuota getCurrentFieldBatchMemory(String field) {
        return this.columnMemoryInfoMap.get((Object)field).columnMemoryQuota;
    }

    public int getCurrentRecordsPerBatch() {
        return this.recordsPerBatch;
    }

    public long getCurrentMemorySizePerBatch() {
        return this.maxMemorySizePerBatch;
    }

    public int getConfigRecordsPerBatch() {
        return this.configRecordsPerBatch;
    }

    public long getConfigMemorySizePerBatch() {
        return this.configMemorySizePerBatch;
    }

    public void onEndOfBatch(int batchNumRecords, List<VarLenColumnBatchStats> batchStats) {
        this.columnPrecisionChanged = this.overflowOptimizer.onEndOfBatch(batchNumRecords, batchStats);
    }

    public void close() {
        for (String field : this.fieldOverflowMap.keySet()) {
            this.releaseFieldOverflowContainer(field, false);
        }
        this.fieldOverflowMap.clear();
    }

    private long getConfiguredMaxBatchMemory(OptionManager options) {
        long maxMemory = options.getLong("store.parquet.flat.batch.memory_size");
        if (maxMemory <= 0L) {
            maxMemory = options.getLong("drill.exec.memory.operator.output_batch_size");
        }
        return maxMemory;
    }

    private int normalizeNumRecordsPerBatch() {
        String message;
        int normalizedNumRecords = this.configRecordsPerBatch;
        if (this.configRecordsPerBatch <= 0) {
            String message2 = String.format("Invalid Parquet number of record(s) per batch [%d]", this.configRecordsPerBatch);
            throw new IllegalArgumentException(message2);
        }
        if ((long)normalizedNumRecords > this.totalRecordsToRead) {
            if (logger.isDebugEnabled()) {
                message = String.format("The requested number of record(s) to read is lower than the records per batch; updating the number of record(s) per batch from [%d] to [%d]", normalizedNumRecords, this.totalRecordsToRead);
                logger.debug(message);
            }
            normalizedNumRecords = (int)this.totalRecordsToRead;
        }
        if (this.batchStatsContext.isEnableBatchSzLogging()) {
            message = String.format("The Parquet reader number of record(s) has been set to [%d]", normalizedNumRecords);
            RecordBatchStats.logRecordBatchStats(message, this.batchStatsContext);
        }
        return normalizedNumRecords;
    }

    private long normalizeMemorySizePerBatch() {
        long normalizedMemorySize = this.configMemorySizePerBatch;
        if (normalizedMemorySize <= 0L) {
            String message = String.format("Invalid Parquet memory per batch [%d] byte(s)", this.configMemorySizePerBatch);
            throw new IllegalArgumentException(message);
        }
        int numColumns = this.schema.getColumnMetadata().size();
        if (numColumns == 0) {
            return normalizedMemorySize;
        }
        long memorySizePerColumn = normalizedMemorySize / (long)numColumns;
        if (memorySizePerColumn < (long)MIN_COLUMN_MEMORY_SZ) {
            long prevValue = normalizedMemorySize;
            normalizedMemorySize = MIN_COLUMN_MEMORY_SZ * numColumns;
            String message = String.format("The Parquet memory per batch [%d] byte(s) is too low for this query ; using [%d] bytes", prevValue, normalizedMemorySize);
            logger.warn(message);
        }
        if (this.batchStatsContext.isEnableBatchSzLogging()) {
            RecordBatchStats.printConfiguredBatchSize(this.batchStatsContext, (int)normalizedMemorySize);
        }
        return normalizedMemorySize;
    }

    private void loadColumnsPrecisionInfo() {
        assert (this.fixedLengthColumns.size() == 0);
        assert (this.variableLengthColumns.size() == 0);
        for (ParquetColumnMetadata columnMetadata : this.schema.getColumnMetadata()) {
            assert (!columnMetadata.isRepeated()) : "This reader doesn't handle repeated columns..";
            ColumnMemoryInfo columnMemoryInfo = new ColumnMemoryInfo();
            this.columnMemoryInfoMap.put(columnMetadata.getField().getName(), columnMemoryInfo);
            if (columnMetadata.isFixedLength()) {
                columnMemoryInfo.columnMeta = columnMetadata;
                columnMemoryInfo.columnPrecision = BatchSizingMemoryUtil.getFixedColumnTypePrecision(columnMetadata);
                columnMemoryInfo.columnMemoryQuota.reset();
                this.fixedLengthColumns.add(columnMemoryInfo);
                continue;
            }
            columnMemoryInfo.columnMeta = columnMetadata;
            columnMemoryInfo.columnPrecision = BatchSizingMemoryUtil.getAvgVariableLengthColumnTypePrecision(columnMetadata);
            columnMemoryInfo.columnMemoryQuota.reset();
            this.variableLengthColumns.add(columnMemoryInfo);
        }
    }

    private void assignColumnsBatchMemory() {
        if (this.getNumColumns() == 0 || this.maxRecordsPerBatch == 0) {
            return;
        }
        int originalRecordsPerBatch = this.recordsPerBatch = this.maxRecordsPerBatch;
        this.assignFineGrainedMemoryQuota();
        if (this.batchStatsContext.isEnableBatchSzLogging()) {
            assert (this.recordsPerBatch <= this.maxRecordsPerBatch);
            if (originalRecordsPerBatch != this.recordsPerBatch) {
                String message = String.format("The Parquet records per batch [%d] has been decreased to [%d]", originalRecordsPerBatch, this.recordsPerBatch);
                RecordBatchStats.logRecordBatchStats(message, this.batchStatsContext);
            }
            this.dumpColumnMemoryQuotas();
        }
    }

    private void assignFineGrainedMemoryQuota() {
        block4: {
            MemoryRequirementContainer requiredMemory = new MemoryRequirementContainer();
            int newRecordsPerBatch = this.recordsPerBatch;
            do {
                this.recordsPerBatch = newRecordsPerBatch;
                double neededMemoryRatio = this.computeNeededMemoryRatio(requiredMemory);
                assert (neededMemoryRatio <= 1.0);
                newRecordsPerBatch = (int)((double)this.recordsPerBatch * neededMemoryRatio);
                assert (newRecordsPerBatch <= this.recordsPerBatch);
                if (newRecordsPerBatch > 1) continue;
                this.recordsPerBatch = 1;
                this.computeNeededMemoryRatio(requiredMemory);
                break block4;
            } while (newRecordsPerBatch < this.recordsPerBatch);
            assert (this.recordsPerBatch == newRecordsPerBatch);
            this.distributeExtraMemorySpace(requiredMemory);
        }
    }

    private void distributeExtraMemorySpace(MemoryRequirementContainer requiredMemory) {
        int numVariableLengthColumns = this.variableLengthColumns.size();
        if (numVariableLengthColumns == 0) {
            return;
        }
        long totalMemoryNeeded = requiredMemory.fixedLenRequiredMemory + requiredMemory.variableLenRequiredMemory;
        long extraMemorySpace = this.maxMemorySizePerBatch - totalMemoryNeeded;
        long perColumnExtraSpace = extraMemorySpace / (long)numVariableLengthColumns;
        if (perColumnExtraSpace == 0L) {
            return;
        }
        for (ColumnMemoryInfo columnInfo : this.variableLengthColumns) {
            columnInfo.columnMemoryQuota.maxMemoryUsage += perColumnExtraSpace;
        }
    }

    private int getNumColumns() {
        return this.fixedLengthColumns.size() + this.variableLengthColumns.size();
    }

    private boolean releaseFieldOverflowContainer(String field, boolean remove) {
        FieldOverflowStateContainer container = this.getFieldOverflowContainer(field);
        if (container == null) {
            return false;
        }
        container.release();
        container.overflowDef = null;
        container.overflowState = null;
        if (remove) {
            this.fieldOverflowMap.remove(field);
        }
        return remove;
    }

    private long computeVectorMemory(ColumnMemoryInfo columnInfo, int numValues) {
        if (columnInfo.columnMeta.isFixedLength()) {
            return BatchSizingMemoryUtil.computeFixedLengthVectorMemory(columnInfo.columnMeta, numValues);
        }
        return BatchSizingMemoryUtil.computeVariableLengthVectorMemory(columnInfo.columnMeta, columnInfo.columnPrecision, numValues);
    }

    private double computeNeededMemoryRatio(MemoryRequirementContainer requiredMemory) {
        requiredMemory.reset();
        for (ColumnMemoryInfo columnInfo : this.fixedLengthColumns) {
            columnInfo.columnMemoryQuota.maxMemoryUsage = this.computeVectorMemory(columnInfo, this.recordsPerBatch);
            columnInfo.columnMemoryQuota.maxNumValues = this.recordsPerBatch;
            requiredMemory.fixedLenRequiredMemory += columnInfo.columnMemoryQuota.maxMemoryUsage;
        }
        for (ColumnMemoryInfo columnInfo : this.variableLengthColumns) {
            columnInfo.columnMemoryQuota.maxMemoryUsage = this.computeVectorMemory(columnInfo, this.recordsPerBatch);
            columnInfo.columnMemoryQuota.maxNumValues = this.recordsPerBatch;
            requiredMemory.variableLenRequiredMemory += columnInfo.columnMemoryQuota.maxMemoryUsage;
        }
        long totalMemoryNeeded = requiredMemory.fixedLenRequiredMemory + requiredMemory.variableLenRequiredMemory;
        assert (totalMemoryNeeded > 0L);
        double neededMemoryRatio = (double)this.maxMemorySizePerBatch / (double)totalMemoryNeeded;
        return neededMemoryRatio > 1.0 ? 1.0 : neededMemoryRatio;
    }

    private void dumpColumnMemoryQuotas() {
        StringBuilder msg = new StringBuilder();
        msg.append(": Field Quotas:\n\tName\tType\tPrec\tQuota\n");
        for (ColumnMemoryInfo columnInfo : this.columnMemoryInfoMap.values()) {
            msg.append("\t");
            msg.append("BATCH_STATS");
            msg.append("\t");
            msg.append(columnInfo.columnMeta.getField().getName());
            msg.append("\t");
            RecordBatchSizerManager.printType(columnInfo.columnMeta.getField(), msg);
            msg.append("\t");
            msg.append(columnInfo.columnPrecision);
            msg.append("\t");
            msg.append(columnInfo.columnMemoryQuota.maxMemoryUsage);
            msg.append("\n");
        }
        RecordBatchStats.logRecordBatchStats(msg.toString(), this.batchStatsContext);
    }

    private static void printType(MaterializedField field, StringBuilder msg) {
        TypeProtos.MajorType type = field.getType();
        msg.append(type.getMinorType().name());
        msg.append(':');
        msg.append(type.getMode().name());
    }

    static final class ColumnMemoryInfo {
        ParquetColumnMetadata columnMeta;
        long columnPrecision;
        final ColumnMemoryQuota columnMemoryQuota = new ColumnMemoryQuota();

        ColumnMemoryInfo() {
        }
    }

    public static final class FieldOverflowStateContainer {
        public RecordBatchOverflow.FieldOverflowDefinition overflowDef;
        public FieldOverflowState overflowState;

        public FieldOverflowStateContainer(RecordBatchOverflow.FieldOverflowDefinition overflowDef, FieldOverflowState overflowState) {
            this.overflowDef = overflowDef;
            this.overflowState = overflowState;
        }

        private void release() {
            if (this.overflowDef != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug(String.format("Releasing a buffer of length %d used to handle overflow data", this.overflowDef.buffer.capacity()));
                }
                this.overflowDef.buffer.release();
            }
            this.overflowDef = null;
            this.overflowState = null;
        }
    }

    public static final class ColumnMemoryQuota {
        private long maxMemoryUsage;
        private int maxNumValues;

        public ColumnMemoryQuota() {
        }

        public ColumnMemoryQuota(long maxMemoryUsage) {
            this.maxMemoryUsage = maxMemoryUsage;
        }

        public long getMaxMemoryUsage() {
            return this.maxMemoryUsage;
        }

        public int getMaxNumValues() {
            return this.maxNumValues;
        }

        void reset() {
            this.maxMemoryUsage = 0L;
            this.maxNumValues = 0;
        }
    }

    static final class MemoryRequirementContainer {
        private long fixedLenRequiredMemory;
        private long variableLenRequiredMemory;

        MemoryRequirementContainer() {
        }

        private void reset() {
            this.fixedLenRequiredMemory = 0L;
            this.variableLenRequiredMemory = 0L;
        }
    }

    public static interface FieldOverflowState {
        public void onNewBatchValuesConsumed(int var1);

        public boolean isOverflowDataFullyConsumed();
    }

    public static final class VarLenColumnBatchStats {
        public final ValueVector vector;
        public final int numValuesRead;

        public VarLenColumnBatchStats(ValueVector vector, int numValuesRead) {
            this.vector = vector;
            this.numValuesRead = numValuesRead;
        }
    }
}

