/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import java.io.Closeable;
import java.io.DataInput;
import java.io.EOFException;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.text.DecimalFormat;
import java.util.Arrays;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.BlacklistedDirectories;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.nio.ch.DirectBuffer;

public class FileUtils {
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);
    private static final double KB = 1024.0;
    private static final double MB = 1048576.0;
    private static final double GB = 1.073741824E9;
    private static final double TB = 1.099511627776E12;
    private static final DecimalFormat df = new DecimalFormat("#.##");
    private static final boolean canCleanDirectBuffers;

    public static void createHardLink(String from, String to) {
        FileUtils.createHardLink(new File(from), new File(to));
    }

    public static void createHardLink(File from, File to) {
        if (to.exists()) {
            throw new RuntimeException("Tried to create duplicate hard link to " + to);
        }
        if (!from.exists()) {
            throw new RuntimeException("Tried to hard link to file that does not exist " + from);
        }
        try {
            Files.createLink(to.toPath(), from.toPath());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, to);
        }
    }

    public static File createTempFile(String prefix, String suffix, File directory) {
        try {
            return File.createTempFile(prefix, suffix, directory);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, directory);
        }
    }

    public static File createTempFile(String prefix, String suffix) {
        return FileUtils.createTempFile(prefix, suffix, new File(System.getProperty("java.io.tmpdir")));
    }

    public static void deleteWithConfirm(String file) {
        FileUtils.deleteWithConfirm(new File(file));
    }

    public static void deleteWithConfirm(File file) {
        assert (file.exists()) : "attempted to delete non-existing file " + file.getName();
        if (logger.isDebugEnabled()) {
            logger.debug("Deleting {}", (Object)file.getName());
        }
        try {
            Files.delete(file.toPath());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, file);
        }
    }

    public static void renameWithOutConfirm(String from, String to) {
        block2: {
            try {
                FileUtils.atomicMoveWithFallback(new File(from).toPath(), new File(to).toPath());
            }
            catch (IOException e) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not move file " + from + " to " + to, (Throwable)e);
            }
        }
    }

    public static void renameWithConfirm(String from, String to) {
        FileUtils.renameWithConfirm(new File(from), new File(to));
    }

    public static void renameWithConfirm(File from, File to) {
        assert (from.exists());
        if (logger.isDebugEnabled()) {
            logger.debug(String.format("Renaming %s to %s", from.getPath(), to.getPath()));
        }
        try {
            FileUtils.atomicMoveWithFallback(from.toPath(), to.toPath());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to rename %s to %s", from.getPath(), to.getPath()), e);
        }
    }

    private static void atomicMoveWithFallback(Path from, Path to) throws IOException {
        try {
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            logger.debug("Could not do an atomic move", (Throwable)e);
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void truncate(String path, long size) {
        RandomAccessFile file;
        try {
            file = new RandomAccessFile(path, "rw");
        }
        catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        try {
            file.getChannel().truncate(size);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, path);
        }
        finally {
            FileUtils.closeQuietly(file);
        }
    }

    public static void closeQuietly(Closeable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void close(Closeable ... cs) throws IOException {
        FileUtils.close(Arrays.asList(cs));
    }

    public static void close(Iterable<? extends Closeable> cs) throws IOException {
        IOException e = null;
        for (Closeable closeable : cs) {
            try {
                if (closeable == null) continue;
                closeable.close();
            }
            catch (IOException ex) {
                e = ex;
                logger.warn("Failed closing stream {}", (Object)closeable, (Object)ex);
            }
        }
        if (e != null) {
            throw e;
        }
    }

    public static String getCanonicalPath(String filename) {
        try {
            return new File(filename).getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, filename);
        }
    }

    public static String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, file);
        }
    }

    public static boolean isCleanerAvailable() {
        return canCleanDirectBuffers;
    }

    public static void clean(MappedByteBuffer buffer) {
        ((DirectBuffer)((Object)buffer)).cleaner().clean();
    }

    public static void createDirectory(String directory) {
        FileUtils.createDirectory(new File(directory));
    }

    public static void createDirectory(File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new FSWriteError((Throwable)new IOException("Failed to mkdirs " + directory), directory);
        }
    }

    public static boolean delete(String file) {
        File f = new File(file);
        return f.delete();
    }

    public static void delete(File ... files) {
        for (File file : files) {
            file.delete();
        }
    }

    public static void deleteAsync(final String file) {
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                FileUtils.deleteWithConfirm(new File(file));
            }
        };
        ScheduledExecutors.nonPeriodicTasks.execute(runnable);
    }

    public static String stringifyFileSize(double value) {
        if (value >= 1.099511627776E12) {
            double d = value / 1.099511627776E12;
            String val = df.format(d);
            return val + " TB";
        }
        if (value >= 1.073741824E9) {
            double d = value / 1.073741824E9;
            String val = df.format(d);
            return val + " GB";
        }
        if (value >= 1048576.0) {
            double d = value / 1048576.0;
            String val = df.format(d);
            return val + " MB";
        }
        if (value >= 1024.0) {
            double d = value / 1024.0;
            String val = df.format(d);
            return val + " KB";
        }
        String val = df.format(value);
        return val + " bytes";
    }

    public static void deleteRecursive(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursive(new File(dir, child));
            }
        }
        FileUtils.deleteWithConfirm(dir);
    }

    public static void skipBytesFully(DataInput in, int bytes) throws IOException {
        int skipped;
        for (int n = 0; n < bytes; n += skipped) {
            skipped = in.skipBytes(bytes - n);
            if (skipped != 0) continue;
            throw new EOFException("EOF after " + n + " bytes out of " + bytes);
        }
    }

    public static void handleCorruptSSTable(CorruptSSTableException e) {
        JVMStabilityInspector.inspectThrowable(e);
        switch (DatabaseDescriptor.getDiskFailurePolicy()) {
            case stop_paranoid: {
                StorageService.instance.stopTransports();
            }
        }
    }

    public static void handleFSError(FSError e) {
        JVMStabilityInspector.inspectThrowable(e);
        switch (DatabaseDescriptor.getDiskFailurePolicy()) {
            case stop_paranoid: 
            case stop: {
                StorageService.instance.stopTransports();
                break;
            }
            case best_effort: {
                File directory;
                BlacklistedDirectories.maybeMarkUnwritable(e.path);
                if (!(e instanceof FSReadError) || (directory = BlacklistedDirectories.maybeMarkUnreadable(e.path)) == null) break;
                Keyspace.removeUnreadableSSTables(directory);
                break;
            }
            case ignore: {
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
    }

    public static long folderSize(File directory) {
        long length = 0L;
        for (File file : directory.listFiles()) {
            if (file.isFile()) {
                length += file.length();
                continue;
            }
            length += FileUtils.folderSize(file);
        }
        return length;
    }

    static {
        boolean canClean = false;
        try {
            ByteBuffer buf = ByteBuffer.allocateDirect(1);
            ((DirectBuffer)((Object)buf)).cleaner().clean();
            canClean = true;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            logger.info("Cannot initialize un-mmaper.  (Are you using a non-Oracle JVM?)  Compacted data files will not be removed promptly.  Consider using an Oracle JVM or using standard disk access mode");
        }
        canCleanDirectBuffers = canClean;
    }
}

