/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.gms;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.Uninterruptibles;
import java.lang.management.ManagementFactory;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import org.apache.cassandra.concurrent.DebuggableScheduledThreadPoolExecutor;
import org.apache.cassandra.concurrent.JMXEnabledThreadPoolExecutor;
import org.apache.cassandra.concurrent.Stage;
import org.apache.cassandra.concurrent.StageManager;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.gms.ApplicationState;
import org.apache.cassandra.gms.EchoMessage;
import org.apache.cassandra.gms.EndpointState;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.gms.GossipDigest;
import org.apache.cassandra.gms.GossipDigestSyn;
import org.apache.cassandra.gms.GossiperMBean;
import org.apache.cassandra.gms.HeartBeatState;
import org.apache.cassandra.gms.IEndpointStateChangeSubscriber;
import org.apache.cassandra.gms.IFailureDetectionEventListener;
import org.apache.cassandra.gms.IFailureDetector;
import org.apache.cassandra.gms.VersionedValue;
import org.apache.cassandra.net.IAsyncCallback;
import org.apache.cassandra.net.MessageIn;
import org.apache.cassandra.net.MessageOut;
import org.apache.cassandra.net.MessagingService;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Gossiper
implements IFailureDetectionEventListener,
GossiperMBean {
    public static final String MBEAN_NAME = "org.apache.cassandra.net:type=Gossiper";
    private static final DebuggableScheduledThreadPoolExecutor executor = new DebuggableScheduledThreadPoolExecutor("GossipTasks");
    static final ApplicationState[] STATES = ApplicationState.values();
    static final List<String> DEAD_STATES = Arrays.asList("removing", "removed", "LEFT", "hibernate");
    static ArrayList<String> SILENT_SHUTDOWN_STATES = new ArrayList();
    private ScheduledFuture<?> scheduledGossipTask;
    private static final ReentrantLock taskLock;
    public static final int intervalInMillis = 1000;
    public static final int QUARANTINE_DELAY;
    private static final Logger logger;
    public static final Gossiper instance;
    public static final long aVeryLongTime = 259200000L;
    private static final long MAX_GENERATION_DIFFERENCE = 31536000L;
    private long fatClientTimeout;
    private final Random random = new Random();
    private final Comparator<InetAddress> inetcomparator = new Comparator<InetAddress>(){

        @Override
        public int compare(InetAddress addr1, InetAddress addr2) {
            return addr1.getHostAddress().compareTo(addr2.getHostAddress());
        }
    };
    private final List<IEndpointStateChangeSubscriber> subscribers = new CopyOnWriteArrayList<IEndpointStateChangeSubscriber>();
    private final Set<InetAddress> liveEndpoints = new ConcurrentSkipListSet<InetAddress>(this.inetcomparator);
    private final Map<InetAddress, Long> unreachableEndpoints = new ConcurrentHashMap<InetAddress, Long>();
    private final Set<InetAddress> seeds = new ConcurrentSkipListSet<InetAddress>(this.inetcomparator);
    final ConcurrentMap<InetAddress, EndpointState> endpointStateMap = new ConcurrentHashMap<InetAddress, EndpointState>();
    private final Map<InetAddress, Long> justRemovedEndpoints = new ConcurrentHashMap<InetAddress, Long>();
    private final Map<InetAddress, Long> expireTimeEndpointMap = new ConcurrentHashMap<InetAddress, Long>();
    private boolean inShadowRound = false;
    private volatile long lastProcessedMessageAt = System.currentTimeMillis();

    private Gossiper() {
        this.fatClientTimeout = QUARANTINE_DELAY / 2;
        FailureDetector.instance.registerFailureDetectionEventListener(this);
        try {
            MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
            mbs.registerMBean(this, new ObjectName(MBEAN_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public void setLastProcessedMessageAt(long timeInMillis) {
        this.lastProcessedMessageAt = timeInMillis;
    }

    public boolean seenAnySeed() {
        for (Map.Entry entry : this.endpointStateMap.entrySet()) {
            if (this.seeds.contains(entry.getKey())) {
                return true;
            }
            try {
                VersionedValue internalIp = ((EndpointState)entry.getValue()).getApplicationState(ApplicationState.INTERNAL_IP);
                if (internalIp == null || !this.seeds.contains(InetAddress.getByName(internalIp.value))) continue;
                return true;
            }
            catch (UnknownHostException e) {
                throw new RuntimeException(e);
            }
        }
        return false;
    }

    public void register(IEndpointStateChangeSubscriber subscriber) {
        this.subscribers.add(subscriber);
    }

    public void unregister(IEndpointStateChangeSubscriber subscriber) {
        this.subscribers.remove(subscriber);
    }

    public Set<InetAddress> getLiveMembers() {
        HashSet<InetAddress> liveMembers = new HashSet<InetAddress>(this.liveEndpoints);
        if (!liveMembers.contains(FBUtilities.getBroadcastAddress())) {
            liveMembers.add(FBUtilities.getBroadcastAddress());
        }
        return liveMembers;
    }

    public Set<InetAddress> getLiveTokenOwners() {
        return StorageService.instance.getLiveRingMembers(true);
    }

    public Set<InetAddress> getUnreachableMembers() {
        return this.unreachableEndpoints.keySet();
    }

    public Set<InetAddress> getUnreachableTokenOwners() {
        HashSet<InetAddress> tokenOwners = new HashSet<InetAddress>();
        for (InetAddress endpoint : this.unreachableEndpoints.keySet()) {
            if (!StorageService.instance.getTokenMetadata().isMember(endpoint)) continue;
            tokenOwners.add(endpoint);
        }
        return tokenOwners;
    }

    public long getEndpointDowntime(InetAddress ep) {
        Long downtime = this.unreachableEndpoints.get(ep);
        if (downtime != null) {
            return TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - downtime);
        }
        return 0L;
    }

    private boolean isShutdown(InetAddress endpoint) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null) {
            return false;
        }
        if (epState.getApplicationState(ApplicationState.STATUS) == null) {
            return false;
        }
        String value = epState.getApplicationState((ApplicationState)ApplicationState.STATUS).value;
        String[] pieces = value.split(VersionedValue.DELIMITER_STR, -1);
        assert (pieces.length > 0);
        String state = pieces[0];
        return state.equals("shutdown");
    }

    @Override
    public void convict(InetAddress endpoint, double phi) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null) {
            return;
        }
        logger.debug("Convicting {} with status {} - alive {}", new Object[]{endpoint, Gossiper.getGossipStatus(epState), epState.isAlive()});
        if (!epState.isAlive()) {
            return;
        }
        if (this.isShutdown(endpoint)) {
            this.markAsShutdown(endpoint);
        } else {
            this.markDead(endpoint, epState);
        }
    }

    protected void markAsShutdown(InetAddress endpoint) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null) {
            return;
        }
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.shutdown(true));
        epState.getHeartBeatState().forceHighestPossibleVersionUnsafe();
        this.markDead(endpoint, epState);
        FailureDetector.instance.forceConviction(endpoint);
    }

    int getMaxEndpointStateVersion(EndpointState epState) {
        int maxVersion = epState.getHeartBeatState().getHeartBeatVersion();
        for (Map.Entry<ApplicationState, VersionedValue> state : epState.states()) {
            maxVersion = Math.max(maxVersion, state.getValue().version);
        }
        return maxVersion;
    }

    private void evictFromMembership(InetAddress endpoint) {
        this.unreachableEndpoints.remove(endpoint);
        this.endpointStateMap.remove(endpoint);
        this.expireTimeEndpointMap.remove(endpoint);
        this.quarantineEndpoint(endpoint);
        if (logger.isDebugEnabled()) {
            logger.debug("evicting {} from gossip", (Object)endpoint);
        }
    }

    public void removeEndpoint(InetAddress endpoint) {
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onRemove(endpoint);
        }
        if (this.seeds.contains(endpoint)) {
            this.buildSeedsList();
            this.seeds.remove(endpoint);
            logger.info("removed {} from seeds, updated seeds list = {}", (Object)endpoint, this.seeds);
        }
        this.liveEndpoints.remove(endpoint);
        this.unreachableEndpoints.remove(endpoint);
        FailureDetector.instance.remove(endpoint);
        MessagingService.instance().resetVersion(endpoint);
        this.quarantineEndpoint(endpoint);
        MessagingService.instance().destroyConnectionPool(endpoint);
        if (logger.isDebugEnabled()) {
            logger.debug("removing endpoint {}", (Object)endpoint);
        }
    }

    private void quarantineEndpoint(InetAddress endpoint) {
        this.quarantineEndpoint(endpoint, System.currentTimeMillis());
    }

    private void quarantineEndpoint(InetAddress endpoint, long quarantineExpiration) {
        this.justRemovedEndpoints.put(endpoint, quarantineExpiration);
    }

    public void replacementQuarantine(InetAddress endpoint) {
        logger.debug("");
        this.quarantineEndpoint(endpoint, System.currentTimeMillis() + (long)QUARANTINE_DELAY);
    }

    public void replacedEndpoint(InetAddress endpoint) {
        this.removeEndpoint(endpoint);
        this.evictFromMembership(endpoint);
        this.replacementQuarantine(endpoint);
    }

    private void makeRandomGossipDigest(List<GossipDigest> gDigests) {
        int generation = 0;
        int maxVersion = 0;
        ArrayList endpoints = new ArrayList(this.endpointStateMap.keySet());
        Collections.shuffle(endpoints, this.random);
        for (InetAddress endpoint : endpoints) {
            EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
            if (epState != null) {
                generation = epState.getHeartBeatState().getGeneration();
                maxVersion = this.getMaxEndpointStateVersion(epState);
            }
            gDigests.add(new GossipDigest(endpoint, generation, maxVersion));
        }
        if (logger.isTraceEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (GossipDigest gDigest : gDigests) {
                sb.append(gDigest);
                sb.append(" ");
            }
            logger.trace("Gossip Digests are : {}", (Object)sb);
        }
    }

    public void advertiseRemoving(InetAddress endpoint, UUID hostId, UUID localHostId) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        int generation = epState.getHeartBeatState().getGeneration();
        logger.info("Removing host: {}", (Object)hostId);
        logger.info("Sleeping for {}ms to ensure {} does not change", (Object)StorageService.RING_DELAY, (Object)endpoint);
        Uninterruptibles.sleepUninterruptibly((long)StorageService.RING_DELAY, (TimeUnit)TimeUnit.MILLISECONDS);
        epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState.getHeartBeatState().getGeneration() != generation) {
            throw new RuntimeException("Endpoint " + endpoint + " generation changed while trying to remove it");
        }
        logger.info("Advertising removal for {}", (Object)endpoint);
        epState.updateTimestamp();
        epState.getHeartBeatState().forceNewerGenerationUnsafe();
        EnumMap<ApplicationState, VersionedValue> states = new EnumMap<ApplicationState, VersionedValue>(ApplicationState.class);
        states.put(ApplicationState.STATUS, StorageService.instance.valueFactory.removingNonlocal(hostId));
        states.put(ApplicationState.REMOVAL_COORDINATOR, StorageService.instance.valueFactory.removalCoordinator(localHostId));
        epState.addApplicationStates(states);
        this.endpointStateMap.put(endpoint, epState);
    }

    public void advertiseTokenRemoved(InetAddress endpoint, UUID hostId) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        epState.updateTimestamp();
        epState.getHeartBeatState().forceNewerGenerationUnsafe();
        long expireTime = Gossiper.computeExpireTime();
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.removedNonlocal(hostId, expireTime));
        logger.info("Completing removal of {}", (Object)endpoint);
        this.addExpireTimeForEndpoint(endpoint, expireTime);
        this.endpointStateMap.put(endpoint, epState);
        Uninterruptibles.sleepUninterruptibly((long)2000L, (TimeUnit)TimeUnit.MILLISECONDS);
    }

    @Override
    public void unsafeAssassinateEndpoint(String address) throws UnknownHostException {
        logger.warn("Gossiper.unsafeAssassinateEndpoint is deprecated and will be removed in the next release; use assassinateEndpoint instead");
        this.assassinateEndpoint(address);
    }

    @Override
    public void assassinateEndpoint(String address) throws UnknownHostException {
        InetAddress endpoint = InetAddress.getByName(address);
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        Collection<Token> tokens = null;
        logger.warn("Assassinating {} via gossip", (Object)endpoint);
        if (epState == null) {
            epState = new EndpointState(new HeartBeatState((int)((System.currentTimeMillis() + 60000L) / 1000L), 9999));
        } else {
            try {
                tokens = StorageService.instance.getTokenMetadata().getTokens(endpoint);
            }
            catch (Throwable th) {
                JVMStabilityInspector.inspectThrowable(th);
                logger.warn("Unable to calculate tokens for {}.  Will use a random one", (Object)address);
                tokens = Collections.singletonList(StorageService.getPartitioner().getRandomToken());
            }
            int generation = epState.getHeartBeatState().getGeneration();
            int heartbeat = epState.getHeartBeatState().getHeartBeatVersion();
            logger.info("Sleeping for {}ms to ensure {} does not change", (Object)StorageService.RING_DELAY, (Object)endpoint);
            Uninterruptibles.sleepUninterruptibly((long)StorageService.RING_DELAY, (TimeUnit)TimeUnit.MILLISECONDS);
            EndpointState newState = (EndpointState)this.endpointStateMap.get(endpoint);
            if (newState == null) {
                logger.warn("Endpoint {} disappeared while trying to assassinate, continuing anyway", (Object)endpoint);
            } else {
                if (newState.getHeartBeatState().getGeneration() != generation) {
                    throw new RuntimeException("Endpoint still alive: " + endpoint + " generation changed while trying to assassinate it");
                }
                if (newState.getHeartBeatState().getHeartBeatVersion() != heartbeat) {
                    throw new RuntimeException("Endpoint still alive: " + endpoint + " heartbeat changed while trying to assassinate it");
                }
            }
            epState.updateTimestamp();
            epState.getHeartBeatState().forceNewerGenerationUnsafe();
        }
        epState.addApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.left(tokens, Gossiper.computeExpireTime()));
        this.handleMajorStateChange(endpoint, epState);
        Uninterruptibles.sleepUninterruptibly((long)4000L, (TimeUnit)TimeUnit.MILLISECONDS);
        logger.warn("Finished assassinating {}", (Object)endpoint);
    }

    public boolean isKnownEndpoint(InetAddress endpoint) {
        return this.endpointStateMap.containsKey(endpoint);
    }

    public int getCurrentGenerationNumber(InetAddress endpoint) {
        return ((EndpointState)this.endpointStateMap.get(endpoint)).getHeartBeatState().getGeneration();
    }

    private boolean sendGossip(MessageOut<GossipDigestSyn> message, Set<InetAddress> epSet) {
        ImmutableList liveEndpoints = ImmutableList.copyOf(epSet);
        int size = liveEndpoints.size();
        if (size < 1) {
            return false;
        }
        int index = size == 1 ? 0 : this.random.nextInt(size);
        InetAddress to = (InetAddress)liveEndpoints.get(index);
        if (logger.isTraceEnabled()) {
            logger.trace("Sending a GossipDigestSyn to {} ...", (Object)to);
        }
        MessagingService.instance().sendOneWay(message, to);
        return this.seeds.contains(to);
    }

    private boolean doGossipToLiveMember(MessageOut<GossipDigestSyn> message) {
        int size = this.liveEndpoints.size();
        if (size == 0) {
            return false;
        }
        return this.sendGossip(message, this.liveEndpoints);
    }

    private void maybeGossipToUnreachableMember(MessageOut<GossipDigestSyn> message) {
        double liveEndpointCount = this.liveEndpoints.size();
        double unreachableEndpointCount = this.unreachableEndpoints.size();
        if (unreachableEndpointCount > 0.0) {
            double prob = unreachableEndpointCount / (liveEndpointCount + 1.0);
            double randDbl = this.random.nextDouble();
            if (randDbl < prob) {
                this.sendGossip(message, this.unreachableEndpoints.keySet());
            }
        }
    }

    private void maybeGossipToSeed(MessageOut<GossipDigestSyn> prod) {
        int size = this.seeds.size();
        if (size > 0) {
            if (size == 1 && this.seeds.contains(FBUtilities.getBroadcastAddress())) {
                return;
            }
            if (this.liveEndpoints.size() == 0) {
                this.sendGossip(prod, this.seeds);
            } else {
                double probability = (double)this.seeds.size() / (double)(this.liveEndpoints.size() + this.unreachableEndpoints.size());
                double randDbl = this.random.nextDouble();
                if (randDbl <= probability) {
                    this.sendGossip(prod, this.seeds);
                }
            }
        }
    }

    public boolean isGossipOnlyMember(InetAddress endpoint) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null) {
            return false;
        }
        return !this.isDeadState(epState) && !StorageService.instance.getTokenMetadata().isMember(endpoint);
    }

    public boolean isSafeForBootstrap(InetAddress endpoint) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (epState == null || this.isDeadState(epState)) {
            return true;
        }
        ArrayList<String> unsafeStatuses = new ArrayList<String>(){
            {
                this.add("");
                this.add("NORMAL");
                this.add("shutdown");
            }
        };
        String status = Gossiper.getGossipStatus(epState);
        return !unsafeStatuses.contains(status);
    }

    private void doStatusCheck() {
        if (logger.isTraceEnabled()) {
            logger.trace("Performing status check ...");
        }
        long now = System.currentTimeMillis();
        long nowNano = System.nanoTime();
        long pending = (Long)((JMXEnabledThreadPoolExecutor)StageManager.getStage((Stage)Stage.GOSSIP)).metrics.pendingTasks.getValue();
        if (pending > 0L && this.lastProcessedMessageAt < now - 1000L) {
            Uninterruptibles.sleepUninterruptibly((long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
            if (this.lastProcessedMessageAt < now - 1000L) {
                logger.warn("Gossip stage has {} pending tasks; skipping status check (no nodes will be marked down)", (Object)pending);
                return;
            }
        }
        Set eps = this.endpointStateMap.keySet();
        for (InetAddress inetAddress : eps) {
            if (inetAddress.equals(FBUtilities.getBroadcastAddress())) continue;
            FailureDetector.instance.interpret(inetAddress);
            EndpointState epState = (EndpointState)this.endpointStateMap.get(inetAddress);
            if (epState == null) continue;
            if (this.isGossipOnlyMember(inetAddress) && !this.justRemovedEndpoints.containsKey(inetAddress) && TimeUnit.NANOSECONDS.toMillis(nowNano - epState.getUpdateTimestamp()) > this.fatClientTimeout) {
                logger.info("FatClient {} has been silent for {}ms, removing from gossip", (Object)inetAddress, (Object)this.fatClientTimeout);
                this.removeEndpoint(inetAddress);
                this.evictFromMembership(inetAddress);
            }
            long expireTime = this.getExpireTimeForEndpoint(inetAddress);
            if (epState.isAlive() || now <= expireTime || StorageService.instance.getTokenMetadata().isMember(inetAddress)) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("time is expiring for endpoint : {} ({})", (Object)inetAddress, (Object)expireTime);
            }
            this.evictFromMembership(inetAddress);
        }
        if (!this.justRemovedEndpoints.isEmpty()) {
            for (Map.Entry entry : this.justRemovedEndpoints.entrySet()) {
                if (now - (Long)entry.getValue() <= (long)QUARANTINE_DELAY) continue;
                if (logger.isDebugEnabled()) {
                    logger.debug("{} elapsed, {} gossip quarantine over", (Object)QUARANTINE_DELAY, entry.getKey());
                }
                this.justRemovedEndpoints.remove(entry.getKey());
            }
        }
    }

    protected long getExpireTimeForEndpoint(InetAddress endpoint) {
        Long storedTime = this.expireTimeEndpointMap.get(endpoint);
        return storedTime == null ? Gossiper.computeExpireTime() : storedTime;
    }

    public EndpointState getEndpointStateForEndpoint(InetAddress ep) {
        return (EndpointState)this.endpointStateMap.get(ep);
    }

    public void resetEndpointStateMap() {
        this.endpointStateMap.clear();
        this.unreachableEndpoints.clear();
        this.liveEndpoints.clear();
    }

    public Set<Map.Entry<InetAddress, EndpointState>> getEndpointStates() {
        return this.endpointStateMap.entrySet();
    }

    public UUID getHostId(InetAddress endpoint) {
        return UUID.fromString(this.getEndpointStateForEndpoint((InetAddress)endpoint).getApplicationState((ApplicationState)ApplicationState.HOST_ID).value);
    }

    EndpointState getStateForVersionBiggerThan(InetAddress forEndpoint, int version) {
        EndpointState epState = (EndpointState)this.endpointStateMap.get(forEndpoint);
        EndpointState reqdEndpointState = null;
        if (epState != null) {
            int localHbVersion = epState.getHeartBeatState().getHeartBeatVersion();
            if (localHbVersion > version) {
                reqdEndpointState = new EndpointState(epState.getHeartBeatState());
                if (logger.isTraceEnabled()) {
                    logger.trace("local heartbeat version {} greater than {} for {}", new Object[]{localHbVersion, version, forEndpoint});
                }
            }
            EnumMap<ApplicationState, VersionedValue> states = new EnumMap<ApplicationState, VersionedValue>(ApplicationState.class);
            for (Map.Entry<ApplicationState, VersionedValue> entry : epState.states()) {
                VersionedValue value = entry.getValue();
                if (value.version <= version) continue;
                if (reqdEndpointState == null) {
                    reqdEndpointState = new EndpointState(epState.getHeartBeatState());
                }
                ApplicationState key = entry.getKey();
                if (logger.isTraceEnabled()) {
                    logger.trace("Adding state {}: {}", (Object)key, (Object)value.value);
                }
                states.put(key, value);
            }
            if (reqdEndpointState != null) {
                reqdEndpointState.addApplicationStates(states);
            }
        }
        return reqdEndpointState;
    }

    public int compareEndpointStartup(InetAddress addr1, InetAddress addr2) {
        EndpointState ep1 = this.getEndpointStateForEndpoint(addr1);
        EndpointState ep2 = this.getEndpointStateForEndpoint(addr2);
        assert (ep1 != null && ep2 != null);
        return ep1.getHeartBeatState().getGeneration() - ep2.getHeartBeatState().getGeneration();
    }

    void notifyFailureDetector(Map<InetAddress, EndpointState> remoteEpStateMap) {
        for (Map.Entry<InetAddress, EndpointState> entry : remoteEpStateMap.entrySet()) {
            this.notifyFailureDetector(entry.getKey(), entry.getValue());
        }
    }

    void notifyFailureDetector(InetAddress endpoint, EndpointState remoteEndpointState) {
        EndpointState localEndpointState = (EndpointState)this.endpointStateMap.get(endpoint);
        if (localEndpointState != null) {
            IFailureDetector fd = FailureDetector.instance;
            int localGeneration = localEndpointState.getHeartBeatState().getGeneration();
            int remoteGeneration = remoteEndpointState.getHeartBeatState().getGeneration();
            if (remoteGeneration > localGeneration) {
                localEndpointState.updateTimestamp();
                if (!localEndpointState.isAlive()) {
                    logger.debug("Clearing interval times for {} due to generation change", (Object)endpoint);
                    fd.remove(endpoint);
                }
                fd.report(endpoint);
                return;
            }
            if (remoteGeneration == localGeneration) {
                int localVersion = this.getMaxEndpointStateVersion(localEndpointState);
                int remoteVersion = remoteEndpointState.getHeartBeatState().getHeartBeatVersion();
                if (remoteVersion > localVersion) {
                    localEndpointState.updateTimestamp();
                    fd.report(endpoint);
                }
            }
        }
    }

    private void markAlive(final InetAddress addr, final EndpointState localState) {
        if (MessagingService.instance().getVersion(addr) < 7) {
            this.realMarkAlive(addr, localState);
            return;
        }
        localState.markDead();
        MessageOut<EchoMessage> echoMessage = new MessageOut<EchoMessage>(MessagingService.Verb.ECHO, EchoMessage.instance, EchoMessage.serializer);
        logger.trace("Sending a EchoMessage to {}", (Object)addr);
        IAsyncCallback echoHandler = new IAsyncCallback(){

            @Override
            public boolean isLatencyForSnitch() {
                return false;
            }

            public void response(MessageIn msg) {
                Gossiper.this.realMarkAlive(addr, localState);
            }
        };
        MessagingService.instance().sendRR(echoMessage, addr, echoHandler);
    }

    @VisibleForTesting
    public void realMarkAlive(InetAddress addr, EndpointState localState) {
        if (logger.isTraceEnabled()) {
            logger.trace("marking as alive {}", (Object)addr);
        }
        localState.markAlive();
        localState.updateTimestamp();
        this.liveEndpoints.add(addr);
        this.unreachableEndpoints.remove(addr);
        this.expireTimeEndpointMap.remove(addr);
        logger.debug("removing expire time for endpoint : {}", (Object)addr);
        logger.info("InetAddress {} is now UP", (Object)addr);
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onAlive(addr, localState);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Notified {}", this.subscribers);
        }
    }

    @VisibleForTesting
    public void markDead(InetAddress addr, EndpointState localState) {
        if (logger.isTraceEnabled()) {
            logger.trace("marking as down {}", (Object)addr);
        }
        localState.markDead();
        this.liveEndpoints.remove(addr);
        this.unreachableEndpoints.put(addr, System.nanoTime());
        logger.info("InetAddress {} is now DOWN", (Object)addr);
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onDead(addr, localState);
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Notified {}", this.subscribers);
        }
    }

    private void handleMajorStateChange(InetAddress ep, EndpointState epState) {
        EndpointState localEpState = (EndpointState)this.endpointStateMap.get(ep);
        if (!this.isDeadState(epState)) {
            if (localEpState != null) {
                logger.info("Node {} has restarted, now UP", (Object)ep);
            } else {
                logger.info("Node {} is now part of the cluster", (Object)ep);
            }
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Adding endpoint state for {}", (Object)ep);
        }
        this.endpointStateMap.put(ep, epState);
        if (localEpState != null) {
            for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
                subscriber.onRestart(ep, localEpState);
            }
        }
        if (!this.isDeadState(epState)) {
            this.markAlive(ep, epState);
        } else {
            logger.debug("Not marking {} alive due to dead state", (Object)ep);
            this.markDead(ep, epState);
        }
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onJoin(ep, epState);
        }
        if (this.isShutdown(ep)) {
            this.markAsShutdown(ep);
        }
    }

    public boolean isAlive(InetAddress endpoint) {
        EndpointState epState = this.getEndpointStateForEndpoint(endpoint);
        if (epState == null) {
            return false;
        }
        return epState.isAlive() && !this.isDeadState(epState);
    }

    public boolean isDeadState(EndpointState epState) {
        String status = Gossiper.getGossipStatus(epState);
        if (status.isEmpty()) {
            return false;
        }
        return DEAD_STATES.contains(status);
    }

    public boolean isSilentShutdownState(EndpointState epState) {
        String status = Gossiper.getGossipStatus(epState);
        if (status.isEmpty()) {
            return false;
        }
        return SILENT_SHUTDOWN_STATES.contains(status);
    }

    private static String getGossipStatus(EndpointState epState) {
        if (epState == null || epState.getApplicationState(ApplicationState.STATUS) == null) {
            return "";
        }
        String value = epState.getApplicationState((ApplicationState)ApplicationState.STATUS).value;
        String[] pieces = value.split(VersionedValue.DELIMITER_STR, -1);
        assert (pieces.length > 0);
        return pieces[0];
    }

    void applyStateLocally(Map<InetAddress, EndpointState> epStateMap) {
        for (Map.Entry<InetAddress, EndpointState> entry : epStateMap.entrySet()) {
            InetAddress ep = entry.getKey();
            if (ep.equals(FBUtilities.getBroadcastAddress()) && !this.isInShadowRound()) continue;
            if (this.justRemovedEndpoints.containsKey(ep)) {
                if (!logger.isTraceEnabled()) continue;
                logger.trace("Ignoring gossip for {} because it is quarantined", (Object)ep);
                continue;
            }
            EndpointState localEpStatePtr = (EndpointState)this.endpointStateMap.get(ep);
            EndpointState remoteState = entry.getValue();
            if (localEpStatePtr != null) {
                int localGeneration = localEpStatePtr.getHeartBeatState().getGeneration();
                int remoteGeneration = remoteState.getHeartBeatState().getGeneration();
                if (logger.isTraceEnabled()) {
                    logger.trace("{} local generation {}, remote generation {}", new Object[]{ep, localGeneration, remoteGeneration});
                }
                if (localGeneration != 0 && (long)remoteGeneration > (long)localGeneration + 31536000L) {
                    logger.warn("received an invalid gossip generation for peer {}; local generation = {}, received generation = {}", new Object[]{ep, localGeneration, remoteGeneration});
                    continue;
                }
                if (remoteGeneration > localGeneration) {
                    if (logger.isTraceEnabled()) {
                        logger.trace("Updating heartbeat state generation to {} from {} for {}", new Object[]{remoteGeneration, localGeneration, ep});
                    }
                    this.handleMajorStateChange(ep, remoteState);
                    continue;
                }
                if (remoteGeneration == localGeneration) {
                    int localMaxVersion = this.getMaxEndpointStateVersion(localEpStatePtr);
                    int remoteMaxVersion = this.getMaxEndpointStateVersion(remoteState);
                    if (remoteMaxVersion > localMaxVersion) {
                        this.applyNewStates(ep, localEpStatePtr, remoteState);
                    } else if (logger.isTraceEnabled()) {
                        logger.trace("Ignoring remote version {} <= {} for {}", new Object[]{remoteMaxVersion, localMaxVersion, ep});
                    }
                    if (localEpStatePtr.isAlive() || this.isDeadState(localEpStatePtr)) continue;
                    this.markAlive(ep, localEpStatePtr);
                    continue;
                }
                if (!logger.isTraceEnabled()) continue;
                logger.trace("Ignoring remote generation {} < {}", (Object)remoteGeneration, (Object)localGeneration);
                continue;
            }
            FailureDetector.instance.report(ep);
            this.handleMajorStateChange(ep, remoteState);
        }
    }

    private void applyNewStates(InetAddress addr, EndpointState localState, EndpointState remoteState) {
        int oldVersion = localState.getHeartBeatState().getHeartBeatVersion();
        localState.setHeartBeatState(remoteState.getHeartBeatState());
        if (logger.isTraceEnabled()) {
            logger.trace("Updating heartbeat state version to {} from {} for {} ...", new Object[]{localState.getHeartBeatState().getHeartBeatVersion(), oldVersion, addr});
        }
        Set<Map.Entry<ApplicationState, VersionedValue>> remoteStates = remoteState.states();
        assert (remoteState.getHeartBeatState().getGeneration() == localState.getHeartBeatState().getGeneration());
        localState.addApplicationStates(remoteStates);
        for (Map.Entry<ApplicationState, VersionedValue> remoteEntry : remoteStates) {
            this.doOnChangeNotifications(addr, remoteEntry.getKey(), remoteEntry.getValue());
        }
    }

    private void doBeforeChangeNotifications(InetAddress addr, EndpointState epState, ApplicationState apState, VersionedValue newValue) {
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.beforeChange(addr, epState, apState, newValue);
        }
    }

    private void doOnChangeNotifications(InetAddress addr, ApplicationState state, VersionedValue value) {
        for (IEndpointStateChangeSubscriber subscriber : this.subscribers) {
            subscriber.onChange(addr, state, value);
        }
    }

    private void requestAll(GossipDigest gDigest, List<GossipDigest> deltaGossipDigestList, int remoteGeneration) {
        deltaGossipDigestList.add(new GossipDigest(gDigest.getEndpoint(), remoteGeneration, 0));
        if (logger.isTraceEnabled()) {
            logger.trace("requestAll for {}", (Object)gDigest.getEndpoint());
        }
    }

    private void sendAll(GossipDigest gDigest, Map<InetAddress, EndpointState> deltaEpStateMap, int maxRemoteVersion) {
        EndpointState localEpStatePtr = this.getStateForVersionBiggerThan(gDigest.getEndpoint(), maxRemoteVersion);
        if (localEpStatePtr != null) {
            deltaEpStateMap.put(gDigest.getEndpoint(), localEpStatePtr);
        }
    }

    void examineGossiper(List<GossipDigest> gDigestList, List<GossipDigest> deltaGossipDigestList, Map<InetAddress, EndpointState> deltaEpStateMap) {
        if (gDigestList.size() == 0) {
            logger.debug("Shadow request received, adding all states");
            for (Map.Entry entry : this.endpointStateMap.entrySet()) {
                gDigestList.add(new GossipDigest((InetAddress)entry.getKey(), 0, 0));
            }
        }
        for (GossipDigest gossipDigest : gDigestList) {
            int remoteGeneration = gossipDigest.getGeneration();
            int maxRemoteVersion = gossipDigest.getMaxVersion();
            EndpointState epStatePtr = (EndpointState)this.endpointStateMap.get(gossipDigest.getEndpoint());
            if (epStatePtr != null) {
                int localGeneration = epStatePtr.getHeartBeatState().getGeneration();
                int maxLocalVersion = this.getMaxEndpointStateVersion(epStatePtr);
                if (remoteGeneration == localGeneration && maxRemoteVersion == maxLocalVersion) continue;
                if (remoteGeneration > localGeneration) {
                    this.requestAll(gossipDigest, deltaGossipDigestList, remoteGeneration);
                    continue;
                }
                if (remoteGeneration < localGeneration) {
                    this.sendAll(gossipDigest, deltaEpStateMap, 0);
                    continue;
                }
                if (remoteGeneration != localGeneration) continue;
                if (maxRemoteVersion > maxLocalVersion) {
                    deltaGossipDigestList.add(new GossipDigest(gossipDigest.getEndpoint(), remoteGeneration, maxLocalVersion));
                    continue;
                }
                if (maxRemoteVersion >= maxLocalVersion) continue;
                this.sendAll(gossipDigest, deltaEpStateMap, maxRemoteVersion);
                continue;
            }
            this.requestAll(gossipDigest, deltaGossipDigestList, remoteGeneration);
        }
    }

    public void start(int generationNumber) {
        this.start(generationNumber, new EnumMap<ApplicationState, VersionedValue>(ApplicationState.class));
    }

    public void start(int generationNbr, Map<ApplicationState, VersionedValue> preloadLocalStates) {
        this.buildSeedsList();
        this.maybeInitializeLocalState(generationNbr);
        EndpointState localState = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        localState.addApplicationStates(preloadLocalStates);
        DatabaseDescriptor.getEndpointSnitch().gossiperStarting();
        if (logger.isTraceEnabled()) {
            logger.trace("gossip started with generation {}", (Object)localState.getHeartBeatState().getGeneration());
        }
        this.scheduledGossipTask = executor.scheduleWithFixedDelay(new GossipTask(), 1000L, 1000L, TimeUnit.MILLISECONDS);
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void doShadowRound() {
        this.buildSeedsList();
        ArrayList<GossipDigest> gDigests = new ArrayList<GossipDigest>();
        GossipDigestSyn digestSynMessage = new GossipDigestSyn(DatabaseDescriptor.getClusterName(), DatabaseDescriptor.getPartitionerName(), gDigests);
        MessageOut<GossipDigestSyn> message = new MessageOut<GossipDigestSyn>(MessagingService.Verb.GOSSIP_DIGEST_SYN, digestSynMessage, GossipDigestSyn.serializer);
        this.inShadowRound = true;
        for (InetAddress seed : this.seeds) {
            MessagingService.instance().sendOneWay(message, seed);
        }
        int slept = 0;
        try {
            do {
                Thread.sleep(1000L);
                if (!this.inShadowRound) return;
            } while ((slept += 1000) <= StorageService.RING_DELAY);
            throw new RuntimeException("Unable to gossip with any seeds");
        }
        catch (InterruptedException wtf) {
            throw new RuntimeException(wtf);
        }
    }

    private void buildSeedsList() {
        for (InetAddress seed : DatabaseDescriptor.getSeeds()) {
            if (seed.equals(FBUtilities.getBroadcastAddress())) continue;
            this.seeds.add(seed);
        }
    }

    public void maybeInitializeLocalState(int generationNbr) {
        HeartBeatState hbState = new HeartBeatState(generationNbr);
        EndpointState localState = new EndpointState(hbState);
        localState.markAlive();
        this.endpointStateMap.putIfAbsent(FBUtilities.getBroadcastAddress(), localState);
    }

    public void forceNewerGeneration() {
        EndpointState epstate = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        epstate.getHeartBeatState().forceNewerGenerationUnsafe();
    }

    public void addSavedEndpoint(InetAddress ep) {
        if (ep.equals(FBUtilities.getBroadcastAddress())) {
            logger.debug("Attempt to add self as saved endpoint");
            return;
        }
        EndpointState epState = (EndpointState)this.endpointStateMap.get(ep);
        if (epState != null) {
            logger.debug("not replacing a previous epState for {}, but reusing it: {}", (Object)ep, (Object)epState);
            epState.setHeartBeatState(new HeartBeatState(0));
        } else {
            epState = new EndpointState(new HeartBeatState(0));
        }
        epState.markDead();
        this.endpointStateMap.put(ep, epState);
        this.unreachableEndpoints.put(ep, System.nanoTime());
        if (logger.isTraceEnabled()) {
            logger.trace("Adding saved endpoint {} {}", (Object)ep, (Object)epState.getHeartBeatState().getGeneration());
        }
    }

    private void addLocalApplicationStateInternal(ApplicationState state, VersionedValue value) {
        assert (taskLock.isHeldByCurrentThread());
        EndpointState epState = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        InetAddress epAddr = FBUtilities.getBroadcastAddress();
        assert (epState != null);
        this.doBeforeChangeNotifications(epAddr, epState, state, value);
        value = StorageService.instance.valueFactory.cloneWithHigherVersion(value);
        epState.addApplicationState(state, value);
        this.doOnChangeNotifications(epAddr, state, value);
    }

    public void addLocalApplicationState(ApplicationState applicationState, VersionedValue value) {
        this.addLocalApplicationStates(Arrays.asList(Pair.create(applicationState, value)));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void addLocalApplicationStates(List<Pair<ApplicationState, VersionedValue>> states) {
        taskLock.lock();
        try {
            for (Pair<ApplicationState, VersionedValue> pair : states) {
                this.addLocalApplicationStateInternal((ApplicationState)((Object)pair.left), (VersionedValue)pair.right);
            }
        }
        finally {
            taskLock.unlock();
        }
    }

    public void stop() {
        EndpointState mystate = (EndpointState)this.endpointStateMap.get(FBUtilities.getBroadcastAddress());
        if (mystate != null && !this.isSilentShutdownState(mystate)) {
            logger.info("Announcing shutdown");
            this.addLocalApplicationState(ApplicationState.STATUS, StorageService.instance.valueFactory.shutdown(true));
            MessageOut message = new MessageOut(MessagingService.Verb.GOSSIP_SHUTDOWN);
            for (InetAddress ep : this.liveEndpoints) {
                MessagingService.instance().sendOneWay(message, ep);
            }
            Uninterruptibles.sleepUninterruptibly((long)Integer.getInteger("cassandra.shutdown_announce_in_ms", 2000).intValue(), (TimeUnit)TimeUnit.MILLISECONDS);
        } else {
            logger.warn("No local state or state is in silent shutdown, not announcing shutdown");
        }
        if (this.scheduledGossipTask != null) {
            this.scheduledGossipTask.cancel(false);
        }
    }

    public boolean isEnabled() {
        return this.scheduledGossipTask != null && !this.scheduledGossipTask.isCancelled();
    }

    protected void finishShadowRound() {
        if (this.inShadowRound) {
            this.inShadowRound = false;
        }
    }

    protected boolean isInShadowRound() {
        return this.inShadowRound;
    }

    @VisibleForTesting
    public void initializeNodeUnsafe(InetAddress addr, UUID uuid, int generationNbr) {
        HeartBeatState hbState = new HeartBeatState(generationNbr);
        EndpointState newState = new EndpointState(hbState);
        newState.markAlive();
        EndpointState oldState = this.endpointStateMap.putIfAbsent(addr, newState);
        EndpointState localState = oldState == null ? newState : oldState;
        EnumMap<ApplicationState, VersionedValue> states = new EnumMap<ApplicationState, VersionedValue>(ApplicationState.class);
        states.put(ApplicationState.NET_VERSION, StorageService.instance.valueFactory.networkVersion());
        states.put(ApplicationState.HOST_ID, StorageService.instance.valueFactory.hostId(uuid));
        localState.addApplicationStates(states);
    }

    @VisibleForTesting
    public void injectApplicationState(InetAddress endpoint, ApplicationState state, VersionedValue value) {
        EndpointState localState = (EndpointState)this.endpointStateMap.get(endpoint);
        localState.addApplicationState(state, value);
    }

    @Override
    public long getEndpointDowntime(String address) throws UnknownHostException {
        return this.getEndpointDowntime(InetAddress.getByName(address));
    }

    @Override
    public int getCurrentGenerationNumber(String address) throws UnknownHostException {
        return this.getCurrentGenerationNumber(InetAddress.getByName(address));
    }

    public void addExpireTimeForEndpoint(InetAddress endpoint, long expireTime) {
        if (logger.isDebugEnabled()) {
            logger.debug("adding expire time for endpoint : {} ({})", (Object)endpoint, (Object)expireTime);
        }
        this.expireTimeEndpointMap.put(endpoint, expireTime);
    }

    public static long computeExpireTime() {
        return System.currentTimeMillis() + 259200000L;
    }

    static {
        SILENT_SHUTDOWN_STATES.addAll(DEAD_STATES);
        SILENT_SHUTDOWN_STATES.add("BOOT");
        taskLock = new ReentrantLock();
        QUARANTINE_DELAY = StorageService.RING_DELAY * 2;
        logger = LoggerFactory.getLogger(Gossiper.class);
        instance = new Gossiper();
    }

    private class GossipTask
    implements Runnable {
        private GossipTask() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                MessagingService.instance().waitUntilListening();
                taskLock.lock();
                ((EndpointState)Gossiper.this.endpointStateMap.get(FBUtilities.getBroadcastAddress())).getHeartBeatState().updateHeartBeat();
                if (logger.isTraceEnabled()) {
                    logger.trace("My heartbeat is now {}", (Object)((EndpointState)Gossiper.this.endpointStateMap.get(FBUtilities.getBroadcastAddress())).getHeartBeatState().getHeartBeatVersion());
                }
                ArrayList<GossipDigest> gDigests = new ArrayList<GossipDigest>();
                instance.makeRandomGossipDigest(gDigests);
                if (gDigests.size() > 0) {
                    GossipDigestSyn digestSynMessage = new GossipDigestSyn(DatabaseDescriptor.getClusterName(), DatabaseDescriptor.getPartitionerName(), gDigests);
                    MessageOut<GossipDigestSyn> message = new MessageOut<GossipDigestSyn>(MessagingService.Verb.GOSSIP_DIGEST_SYN, digestSynMessage, GossipDigestSyn.serializer);
                    boolean gossipedToSeed = Gossiper.this.doGossipToLiveMember(message);
                    Gossiper.this.maybeGossipToUnreachableMember(message);
                    if (!gossipedToSeed || Gossiper.this.liveEndpoints.size() < Gossiper.this.seeds.size()) {
                        Gossiper.this.maybeGossipToSeed(message);
                    }
                    Gossiper.this.doStatusCheck();
                }
            }
            catch (Exception e) {
                JVMStabilityInspector.inspectThrowable(e);
                logger.error("Gossip error", (Throwable)e);
            }
            finally {
                taskLock.unlock();
            }
        }
    }
}

