/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.service;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Directories;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.StartupException;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.service.StartupCheck;
import org.apache.cassandra.utils.CLibrary;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.SigarLibrary;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StartupChecks {
    private static final Logger logger = LoggerFactory.getLogger(StartupChecks.class);
    private final List<StartupCheck> preFlightChecks = new ArrayList<StartupCheck>();
    private final List<StartupCheck> DEFAULT_TESTS = ImmutableList.of((Object)checkJemalloc, (Object)checkValidLaunchDate, (Object)checkJMXPorts, (Object)inspectJvmOptions, (Object)checkJnaInitialization, (Object)initSigarLibrary, (Object)checkDataDirs, (Object)checkSSTablesFormat, (Object)checkSystemKeyspaceState);
    public static final StartupCheck checkJemalloc = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            if (FBUtilities.isWindows()) {
                return;
            }
            String jemalloc = System.getProperty("cassandra.libjemalloc");
            if (jemalloc == null) {
                logger.warn("jemalloc shared library could not be preloaded to speed up memory allocations");
            } else if ("-".equals(jemalloc)) {
                logger.info("jemalloc preload explicitly disabled");
            } else {
                logger.info("jemalloc seems to be preloaded from {}", (Object)jemalloc);
            }
        }
    };
    public static final StartupCheck checkValidLaunchDate = new StartupCheck(){
        private static final long EARLIEST_LAUNCH_DATE = 1215820800000L;

        @Override
        public void execute() throws StartupException {
            long now = System.currentTimeMillis();
            if (now < 1215820800000L) {
                throw new StartupException(1, String.format("current machine time is %s, but that is seemingly incorrect. exiting now.", new Date(now).toString()));
            }
        }
    };
    public static final StartupCheck checkJMXPorts = new StartupCheck(){

        @Override
        public void execute() {
            String jmxPort = System.getProperty("com.sun.management.jmxremote.port");
            if (jmxPort == null) {
                logger.warn("JMX is not enabled to receive remote connections. Please see cassandra-env.sh for more info.");
                jmxPort = System.getProperty("cassandra.jmx.local.port");
                if (jmxPort == null) {
                    logger.error("cassandra.jmx.local.port missing from cassandra-env.sh, unable to start local JMX service.");
                }
            } else {
                logger.info("JMX is enabled to receive remote connections on port: " + jmxPort);
            }
        }
    };
    public static final StartupCheck inspectJvmOptions = new StartupCheck(){

        @Override
        public void execute() {
            String javaVmName;
            if (!DatabaseDescriptor.hasLargeAddressSpace()) {
                logger.warn("32bit JVM detected.  It is recommended to run Cassandra on a 64bit JVM for better performance.");
            }
            if ((javaVmName = System.getProperty("java.vm.name")).contains("OpenJDK")) {
                logger.warn("OpenJDK is not recommended. Please upgrade to the newest Oracle Java release");
            } else if (!javaVmName.contains("HotSpot")) {
                logger.warn("Non-Oracle JVM detected.  Some features, such as immediate unmap of compacted SSTables, may not work as intended");
            }
        }
    };
    public static final StartupCheck checkJnaInitialization = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            if (!CLibrary.jnaAvailable()) {
                throw new StartupException(3, "JNA failing to initialize properly. ");
            }
        }
    };
    public static final StartupCheck initSigarLibrary = new StartupCheck(){

        @Override
        public void execute() {
            new SigarLibrary().warnIfRunningInDegradedMode();
        }
    };
    public static final StartupCheck checkDataDirs = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            Iterable dirs = Iterables.concat(Arrays.asList(DatabaseDescriptor.getAllDataFileLocations()), Arrays.asList(DatabaseDescriptor.getCommitLogLocation(), DatabaseDescriptor.getSavedCachesLocation()));
            for (String dataDir : dirs) {
                logger.debug("Checking directory {}", (Object)dataDir);
                File dir = new File(dataDir);
                if (!dir.exists()) {
                    logger.warn("Directory {} doesn't exist", (Object)dataDir);
                    if (!dir.mkdirs()) {
                        throw new StartupException(3, "Has no permission to create directory " + dataDir);
                    }
                }
                if (Directories.verifyFullPermissions(dir, dataDir)) continue;
                throw new StartupException(3, "Insufficient permissions on directory " + dataDir);
            }
        }
    };
    public static final StartupCheck checkSSTablesFormat = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            final HashSet invalid = new HashSet();
            SimpleFileVisitor<Path> sstableVisitor = new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (!file.toString().endsWith(".db")) {
                        return FileVisitResult.CONTINUE;
                    }
                    try {
                        if (!Descriptor.fromFilename(file.toString()).isCompatible()) {
                            invalid.add(file.toString());
                        }
                    }
                    catch (Exception e) {
                        invalid.add(file.toString());
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {
                    String name = dir.getFileName().toString();
                    return name.equals("snapshots") || name.equals("backups") ? FileVisitResult.SKIP_SUBTREE : FileVisitResult.CONTINUE;
                }
            };
            for (String dataDir : DatabaseDescriptor.getAllDataFileLocations()) {
                try {
                    Files.walkFileTree(Paths.get(dataDir, new String[0]), (FileVisitor<? super Path>)sstableVisitor);
                }
                catch (IOException e) {
                    throw new StartupException(3, "Unable to verify sstable files on disk", e);
                }
            }
            if (!invalid.isEmpty()) {
                throw new StartupException(3, String.format("Detected unreadable sstables %s, please check NEWS.txt and ensure that you have upgraded through all required intermediate versions, running upgradesstables", Joiner.on((String)",").join(invalid)));
            }
        }
    };
    public static final StartupCheck checkSystemKeyspaceState = new StartupCheck(){

        @Override
        public void execute() throws StartupException {
            for (CFMetaData cfm : Schema.instance.getKeyspaceMetaData("system").values()) {
                ColumnFamilyStore.scrubDataDirectories(cfm);
            }
            try {
                SystemKeyspace.checkHealth();
            }
            catch (ConfigurationException e) {
                throw new StartupException(100, "Fatal exception during initialization", e);
            }
        }
    };

    public StartupChecks withDefaultTests() {
        this.preFlightChecks.addAll(this.DEFAULT_TESTS);
        return this;
    }

    public StartupChecks withTest(StartupCheck test) {
        this.preFlightChecks.add(test);
        return this;
    }

    public void verify() throws StartupException {
        for (StartupCheck test : this.preFlightChecks) {
            test.execute();
        }
    }
}

