/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import java.io.DataInput;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.utils.FBUtilities;
import org.json.simple.JSONValue;

public class CommitLogDescriptor {
    private static final String SEPARATOR = "-";
    private static final String FILENAME_PREFIX = "CommitLog-";
    private static final String FILENAME_EXTENSION = ".log";
    private static final Pattern COMMIT_LOG_FILE_PATTERN = Pattern.compile("CommitLog-((\\d+)(-\\d+)?).log");
    static final String COMPRESSION_PARAMETERS_KEY = "compressionParameters";
    static final String COMPRESSION_CLASS_KEY = "compressionClass";
    public static final int VERSION_12 = 2;
    public static final int VERSION_20 = 3;
    public static final int VERSION_21 = 4;
    public static final int VERSION_22 = 5;
    public static final int VERSION_30 = 6;
    @VisibleForTesting
    public static final int current_version = 6;
    final int version;
    public final long id;
    public final ParameterizedClass compression;
    private final EncryptionContext encryptionContext;

    public CommitLogDescriptor(int version, long id, ParameterizedClass compression, EncryptionContext encryptionContext) {
        this.version = version;
        this.id = id;
        this.compression = compression;
        this.encryptionContext = encryptionContext;
    }

    public CommitLogDescriptor(long id, ParameterizedClass compression, EncryptionContext encryptionContext) {
        this(6, id, compression, encryptionContext);
    }

    public static void writeHeader(ByteBuffer out, CommitLogDescriptor descriptor) {
        CommitLogDescriptor.writeHeader(out, descriptor, Collections.emptyMap());
    }

    public static void writeHeader(ByteBuffer out, CommitLogDescriptor descriptor, Map<String, String> additionalHeaders) {
        CRC32 crc = new CRC32();
        out.putInt(descriptor.version);
        FBUtilities.updateChecksumInt(crc, descriptor.version);
        out.putLong(descriptor.id);
        FBUtilities.updateChecksumInt(crc, (int)(descriptor.id & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(crc, (int)(descriptor.id >>> 32));
        if (descriptor.version >= 5) {
            String parametersString = CommitLogDescriptor.constructParametersString(descriptor.compression, descriptor.encryptionContext, additionalHeaders);
            byte[] parametersBytes = parametersString.getBytes(StandardCharsets.UTF_8);
            if (parametersBytes.length != ((short)parametersBytes.length & 0xFFFF)) {
                throw new ConfigurationException(String.format("Compression parameters too long, length %d cannot be above 65535.", parametersBytes.length));
            }
            out.putShort((short)parametersBytes.length);
            FBUtilities.updateChecksumInt(crc, parametersBytes.length);
            out.put(parametersBytes);
            crc.update(parametersBytes, 0, parametersBytes.length);
        } else assert (descriptor.compression == null);
        out.putInt((int)crc.getValue());
    }

    @VisibleForTesting
    static String constructParametersString(ParameterizedClass compression, EncryptionContext encryptionContext, Map<String, String> additionalHeaders) {
        TreeMap<String, Object> params = new TreeMap<String, Object>();
        if (compression != null) {
            params.put(COMPRESSION_PARAMETERS_KEY, compression.parameters);
            params.put(COMPRESSION_CLASS_KEY, compression.class_name);
        }
        if (encryptionContext != null) {
            params.putAll(encryptionContext.toHeaderParameters());
        }
        params.putAll(additionalHeaders);
        return JSONValue.toJSONString(params);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static CommitLogDescriptor fromHeader(File file, EncryptionContext encryptionContext) {
        try (RandomAccessFile raf = new RandomAccessFile(file, "r");){
            assert (raf.getFilePointer() == 0L);
            CommitLogDescriptor commitLogDescriptor = CommitLogDescriptor.readHeader(raf, encryptionContext);
            return commitLogDescriptor;
        }
        catch (EOFException e) {
            throw new RuntimeException(e);
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, file);
        }
    }

    public static CommitLogDescriptor readHeader(DataInput input, EncryptionContext encryptionContext) throws IOException {
        CRC32 checkcrc = new CRC32();
        int version = input.readInt();
        FBUtilities.updateChecksumInt(checkcrc, version);
        long id = input.readLong();
        FBUtilities.updateChecksumInt(checkcrc, (int)(id & 0xFFFFFFFFL));
        FBUtilities.updateChecksumInt(checkcrc, (int)(id >>> 32));
        int parametersLength = 0;
        if (version >= 5) {
            parametersLength = input.readShort() & 0xFFFF;
            FBUtilities.updateChecksumInt(checkcrc, parametersLength);
        }
        byte[] parametersBytes = new byte[parametersLength];
        input.readFully(parametersBytes);
        checkcrc.update(parametersBytes, 0, parametersBytes.length);
        int crc = input.readInt();
        if (crc == (int)checkcrc.getValue()) {
            Map map = (Map)JSONValue.parse((String)new String(parametersBytes, StandardCharsets.UTF_8));
            return new CommitLogDescriptor(version, id, CommitLogDescriptor.parseCompression(map), EncryptionContext.createFromMap(map, encryptionContext));
        }
        return null;
    }

    @VisibleForTesting
    static ParameterizedClass parseCompression(Map<?, ?> params) {
        if (params == null || params.isEmpty()) {
            return null;
        }
        String className = (String)params.get(COMPRESSION_CLASS_KEY);
        if (className == null) {
            return null;
        }
        Map cparams = (Map)params.get(COMPRESSION_PARAMETERS_KEY);
        return new ParameterizedClass(className, cparams);
    }

    public static CommitLogDescriptor fromFileName(String name) {
        Matcher matcher = COMMIT_LOG_FILE_PATTERN.matcher(name);
        if (!matcher.matches()) {
            throw new RuntimeException("Cannot parse the version of the file: " + name);
        }
        if (matcher.group(3) == null) {
            throw new UnsupportedOperationException("Commitlog segment is too old to open; upgrade to 1.2.5+ first");
        }
        long id = Long.parseLong(matcher.group(3).split(SEPARATOR)[1]);
        return new CommitLogDescriptor(Integer.parseInt(matcher.group(2)), id, null, new EncryptionContext());
    }

    public int getMessagingVersion() {
        switch (this.version) {
            case 2: {
                return 6;
            }
            case 3: {
                return 7;
            }
            case 4: {
                return 8;
            }
            case 5: {
                return 9;
            }
            case 6: {
                return 10;
            }
        }
        throw new IllegalStateException("Unknown commitlog version " + this.version);
    }

    public String fileName() {
        return FILENAME_PREFIX + this.version + SEPARATOR + this.id + FILENAME_EXTENSION;
    }

    public static boolean isValid(String filename) {
        return COMMIT_LOG_FILE_PATTERN.matcher(filename).matches();
    }

    public EncryptionContext getEncryptionContext() {
        return this.encryptionContext;
    }

    public String toString() {
        return "(" + this.version + "," + this.id + (this.compression != null ? "," + this.compression : "") + ")";
    }

    public boolean equals(Object that) {
        return that instanceof CommitLogDescriptor && this.equals((CommitLogDescriptor)that);
    }

    public boolean equalsIgnoringCompression(CommitLogDescriptor that) {
        return this.version == that.version && this.id == that.id;
    }

    public boolean equals(CommitLogDescriptor that) {
        return this.equalsIgnoringCompression(that) && Objects.equal((Object)this.compression, (Object)that.compression) && Objects.equal((Object)this.encryptionContext, (Object)that.encryptionContext);
    }
}

