// $ANTLR 2.7.7 (20060906): "distinguishedName.g" -> "AntlrDnParser.java"$

/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *  
 *    http://www.apache.org/licenses/LICENSE-2.0
 *  
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License. 
 *  
 */
package org.apache.directory.api.ldap.model.name;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.directory.api.ldap.model.exception.LdapInvalidDnException;
import javax.naming.NameParser;
import org.apache.directory.api.ldap.model.entry.Value;
import org.apache.directory.api.ldap.model.schema.SchemaManager;
import org.apache.directory.api.ldap.model.schema.AttributeType;
import org.apache.directory.api.util.ExpansibleByteBuffer;
import org.apache.directory.api.util.Strings;
import org.apache.directory.api.util.Unicode;

import antlr.TokenBuffer;
import antlr.TokenStreamException;
import antlr.TokenStreamIOException;
import antlr.ANTLRException;
import antlr.LLkParser;
import antlr.Token;
import antlr.TokenStream;
import antlr.RecognitionException;
import antlr.NoViableAltException;
import antlr.MismatchedTokenException;
import antlr.SemanticException;
import antlr.ParserSharedInputState;
import antlr.collections.impl.BitSet;

/**
 * An antlr generated Dn parser.
 *
 * @author <a href="mailto:dev@directory.apache.org">Apache Directory Project</a>
 */
public class AntlrDnParser extends antlr.LLkParser       implements AntlrDnTokenTypes
 {

    private void matchedProduction( String msg )
    {
    }

    /**
     * This class is used to store the decoded value
     */
    private static class UpAndNormValue
    {
        // The value as a byte array
        ExpansibleByteBuffer bytes = new ExpansibleByteBuffer();

        // The user provided value
        StringBuilder upValue = new StringBuilder();

        // The normalized value
        StringBuilder normValue = new StringBuilder();

        // A flag set to false if we have a binary value
        boolean isHR = true;
    }

protected AntlrDnParser(TokenBuffer tokenBuf, int k) {
  super(tokenBuf,k);
  tokenNames = _tokenNames;
}

public AntlrDnParser(TokenBuffer tokenBuf) {
  this(tokenBuf,3);
}

protected AntlrDnParser(TokenStream lexer, int k) {
  super(lexer,k);
  tokenNames = _tokenNames;
}

public AntlrDnParser(TokenStream lexer) {
  this(lexer,3);
}

public AntlrDnParser(ParserSharedInputState state) {
  super(state,3);
  tokenNames = _tokenNames;
}

/**
 * Parses a Dn string.
 *
 * RFC 4514, Section 3
 * <pre>
 * distinguishedName = [ relativeDistinguishedName
 *     *( COMMA relativeDistinguishedName ) ]
 * </pre>
 *
 * RFC 2253, Section 3
 * <pre>
 * distinguishedName = [name] 
 * name       = name-component *("," name-component)
 * </pre>
 *
 * RFC 1779, Section 2.3
 * <pre>
 * &lt;name&gt; ::= &lt;name-component&gt; ( &lt;spaced-separator&gt; )
 *        | &lt;name-component&gt; &lt;spaced-separator&gt; &lt;name&gt;
 * &lt;spaced-separator&gt; ::= &lt;optional-space&gt;
 *             &lt;separator&gt;
 *             &lt;optional-space&gt;
 * &lt;separator&gt; ::=  "," | ";"
 * &lt;optional-space&gt; ::= ( &lt;CR&gt; ) *( " " )
 * </pre>
 *
 * @param schemaManager The SchemaManager
 * @param dn The Dn to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void distinguishedName(
		SchemaManager schemaManager, Dn dn
	) throws RecognitionException, TokenStreamException {
		
		
		matchedProduction( "distinguishedName()" );
		Rdn rdn = new Rdn( schemaManager );
		
		
		{
		switch ( LA(1)) {
		case SPACE:
		case NUMERICOID:
		case ALPHA:
		{
			relativeDistinguishedName(schemaManager, rdn);
			
			try
			{ 
			dn.add( rdn );
			
			}
			catch ( LdapInvalidDnException lide )
			{
			// Do nothing, can't get an exception here
			} 
			
			{
			_loop53:
			do {
				if ((LA(1)==COMMA||LA(1)==SEMI)) {
					{
					switch ( LA(1)) {
					case COMMA:
					{
						match(COMMA);
						break;
					}
					case SEMI:
					{
						match(SEMI);
						break;
					}
					default:
					{
						throw new NoViableAltException(LT(1), getFilename());
					}
					}
					}
					
					rdn = new Rdn( schemaManager );
					
					relativeDistinguishedName(schemaManager, rdn);
					
					try
					{ 
					dn.add( rdn ); 
					}
					catch ( LdapInvalidDnException lide )
					{
					// Do nothing, can't get an exception here
					} 
					
				}
				else {
					break _loop53;
				}
				
			} while (true);
			}
			match(Token.EOF_TYPE);
			break;
		}
		case EOF:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
	}
	
/**
 * Parses a Rdn string.
 *
 * RFC 4514, Section 3
 * <pre>
 * relativeDistinguishedName = attributeTypeAndValue
 *     *( PLUS attributeTypeAndValue )
 * </pre>
 *
 * RFC 2253, Section 3
 * <pre>
 * name-component = attributeTypeAndValue *("+" attributeTypeAndValue)
 * </pre>
 *
 * RFC 1779, Section 2.3
 * <pre>
 * &lt;name-component&gt; ::= &lt;attribute&gt;
 *     | &lt;attribute&gt; &lt;optional-space&gt; "+"
 *       &lt;optional-space&gt; &lt;name-component&gt;
 * </pre>
 *
 * @param schemaManager The SchemaManager
 * @param rdn The Rdn to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void relativeDistinguishedName(
		SchemaManager schemaManager, Rdn rdn
	) throws RecognitionException, TokenStreamException {
		
		
		matchedProduction( "relativeDistinguishedName()" );
		String tmp;
		
		// The rdnStr variable is used to gather the full RDN string
		// as provided
		StringBuilder rdnStr = new StringBuilder();
		StringBuilder rdnNormStr = new StringBuilder();
		int avaPos = 0;
		Ava currentAva;
		AttributeType attributeType;
		Value val;
		
		
		{
		tmp=attributeTypeAndValue(schemaManager, rdn);
		
		rdnStr.append( tmp );
		currentAva = rdn.getAva( avaPos );
		
		attributeType = currentAva.getAttributeType();
		
		if ( schemaManager == null )
		{
		rdnNormStr.append( currentAva.getNormType() );
		rdnNormStr.append( '=' );
		}
		
		val = currentAva.getValue();
		
		if ( ( schemaManager == null ) && ( val != null ) )
		{
		if ( val.getNormalized() != null )
		{
		rdnNormStr.append( val.getNormalized() );
		}
		else
		{
		// We can't tell if the value is HR or not. 
		// Use the Value User Provided value
		rdnNormStr.append( val.getUpValue() );
		}
		}
		
		{
		_loop62:
		do {
			if ((LA(1)==PLUS)) {
				match(PLUS);
				
				rdnStr.append( '+' ); 
				
				if ( schemaManager == null )
				{
				rdnNormStr.append( '+' );
				}
				
				avaPos++;
				
				tmp=attributeTypeAndValue(schemaManager, rdn);
				
				rdnStr.append( tmp );
				currentAva = rdn.getAva( avaPos );
				attributeType = currentAva.getAttributeType();
				
				if ( schemaManager == null )
				{
				rdnNormStr.append( tmp );
				rdnNormStr.append( '=' );
				}
				
				val = currentAva.getValue();
				
				if ( ( schemaManager == null ) && ( val != null ) )
				{
					                if ( val.getNormalized() != null )
					                {
					                    rdnNormStr.append( val.getNormalized() );
					                }
				else
				{
				// We can't tell if the value is HR or not. 
				// Use the Value User Provided value
				rdnNormStr.append( val.getUpValue() );
				}
				}
				
			}
			else {
				break _loop62;
			}
			
		} while (true);
		}
		}
		
		rdn.hashCode();
		rdn.setUpName( rdnStr.toString() );
		
		if ( schemaManager != null )
		{
		// process the multi-value RDN, ordering them by attributes
		boolean isFirst = true;
		
		for ( Ava ava : rdn )
		{
		if ( isFirst )
		{
		isFirst = false;
		}
		else
		{
		rdnNormStr.append( '+' );
		}
		
		rdnNormStr.append( ava.getAttributeType().getOid() );
		rdnNormStr.append( '=' );
		
		val = ava.getValue();
		
		if ( ( val != null ) && ( val.getNormalized() != null ) )
		{
		rdnNormStr.append( val.getNormalized() );
		}
		}
		}
		
		rdn.setNormName( rdnNormStr.toString() );
		
	}
	
/**
 * Parses a Dn string.
 *
 * RFC 4514, Section 3
 * <pre>
 * distinguishedName = [ relativeDistinguishedName
 *     *( COMMA relativeDistinguishedName ) ]
 * </pre>
 *
 * RFC 2253, Section 3
 * <pre>
 * distinguishedName = [name] 
 * name       = name-component *("," name-component)
 * </pre>
 *
 * RFC 1779, Section 2.3
 * <pre>
 * &lt;name&gt; ::= &lt;name-component&gt; ( &lt;spaced-separator&gt; )
 *        | &lt;name-component&gt; &lt;spaced-separator&gt; &lt;name&gt;
 * &lt;spaced-separator&gt; ::= &lt;optional-space&gt;
 *             &lt;separator&gt;
 *             &lt;optional-space&gt;
 * &lt;separator&gt; ::=  "," | ";"
 * &lt;optional-space&gt; ::= ( &lt;CR&gt; ) *( " " )
 * </pre>
 *
 * @param schemaManager The SchemaManager
 * @param rdns The list of Rdns to update
 * @return The normalized Dn
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final String  relativeDistinguishedNames(
		SchemaManager schemaManager, List<Rdn> rdns
	) throws RecognitionException, TokenStreamException {
		String normNameStr;
		
		
		matchedProduction( "relativeDistinguishedNames()" );
		Rdn rdn = new Rdn( schemaManager );
		StringBuilder dnNormSb = new StringBuilder();
		
		
		{
		switch ( LA(1)) {
		case SPACE:
		case NUMERICOID:
		case ALPHA:
		{
			relativeDistinguishedName( schemaManager, rdn);
			
			rdns.add( rdn );
			dnNormSb.append( rdn.getNormName() );
			rdn = new Rdn( schemaManager );
			
			{
			_loop58:
			do {
				if ((LA(1)==COMMA||LA(1)==SEMI)) {
					{
					switch ( LA(1)) {
					case COMMA:
					{
						match(COMMA);
						break;
					}
					case SEMI:
					{
						match(SEMI);
						break;
					}
					default:
					{
						throw new NoViableAltException(LT(1), getFilename());
					}
					}
					}
					relativeDistinguishedName(schemaManager, rdn);
					
					rdns.add( rdn ); 
						            dnNormSb.append( ',' );
						            dnNormSb.append( rdn.getNormName() );
					rdn = new Rdn( schemaManager );
									
				}
				else {
					break _loop58;
				}
				
			} while (true);
			}
			match(Token.EOF_TYPE);
			break;
		}
		case EOF:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
		
		normNameStr = dnNormSb.toString();
		
		return normNameStr;
	}
	
/**
 * RFC 4514, Section 3
 * <pre>
 * attributeTypeAndValue = attributeType EQUALS attributeValue
 * </pre>
 *
 * RFC 2253, Section 3
 * <pre>
 * attributeTypeAndValue = attributeType "=" attributeValue
 * </pre>
 *
 * @param schemaManager The SchemaManager
 * @param rdn The Rdn to update
 * @return The user provided Ava
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final String  attributeTypeAndValue(
		SchemaManager schemaManager, Rdn rdn
	) throws RecognitionException, TokenStreamException {
		String upNameStr;
		
		
		matchedProduction( "attributeTypeAndValue()" );
		String type = null;
		UpAndNormValue value = new UpAndNormValue();
		StringBuilder rdnUpName = new StringBuilder();
		
		
		{
		{
		_loop66:
		do {
			if ((LA(1)==SPACE)) {
				match(SPACE);
				rdnUpName.append( ' ' );
			}
			else {
				break _loop66;
			}
			
		} while (true);
		}
		type=attributeType();
		rdnUpName.append( type );
		{
		_loop68:
		do {
			if ((LA(1)==SPACE)) {
				match(SPACE);
				rdnUpName.append( ' ' );
			}
			else {
				break _loop68;
			}
			
		} while (true);
		}
		match(EQUALS);
		rdnUpName.append( '=' );
		{
		_loop70:
		do {
			if ((LA(1)==SPACE) && (_tokenSet_0.member(LA(2))) && (_tokenSet_1.member(LA(3)))) {
				match(SPACE);
				rdnUpName.append( ' ' );
			}
			else {
				break _loop70;
			}
			
		} while (true);
		}
		attributeValue(value);
		
		try
		{
		// We have to remove the ending spaces that may have been added, as the tutf1 rule
		// cannot be processed
		rdnUpName.append( value.upValue );
		AttributeType attributeType = null;
		Ava ava = null;
		
		if ( schemaManager != null )
		{
		if ( ( type.startsWith( "oid." ) ) || ( type.startsWith( "OID." ) ) )
		{
		type = type.substring( 4 );
		}
		
		attributeType = schemaManager.getAttributeType( type );
		}
		
		if ( ( ( attributeType != null ) && attributeType.isHR() ) || value.isHR )
		{
		int valueLength = value.upValue.length();
		int pos = value.bytes.position();
		
		for ( int i = valueLength - 1; i >= 0; i-- )
		{
		if ( value.upValue.charAt( i ) == ' ' ) 
		{
		if ( i == 0 )
		{
		// The value is empty
		ava = new Ava( schemaManager, type, rdnUpName.toString(), ( String ) null );
		break;
		}
		else if ( value.upValue.charAt( i - 1 ) != '\\' )
		{
		// This is a trailing space, get rid of it
		value.upValue.deleteCharAt( i );
		pos--;
		value.bytes.position( pos );
		}
		else
		{
		// This is an escaped space, get out
		break;
		}
		}
		else
		{
		break;
		}
		}
		
		if ( ava == null )
		{
		ava = new Ava( schemaManager, type, rdnUpName.toString(), Strings.utf8ToString( value.bytes.copyOfUsedBytes() ) );
		}
		}
		else
		{
		ava = new Ava( schemaManager, type, rdnUpName.toString(), value.bytes.copyOfUsedBytes() );
		}
		
		rdn.addAVA( schemaManager, ava );
		}
		catch ( LdapInvalidDnException e )
		{
		throw new SemanticException( e.getMessage() );
		} 
		
		{
		_loop72:
		do {
			if ((LA(1)==SPACE)) {
				match(SPACE);
				rdnUpName.append( ' ' );
			}
			else {
				break _loop72;
			}
			
		} while (true);
		}
		}
		
		upNameStr = rdnUpName.toString();
		
		return upNameStr;
	}
	
/**
 * RFC 4514 Section 3
 *
 * <pre>
 * attributeType = descr / numericoid
 * </pre>
 *
 * @return The AttributeType
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final String  attributeType() throws RecognitionException, TokenStreamException {
		String attributeType;
		
		
		matchedProduction( "attributeType()" );
		
		
		{
		switch ( LA(1)) {
		case ALPHA:
		{
			attributeType=descr();
			break;
		}
		case NUMERICOID:
		{
			attributeType=numericoid();
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
		return attributeType;
	}
	
/**
 * RFC 4514, Section 3
 * <pre>
 * attributeValue = string / hexstring
 * </pre>
 *
 * RFC 2253, Section 3
 * <pre>
 * attributeValue = string
 * string     = *( stringchar / pair )
 *              / "#" hexstring
 *              / QUOTATION *( quotechar / pair ) QUOTATION ; only from v2
 *
 * We still accept both forms, which means we can have a value surrounded by '"'
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void attributeValue(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		
		matchedProduction( "attributeValue()" );
		
		
		{
		switch ( LA(1)) {
		case DQUOTE:
		{
			quotestring(value);
			break;
		}
		case EQUALS:
		case HYPHEN:
		case UNDERSCORE:
		case NUMERICOID:
		case DIGIT:
		case ALPHA:
		case HEXPAIR:
		case ESC:
		case ESCESC:
		case ESCSHARP:
		case UTFMB:
		case CHAR_REST:
		{
			string(value);
			break;
		}
		case HEXVALUE:
		{
			hexstring(value);
			break;
		}
		case EOF:
		case COMMA:
		case PLUS:
		case SEMI:
		case SPACE:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
	}
	
/**
 * RFC 4512 Section 1.4
 *
 * <pre>
 * descr = keystring
 * keystring = leadkeychar *keychar
 * leadkeychar = ALPHA
 * keychar = ALPHA / DIGIT / HYPHEN
 * </pre>
 *
 * We additionally add UNDERSCORE because some servers allow them.
 *
 * @return The description
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final String  descr() throws RecognitionException, TokenStreamException {
		String descr;
		
		Token  leadkeychar = null;
		Token  alpha = null;
		Token  digit = null;
		
		matchedProduction( "descr()" );
		StringBuilder descrSb = new StringBuilder();
		
		
		leadkeychar = LT(1);
		match(ALPHA);
		descrSb.append( leadkeychar.getText() );
		{
		_loop77:
		do {
			switch ( LA(1)) {
			case ALPHA:
			{
				alpha = LT(1);
				match(ALPHA);
				descrSb.append( alpha.getText() );
				break;
			}
			case DIGIT:
			{
				digit = LT(1);
				match(DIGIT);
				descrSb.append( digit.getText() );
				break;
			}
			case HYPHEN:
			{
				match(HYPHEN);
				descrSb.append( '-' );
				break;
			}
			case UNDERSCORE:
			{
				match(UNDERSCORE);
				descrSb.append( '_' );
				break;
			}
			default:
			{
				break _loop77;
			}
			}
		} while (true);
		}
		
		descr = descrSb.toString();
		
		return descr;
	}
	
/**
 * RFC 4512 Section 1.4
 *
 * <pre>
 * numericoid = number 1*( DOT number )
 * number  = DIGIT / ( LDIGIT 1*DIGIT )
 * DIGIT   = %x30 / LDIGIT       ; "0"-"9"
 * LDIGIT  = %x31-39             ; "1"-"9"
 * </pre>
 *
 * @return The numeric OID
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final String  numericoid() throws RecognitionException, TokenStreamException {
		String numericoid = "";
		
		Token  noid = null;
		
		matchedProduction( "numericoid()" );
		
		
		noid = LT(1);
		match(NUMERICOID);
		numericoid = noid.getText();
		return numericoid;
	}
	
/**
 * RFC 2253, Section 3
 * <pre>
 *              / QUOTATION *( quotechar / pair ) QUOTATION ; only from v2
 * quotechar     = &lt;any character except "\" or QUOTATION &gt;
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void quotestring(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  s = null;
		
		matchedProduction( "quotestring()" );
		
		
		{
		match(DQUOTE);
		value.upValue.append( '"' );
		{
		_loop86:
		do {
			switch ( LA(1)) {
			case COMMA:
			case EQUALS:
			case PLUS:
			case HYPHEN:
			case UNDERSCORE:
			case SEMI:
			case LANGLE:
			case RANGLE:
			case SPACE:
			case NUMERICOID_OR_ALPHA_OR_DIGIT:
			case NUMERICOID:
			case DOT:
			case NUMBER:
			case LDIGIT:
			case DIGIT:
			case ALPHA:
			case HEXPAIR_OR_ESCESC_ESCSHARP_OR_ESC:
			case HEX:
			case HEXVALUE_OR_SHARP:
			case HEXVALUE:
			case SHARP:
			case UTFMB:
			case CHAR_REST:
			{
				{
				{
				s = LT(1);
				match(_tokenSet_2);
				}
				
				value.upValue.append( s.getText() );
				value.bytes.append( Strings.getBytesUtf8( s.getText() ) );
				
				}
				break;
			}
			case HEXPAIR:
			case ESC:
			case ESCESC:
			case ESCSHARP:
			{
				pair(value);
				break;
			}
			default:
			{
				break _loop86;
			}
			}
		} while (true);
		}
		match(DQUOTE);
		value.upValue.append( '"' );
		}
	}
	
/**
 * RFC 4514 Section 3
 *
 * <pre>
 * ; The following characters are to be escaped when they appear
 * ; in the value to be encoded: ESC, one of &lt;escaped&gt;, &lt;leading&gt;
 * ; SHARP or SPACE, trailing SPACE, and NULL.
 * string =   [ ( leadchar / pair ) [ *( stringchar / pair ) ( trailchar / pair ) ] ]
 * leadchar = LUTF1 | UTFMB
 * stringchar = SUTF1 / UTFMB
 * trailchar = TUTF1 / UTFMB
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void string(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		
		matchedProduction( "string()" );
		
		
		{
		{
		switch ( LA(1)) {
		case EQUALS:
		case HYPHEN:
		case UNDERSCORE:
		case NUMERICOID:
		case DIGIT:
		case ALPHA:
		case CHAR_REST:
		{
			lutf1(value);
			break;
		}
		case UTFMB:
		{
			utfmb(value);
			break;
		}
		case HEXPAIR:
		case ESC:
		case ESCESC:
		case ESCSHARP:
		{
			pair(value);
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
		{
		_loop92:
		do {
			switch ( LA(1)) {
			case UTFMB:
			{
				utfmb(value);
				break;
			}
			case HEXPAIR:
			case ESC:
			case ESCESC:
			case ESCSHARP:
			{
				pair(value);
				break;
			}
			default:
				if ((_tokenSet_3.member(LA(1))) && (_tokenSet_4.member(LA(2))) && (_tokenSet_5.member(LA(3)))) {
					sutf1(value);
				}
			else {
				break _loop92;
			}
			}
		} while (true);
		}
		}
	}
	
/**
 * RFC 4514 Section 3
 *
 * <pre>
 * hexstring = SHARP 1*hexpair
 *
 * If in &lt;hexstring&gt; form, a BER representation can be obtained from
 * converting each &lt;hexpair&gt; of the &lt;hexstring&gt; to the octet indicated
 * by the &lt;hexpair&gt;.
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void hexstring(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  hexValue = null;
		
		matchedProduction( "hexstring()" );
		
		
		hexValue = LT(1);
		match(HEXVALUE);
		
		String hexStr = hexValue.getText();
		value.upValue.append( '#' ).append( hexStr );
		value.bytes.append( Strings.toByteArray( hexStr ) );
		value.isHR = false; 
		
	}
	
/**
 * RFC 4514, Section 3
 * <pre>
 * pair = ESC ( ESC / special / hexpair )
 * special = escaped / SPACE / SHARP / EQUALS
 * escaped = DQUOTE / PLUS / COMMA / SEMI / LANGLE / RANGLE
 * hexpair = HEX HEX
 *
 * If in &lt;string&gt; form, a LDAP string representation asserted value can
 * be obtained by replacing (left to right, non-recursively) each &lt;pair&gt;
 * appearing in the &lt;string&gt; as follows:
 *   replace &lt;ESC&gt;&lt;ESC&gt; with &lt;ESC&gt;;
 *   replace &lt;ESC&gt;&lt;special&gt; with &lt;special&gt;;
 *   replace &lt;ESC&gt;&lt;hexpair&gt; with the octet indicated by the &lt;hexpair&gt;.
 * </pre>
 * 
 * RFC 2253, Section 3
 * <pre>
 * pair       = "\" ( special / "\" / QUOTATION / hexpair )
 * special    = "," / "=" / "+" / "&lt;" /  "&gt;" / "#" / ";"
 * </pre>
 * 
 * RFC 1779, Section 2.3
 * <pre>
 * &lt;pair&gt; ::= "\" ( &lt;special&gt; | "\" | '"')
 * &lt;special&gt; ::= "," | "=" | &lt;CR&gt; | "+" | "&lt;" |  "&gt;"
 *           | "#" | ";"
 * </pre>
 * 
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void pair(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  hexpair = null;
		
		matchedProduction( "pair()" );
		char specialChar;
		
		
		switch ( LA(1)) {
		case ESCESC:
		{
			{
			match(ESCESC);
			
			value.upValue.append( "\\\\" );
			value.bytes.append( '\\' );
			
			}
			break;
		}
		case ESCSHARP:
		{
			{
			match(ESCSHARP);
			
			value.upValue.append( "\\#" );
			value.bytes.append( '#' );
			
			}
			break;
		}
		case ESC:
		{
			{
			match(ESC);
			specialChar=special();
			
			value.upValue.append( '\\' ).append( specialChar );
			value.bytes.append( specialChar );
			
			}
			break;
		}
		case HEXPAIR:
		{
			{
			hexpair = LT(1);
			match(HEXPAIR);
			
			value.upValue.append( '\\' ).append( hexpair.getText() );
			value.bytes.append( Strings.toByteArray( hexpair.getText() ) );
			
			}
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
	}
	
/**
 * RFC 4514, Section 3:
 * <pre>
 * LUTF1 = %x01-1F / %x21 / %x24-2A / %x2D-3A /
 *    %x3D / %x3F-5B / %x5D-7F
 *
 * The rule CHAR_REST doesn't contain the following charcters,
 * so we must check them additionally
 *   EQUALS (0x3D)
 *   HYPHEN (0x2D)
 *   UNDERSCORE (0x5F)
 *   DIGIT (0x30-0x39)
 *   ALPHA (0x41-0x5A and 0x61-0x7A)
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void lutf1(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  rest = null;
		Token  digit = null;
		Token  alpha = null;
		Token  numericoid = null;
		
		matchedProduction( "lutf1()" );
		
		
		switch ( LA(1)) {
		case CHAR_REST:
		{
			rest = LT(1);
			match(CHAR_REST);
			
			char c = rest.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c );
			
			break;
		}
		case EQUALS:
		{
			match(EQUALS);
			
			value.upValue.append( '=' );
			value.bytes.append( '=' );
			
			break;
		}
		case HYPHEN:
		{
			match(HYPHEN);
			
			value.upValue.append( '-' );
			value.bytes.append( '-' );
			
			break;
		}
		case UNDERSCORE:
		{
			match(UNDERSCORE);
			
			value.upValue.append( '_' );
			value.bytes.append( '_' );
			
			break;
		}
		case DIGIT:
		{
			digit = LT(1);
			match(DIGIT);
			
			char c = digit.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c );
			
			break;
		}
		case ALPHA:
		{
			alpha = LT(1);
			match(ALPHA);
			
			char c = alpha.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c  );
			
			break;
		}
		case NUMERICOID:
		{
			numericoid = LT(1);
			match(NUMERICOID);
			
			String number = numericoid.getText();
			value.upValue.append( number );
			value.bytes.append( Strings.getBytesUtf8( number ) );
			
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
	}
	
/**
 * Process a UTFMB char
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void utfmb(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  s = null;
		
		matchedProduction( "utfmb()" );
		
		
		s = LT(1);
		match(UTFMB);
		
		char c = s.getText().charAt( 0 );
		value.upValue.append( c );
		value.bytes.append( Unicode.charToBytes( c ) );
		
	}
	
/**
 * RFC 4514, Section 3:
 * <pre>
 * SUTF1 = %x01-21 / %x23-2A / %x2D-3A /
 *    %x3D / %x3F-5B / %x5D-7F
 *
 * The rule CHAR_REST doesn't contain the following charcters,
 * so we must check them additionally
 *   EQUALS (0x3D)
 *   HYPHEN (0x2D)
 *   UNDERSCORE (0x5F)
 *   DIGIT (0x30-0x39)
 *   ALPHA (0x41-0x5A and 0x61-0x7A)
 *   SHARP (0x23)
 *   SPACE (0x20)
 * </pre>
 *
 * @param value The value to update
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final void sutf1(
		UpAndNormValue value
	) throws RecognitionException, TokenStreamException {
		
		Token  rest = null;
		Token  digit = null;
		Token  alpha = null;
		Token  hex = null;
		Token  numericoid = null;
		
		matchedProduction( "sutf1()" );
		
		
		switch ( LA(1)) {
		case CHAR_REST:
		{
			rest = LT(1);
			match(CHAR_REST);
			
			char c = rest.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c );
			
			break;
		}
		case EQUALS:
		{
			match(EQUALS);
			
			value.upValue.append( '=' );
			value.bytes.append( '=' );
			
			break;
		}
		case HYPHEN:
		{
			match(HYPHEN);
			
			value.upValue.append( '-' );
			value.bytes.append( '-' );
			
			break;
		}
		case UNDERSCORE:
		{
			match(UNDERSCORE);
			
			value.upValue.append( '_' );
			value.bytes.append( '_' );
			
			break;
		}
		case DIGIT:
		{
			digit = LT(1);
			match(DIGIT);
			
			char c = digit.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c );
			
			break;
		}
		case ALPHA:
		{
			alpha = LT(1);
			match(ALPHA);
			
			char c = alpha.getText().charAt( 0 );
			value.upValue.append( c );
			value.bytes.append( ( byte ) c );
			
			break;
		}
		case SHARP:
		{
			match(SHARP);
			
			value.upValue.append( '#' );
			value.bytes.append( '#' );
			
			break;
		}
		case SPACE:
		{
			match(SPACE);
			
			value.upValue.append( ' ' );
			value.bytes.append( ' ' );
			
			break;
		}
		case HEXVALUE:
		{
			hex = LT(1);
			match(HEXVALUE);
			
			String hexStr = hex.getText();
			value.upValue.append( '#' ).append( hexStr );
			value.bytes.append( '#' );
			value.bytes.append( Strings.getBytesUtf8( hexStr ) );
			
			break;
		}
		case NUMERICOID:
		{
			numericoid = LT(1);
			match(NUMERICOID);
			
			String number = numericoid.getText();
			value.upValue.append( number );
			value.bytes.append( Strings.getBytesUtf8( number ) );
			
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
	}
	
/**
 * RFC 4514 Section 3
 * 
 * <pre>
 * special = escaped / SPACE / SHARP / EQUALS
 * escaped = DQUOTE / PLUS / COMMA / SEMI / LANGLE / RANGLE
 * </pre>
 *
 * @return The special char
 * @throws RecognitionException If the token is invalid
 * @throws TokenStreamException When we weren't able to fetch a token
 */
	public final char  special() throws RecognitionException, TokenStreamException {
		char special;
		
		
		matchedProduction( "()" );
		
		
		{
		switch ( LA(1)) {
		case DQUOTE:
		{
			match(DQUOTE);
			special = '"';
			break;
		}
		case PLUS:
		{
			match(PLUS);
			special = '+';
			break;
		}
		case COMMA:
		{
			match(COMMA);
			special = ',';
			break;
		}
		case SEMI:
		{
			match(SEMI);
			special = ';';
			break;
		}
		case LANGLE:
		{
			match(LANGLE);
			special = '<';
			break;
		}
		case RANGLE:
		{
			match(RANGLE);
			special = '>';
			break;
		}
		case SPACE:
		{
			match(SPACE);
			special = ' ';
			break;
		}
		case SHARP:
		{
			match(SHARP);
			special = '#';
			break;
		}
		case EQUALS:
		{
			match(EQUALS);
			special = '=';
			break;
		}
		default:
		{
			throw new NoViableAltException(LT(1), getFilename());
		}
		}
		}
		return special;
	}
	
	
	public static final String[] _tokenNames = {
		"<0>",
		"EOF",
		"<2>",
		"NULL_TREE_LOOKAHEAD",
		"COMMA",
		"EQUALS",
		"PLUS",
		"HYPHEN",
		"UNDERSCORE",
		"DQUOTE",
		"SEMI",
		"LANGLE",
		"RANGLE",
		"SPACE",
		"NUMERICOID_OR_ALPHA_OR_DIGIT",
		"NUMERICOID",
		"DOT",
		"NUMBER",
		"LDIGIT",
		"DIGIT",
		"ALPHA",
		"HEXPAIR_OR_ESCESC_ESCSHARP_OR_ESC",
		"HEXPAIR",
		"ESC",
		"ESCESC",
		"ESCSHARP",
		"HEX",
		"HEXVALUE_OR_SHARP",
		"HEXVALUE",
		"SHARP",
		"UTFMB",
		"CHAR_REST"
	};
	
	private static final long[] mk_tokenSet_0() {
		long[] data = { 3554191346L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_0 = new BitSet(mk_tokenSet_0());
	private static final long[] mk_tokenSet_1() {
		long[] data = { 4294967282L, 0L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_1 = new BitSet(mk_tokenSet_1());
	private static final long[] mk_tokenSet_2() {
		long[] data = { 4232052208L, 0L, 0L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_2 = new BitSet(mk_tokenSet_2());
	private static final long[] mk_tokenSet_3() {
		long[] data = { 2954404256L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_3 = new BitSet(mk_tokenSet_3());
	private static final long[] mk_tokenSet_4() {
		long[] data = { 4091061746L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_4 = new BitSet(mk_tokenSet_4());
	private static final long[] mk_tokenSet_5() {
		long[] data = { 4091068402L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_5 = new BitSet(mk_tokenSet_5());
	
	}
