/*
 * Decompiled with CFR 0.152.
 */
package org.apache.directory.api.ldap.model.password;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Base64;
import java.util.Date;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import org.apache.commons.codec.digest.Crypt;
import org.apache.directory.api.i18n.I18n;
import org.apache.directory.api.ldap.model.constants.LdapSecurityConstants;
import org.apache.directory.api.ldap.model.password.BCrypt;
import org.apache.directory.api.ldap.model.password.PasswordDetails;
import org.apache.directory.api.util.DateUtils;
import org.apache.directory.api.util.Strings;
import org.apache.directory.api.util.TimeProvider;

public final class PasswordUtil {
    public static final int SHA1_LENGTH = 20;
    public static final int SHA256_LENGTH = 32;
    public static final int SHA384_LENGTH = 48;
    public static final int SHA512_LENGTH = 64;
    public static final int MD5_LENGTH = 16;
    public static final int PKCS5S2_LENGTH = 32;
    public static final int CRYPT_LENGTH = 11;
    public static final int CRYPT_MD5_LENGTH = 22;
    public static final int CRYPT_SHA256_LENGTH = 43;
    public static final int CRYPT_SHA512_LENGTH = 86;
    public static final int CRYPT_BCRYPT_LENGTH = 31;
    private static final byte[] CRYPT_SALT_CHARS = Strings.getBytesUtf8("./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz");

    private PasswordUtil() {
    }

    public static LdapSecurityConstants findAlgorithm(byte[] credentials) {
        if (credentials == null || credentials.length == 0) {
            return null;
        }
        if (credentials[0] == 123) {
            int pos;
            for (pos = 1; pos < credentials.length && credentials[pos] != 125; ++pos) {
            }
            if (pos < credentials.length) {
                if (pos == 1) {
                    return null;
                }
                String algorithm = Strings.toLowerCaseAscii(Strings.utf8ToString(credentials, 1, pos - 1));
                if (credentials.length > pos + 3 && credentials[pos + 1] == 36 && Character.isDigit(credentials[pos + 2])) {
                    if (credentials[pos + 3] == 36) {
                        algorithm = algorithm + Strings.utf8ToString(credentials, pos + 1, 3);
                    } else if (credentials.length > pos + 4 && credentials[pos + 4] == 36) {
                        algorithm = algorithm + Strings.utf8ToString(credentials, pos + 1, 4);
                    }
                }
                return LdapSecurityConstants.getAlgorithm(algorithm);
            }
            return null;
        }
        return null;
    }

    public static byte[] createStoragePassword(String credentials, LdapSecurityConstants algorithm) {
        return PasswordUtil.createStoragePassword(Strings.getBytesUtf8(credentials), algorithm);
    }

    public static byte[] createStoragePassword(byte[] credentials, LdapSecurityConstants algorithm) {
        byte[] salt;
        if (algorithm == null) {
            return credentials;
        }
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SMD5: {
                salt = new byte[8];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_PKCS5S2: {
                salt = new byte[16];
                new SecureRandom().nextBytes(salt);
                break;
            }
            case HASH_METHOD_CRYPT: {
                salt = PasswordUtil.generateCryptSalt(2);
                break;
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                salt = PasswordUtil.generateCryptSalt(8);
                break;
            }
            case HASH_METHOD_CRYPT_BCRYPT: {
                salt = Strings.getBytesUtf8(BCrypt.genSalt());
                break;
            }
            default: {
                salt = null;
            }
        }
        byte[] hashedPassword = PasswordUtil.encryptPassword(credentials, algorithm, salt);
        StringBuilder sb = new StringBuilder();
        sb.append('{').append(Strings.upperCase(algorithm.getPrefix())).append('}');
        if (algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT || algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT_BCRYPT) {
            sb.append(Strings.utf8ToString(salt));
            sb.append(Strings.utf8ToString(hashedPassword));
        } else if (algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT_MD5 || algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT_SHA256 || algorithm == LdapSecurityConstants.HASH_METHOD_CRYPT_SHA512) {
            sb.append(algorithm.getSubPrefix());
            sb.append(Strings.utf8ToString(salt));
            sb.append('$');
            sb.append(Strings.utf8ToString(hashedPassword));
        } else if (salt != null) {
            byte[] hashedPasswordWithSaltBytes = new byte[hashedPassword.length + salt.length];
            if (algorithm == LdapSecurityConstants.HASH_METHOD_PKCS5S2) {
                PasswordUtil.merge(hashedPasswordWithSaltBytes, salt, hashedPassword);
            } else {
                PasswordUtil.merge(hashedPasswordWithSaltBytes, hashedPassword, salt);
            }
            sb.append(String.valueOf(Base64.getEncoder().encodeToString(hashedPasswordWithSaltBytes)));
        } else {
            sb.append(String.valueOf(Base64.getEncoder().encodeToString(hashedPassword)));
        }
        return Strings.getBytesUtf8(sb.toString());
    }

    public static boolean compareCredentials(byte[] receivedCredentials, byte[] storedCredentials) {
        LdapSecurityConstants algorithm = PasswordUtil.findAlgorithm(storedCredentials);
        if (algorithm != null) {
            PasswordDetails passwordDetails = PasswordUtil.splitCredentials(storedCredentials);
            byte[] userPassword = PasswordUtil.encryptPassword(receivedCredentials, passwordDetails.getAlgorithm(), passwordDetails.getSalt());
            return PasswordUtil.compareBytes(userPassword, passwordDetails.getPassword());
        }
        return PasswordUtil.compareBytes(receivedCredentials, storedCredentials);
    }

    private static boolean compareBytes(byte[] provided, byte[] stored) {
        if (stored == null) {
            return provided == null;
        }
        if (provided == null) {
            return false;
        }
        if (stored.length != provided.length) {
            return false;
        }
        int result = 0;
        for (int i = 0; i < stored.length; ++i) {
            result |= stored[i] ^ provided[i];
        }
        return result == 0;
    }

    private static byte[] encryptPassword(byte[] credentials, LdapSecurityConstants algorithm, byte[] salt) {
        switch (algorithm) {
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA, credentials, salt);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA256, credentials, salt);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA384, credentials, salt);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_SHA512, credentials, salt);
            }
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordUtil.digest(LdapSecurityConstants.HASH_METHOD_MD5, credentials, salt);
            }
            case HASH_METHOD_CRYPT: {
                String saltWithCrypted = Crypt.crypt((String)Strings.utf8ToString(credentials), (String)Strings.utf8ToString(salt));
                String crypted = saltWithCrypted.substring(2);
                return Strings.getBytesUtf8(crypted);
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                String saltWithCrypted2 = Crypt.crypt((String)Strings.utf8ToString(credentials), (String)(algorithm.getSubPrefix() + Strings.utf8ToString(salt)));
                String crypted2 = saltWithCrypted2.substring(saltWithCrypted2.lastIndexOf(36) + 1);
                return Strings.getBytesUtf8(crypted2);
            }
            case HASH_METHOD_CRYPT_BCRYPT: {
                String crypted3 = BCrypt.hashPw(Strings.utf8ToString(credentials), Strings.utf8ToString(salt));
                return Strings.getBytesUtf8(crypted3.substring(crypted3.length() - 31));
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordUtil.generatePbkdf2Hash(credentials, algorithm, salt);
            }
        }
        return credentials;
    }

    private static byte[] digest(LdapSecurityConstants algorithm, byte[] password, byte[] salt) {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance(algorithm.getAlgorithm());
        }
        catch (NoSuchAlgorithmException e1) {
            return null;
        }
        if (salt != null) {
            digest.update(password);
            digest.update(salt);
            return digest.digest();
        }
        return digest.digest(password);
    }

    public static PasswordDetails splitCredentials(byte[] credentials) {
        LdapSecurityConstants algorithm = PasswordUtil.findAlgorithm(credentials);
        if (algorithm == null) {
            return new PasswordDetails(null, null, credentials);
        }
        int algoLength = algorithm.getPrefix().length() + 2;
        switch (algorithm) {
            case HASH_METHOD_SMD5: 
            case HASH_METHOD_MD5: {
                return PasswordUtil.getCredentials(credentials, algoLength, 16, algorithm);
            }
            case HASH_METHOD_SSHA: 
            case HASH_METHOD_SHA: {
                return PasswordUtil.getCredentials(credentials, algoLength, 20, algorithm);
            }
            case HASH_METHOD_SSHA256: 
            case HASH_METHOD_SHA256: {
                return PasswordUtil.getCredentials(credentials, algoLength, 32, algorithm);
            }
            case HASH_METHOD_SSHA384: 
            case HASH_METHOD_SHA384: {
                return PasswordUtil.getCredentials(credentials, algoLength, 48, algorithm);
            }
            case HASH_METHOD_SSHA512: 
            case HASH_METHOD_SHA512: {
                return PasswordUtil.getCredentials(credentials, algoLength, 64, algorithm);
            }
            case HASH_METHOD_PKCS5S2: {
                return PasswordUtil.getPbkdf2Credentials(credentials, algoLength, algorithm);
            }
            case HASH_METHOD_CRYPT: {
                byte[] salt = new byte[2];
                byte[] password = new byte[credentials.length - salt.length - algoLength];
                PasswordUtil.split(credentials, algoLength, salt, password);
                return new PasswordDetails(algorithm, salt, password);
            }
            case HASH_METHOD_CRYPT_BCRYPT: {
                byte[] salt = Arrays.copyOfRange(credentials, algoLength, credentials.length - 31);
                byte[] password = Arrays.copyOfRange(credentials, credentials.length - 31, credentials.length);
                return new PasswordDetails(algorithm, salt, password);
            }
            case HASH_METHOD_CRYPT_MD5: 
            case HASH_METHOD_CRYPT_SHA256: 
            case HASH_METHOD_CRYPT_SHA512: {
                return PasswordUtil.getCryptCredentials(credentials, algoLength += 3, algorithm);
            }
        }
        throw new IllegalArgumentException(I18n.err(I18n.ERR_13010_UNKNOWN_HASH_ALGO, new Object[]{algorithm}));
    }

    private static PasswordDetails getCredentials(byte[] credentials, int algoLength, int hashLen, LdapSecurityConstants algorithm) {
        byte[] passwordAndSalt = Base64.getDecoder().decode(Strings.utf8ToString(credentials, algoLength, credentials.length - algoLength));
        int saltLength = passwordAndSalt.length - hashLen;
        byte[] salt = saltLength == 0 ? null : new byte[saltLength];
        byte[] password = new byte[hashLen];
        PasswordUtil.split(passwordAndSalt, 0, password, salt);
        return new PasswordDetails(algorithm, salt, password);
    }

    private static void split(byte[] all, int offset, byte[] left, byte[] right) {
        System.arraycopy(all, offset, left, 0, left.length);
        if (right != null) {
            System.arraycopy(all, offset + left.length, right, 0, right.length);
        }
    }

    private static void merge(byte[] all, byte[] left, byte[] right) {
        System.arraycopy(left, 0, all, 0, left.length);
        System.arraycopy(right, 0, all, left.length, right.length);
    }

    public static boolean isPwdExpired(String pwdChangedZtime, int pwdMaxAgeSec, TimeProvider timeProvider) {
        Date pwdChangeDate = DateUtils.getDate(pwdChangedZtime);
        long time = (long)pwdMaxAgeSec * 1000L;
        Date expiryDate = DateUtils.getDate(DateUtils.getGeneralizedTime(time += pwdChangeDate.getTime()));
        Date now = DateUtils.getDate(DateUtils.getGeneralizedTime(timeProvider));
        boolean expired = false;
        if (expiryDate.equals(now) || expiryDate.before(now)) {
            expired = true;
        }
        return expired;
    }

    private static byte[] generatePbkdf2Hash(byte[] credentials, LdapSecurityConstants algorithm, byte[] salt) {
        try {
            SecretKeyFactory sk = SecretKeyFactory.getInstance(algorithm.getAlgorithm());
            char[] password = Strings.utf8ToString(credentials).toCharArray();
            PBEKeySpec keySpec = new PBEKeySpec(password, salt, 10000, 256);
            SecretKey key = sk.generateSecret(keySpec);
            return key.getEncoded();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static PasswordDetails getPbkdf2Credentials(byte[] credentials, int algoLength, LdapSecurityConstants algorithm) {
        byte[] passwordAndSalt = Base64.getDecoder().decode(Strings.utf8ToString(credentials, algoLength, credentials.length - algoLength));
        int saltLength = passwordAndSalt.length - 32;
        byte[] salt = new byte[saltLength];
        byte[] password = new byte[32];
        PasswordUtil.split(passwordAndSalt, 0, salt, password);
        return new PasswordDetails(algorithm, salt, password);
    }

    private static byte[] generateCryptSalt(int length) {
        byte[] salt = new byte[length];
        SecureRandom sr = new SecureRandom();
        for (int i = 0; i < salt.length; ++i) {
            salt[i] = CRYPT_SALT_CHARS[sr.nextInt(CRYPT_SALT_CHARS.length)];
        }
        return salt;
    }

    private static PasswordDetails getCryptCredentials(byte[] credentials, int algoLength, LdapSecurityConstants algorithm) {
        int pos;
        for (pos = algoLength; pos < credentials.length && credentials[pos] != 36; ++pos) {
        }
        byte[] salt = Arrays.copyOfRange(credentials, algoLength, pos);
        byte[] password = Arrays.copyOfRange(credentials, pos + 1, credentials.length);
        return new PasswordDetails(algorithm, salt, password);
    }
}

