/**
 * JDBM LICENSE v1.00
 *
 * Redistribution and use of this software and associated documentation
 * ("Software"), with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * 1. Redistributions of source code must retain copyright
 *    statements and notices.  Redistributions must also contain a
 *    copy of this document.
 *
 * 2. Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *
 * 3. The name "JDBM" must not be used to endorse or promote
 *    products derived from this Software without prior written
 *    permission of Cees de Groot.  For written permission,
 *    please contact cg@cdegroot.com.
 *
 * 4. Products derived from this Software may not be called "JDBM"
 *    nor may "JDBM" appear in their names without prior written
 *    permission of Cees de Groot.
 *
 * 5. Due credit should be given to the JDBM Project
 *    (http://jdbm.sourceforge.net/).
 *
 * THIS SOFTWARE IS PROVIDED BY THE JDBM PROJECT AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CEES DE GROOT OR ANY CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright 2000 (C) Cees de Groot. All Rights Reserved.
 * Contributions are Copyright (C) 2000 by their associated contributors.
 *
 * $Id: DataPage.java,v 1.1 2000/05/06 00:00:31 boisvert Exp $
 */
package jdbm.recman;


import jdbm.I18n;


/**
 * Class describing a page that holds data.
 */
final class DataPage extends PageHeader
{
    // offsets
    /** first short in the file after the page header info: 18 byte offset */
    private static final short O_FIRST = PageHeader.SIZE; // short firstrowid

    /** start of the data in this block: 20 byte offset */
    static final short O_DATA = ( short ) ( O_FIRST + Magic.SZ_SHORT );

    /** total amount of data in this page/block: BLOCK_SIZE - 20 bytes */
    static final short DATA_PER_PAGE = ( short ) ( RecordFile.BLOCK_SIZE - O_DATA );


    /**
     * Constructs a data page view from the indicated block.
     */
    DataPage( BlockIo block )
    {
        super( block );
    }


    /**
     * Factory method to create or return a data page for the indicated block.
     */
    static DataPage getDataPageView( BlockIo blockIo )
    {
        BlockView view = blockIo.getView();

        if ( ( view != null ) && ( view instanceof DataPage ) )
        {
            return ( DataPage ) view;
        }
        else
        {
            return new DataPage( blockIo );
        }
    }


    /**
     * @return the first rowid's offset
     */
    short getFirst()
    {
        return blockIo.readShort( O_FIRST );
    }


    /**
     * Sets the first rowid's offset
     */
    void setFirst( short value )
    {
        paranoiaMagicOk();

        if ( value > 0 && value < O_DATA )
        {
            throw new Error( I18n.err( I18n.ERR_543, value ) );
        }

        blockIo.writeShort( O_FIRST, value );
    }


    /**
     * {@inheritDoc}
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        sb.append( "DataPage ( " );

        // The blockIO
        sb.append( super.toString() ).append( ", " );

        // The first rowId
        sb.append( "first rowId: " ).append( getFirst() ).append( ", " );

        // The data per page
        sb.append( "[p:" ).append( getPrev() ).append( ", " );

        // The next page
        sb.append( "n:" ).append( getNext() ).append( "] )" );

        return sb.toString();
    }
}
