/**
 * JDBM LICENSE v1.00
 *
 * Redistribution and use of this software and associated documentation
 * ("Software"), with or without modification, are permitted provided
 * that the following conditions are met:
 *
 * 1. Redistributions of source code must retain copyright
 *    statements and notices.  Redistributions must also contain a
 *    copy of this document.
 *
 * 2. Redistributions in binary form must reproduce the
 *    above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *
 * 3. The name "JDBM" must not be used to endorse or promote
 *    products derived from this Software without prior written
 *    permission of Cees de Groot.  For written permission,
 *    please contact cg@cdegroot.com.
 *
 * 4. Products derived from this Software may not be called "JDBM"
 *    nor may "JDBM" appear in their names without prior written
 *    permission of Cees de Groot. 
 *
 * 5. Due credit should be given to the JDBM Project
 *    (http://jdbm.sourceforge.net/).
 *
 * THIS SOFTWARE IS PROVIDED BY THE JDBM PROJECT AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * CEES DE GROOT OR ANY CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright 2000 (C) Cees de Groot. All Rights Reserved.
 * Contributions are Copyright (C) 2000 by their associated contributors.
 *
 * $Id: TranslationPage.java,v 1.1 2000/05/06 00:00:31 boisvert Exp $
 */
package jdbm.recman;


/**
 * Class describing a page that holds translations from physical rowids
 * to logical rowids. In fact, the page just holds physical rowids - the
 * page's block is the block for the logical rowid, the offset serve
 * as offset for the rowids.
 */
final class TranslationPage extends PageHeader
{
    /** Offset of the PageHeader */
    static final short O_TRANS = PageHeader.SIZE; // short count

    /** Number of PhysicalRowId in this page */
    static final short ELEMS_PER_PAGE = ( RecordFile.BLOCK_SIZE - O_TRANS ) / PhysicalRowId.SIZE;

    /** The table of PhysicalRowId */
    final PhysicalRowId[] slots = new PhysicalRowId[ELEMS_PER_PAGE];


    /**
     * Constructs a data page view from the indicated block.
     */
    TranslationPage( BlockIo blockIo )
    {
        super( blockIo );
    }


    /**
     * Factory method to create or return a data page for the indicated block.
     */
    static TranslationPage getTranslationPageView( BlockIo blockIo )
    {
        BlockView view = blockIo.getView();

        if ( ( view != null ) && view instanceof TranslationPage )
        {
            return ( TranslationPage ) view;
        }
        else
        {
            return new TranslationPage( blockIo );
        }
    }


    /** Returns the value of the indicated rowid on the page */
    PhysicalRowId get( short offset )
    {
        int slot = ( offset - O_TRANS ) / PhysicalRowId.SIZE;

        if ( slots[slot] == null )
        {
            slots[slot] = new PhysicalRowId( blockIo, offset );
        }

        return slots[slot];
    }


    /**
     * {@inheritDoc}
     */
    public String toString()
    {
        StringBuilder sb = new StringBuilder();

        sb.append( "TranslationPage ( " );

        // The blockIO
        sb.append( super.toString() ).append( ", " );

        // Dump the Physical row id
        for ( int i = 0; i < ELEMS_PER_PAGE; i++ )
        {
            if ( slots[i] != null )
            {
                sb.append( ", [" ).append( i ).append( "]=<" ).
                    append( slots[i].getBlock() ).append( ", " ).
                    append( slots[i].getOffset() ).append( ">" );
            }
        }

        sb.append( ")" );
        return sb.toString();
    }
}
