/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.execution.librarycache;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLClassLoader;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.FSDataOutputStream;
import org.apache.flink.core.fs.FileStatus;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.io.StringRecord;
import org.apache.flink.runtime.jobgraph.JobID;
import org.apache.flink.util.StringUtils;

public final class LibraryCacheManager {
    private static LibraryCacheManager libraryManager = null;
    private final ConcurrentMap<LibraryTranslationKey, String> clientPathToCacheName = new ConcurrentHashMap<LibraryTranslationKey, String>();
    private static final String LIBRARYCACHENAME = "libraryCache";
    private static final String HASHING_ALGORITHM = "SHA-1";
    private static final Object LOCK_OBJECT = new Object();
    private final FileSystem fs;
    private final MessageDigest md;
    private final Path libraryCachePath;
    private final ConcurrentMap<JobID, LibraryManagerEntry> libraryManagerEntries = new ConcurrentHashMap<JobID, LibraryManagerEntry>();
    private final ConcurrentMap<JobID, AtomicInteger> libraryReferenceCounter = new ConcurrentHashMap<JobID, AtomicInteger>();
    private final ConcurrentMap<JobID, Object> lockMap = new ConcurrentHashMap<JobID, Object>();

    private static synchronized LibraryCacheManager get() throws IOException {
        if (libraryManager == null) {
            libraryManager = new LibraryCacheManager();
        }
        return libraryManager;
    }

    private LibraryCacheManager() throws IOException {
        String tmp = System.getProperty("java.io.tmpdir");
        if (tmp == null) {
            throw new IOException("Cannot find directory for temporary files");
        }
        this.fs = FileSystem.getLocalFileSystem();
        String userName = System.getProperty("user.name");
        if (userName == null) {
            userName = "default";
        }
        File path = new File(tmp + File.separator + LIBRARYCACHENAME + "-" + userName);
        URI uri = path.toURI();
        this.libraryCachePath = new Path(uri);
        this.fs.mkdirs(this.libraryCachePath);
        try {
            this.md = MessageDigest.getInstance(HASHING_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IOException("Cannot find algorithm SHA-1: " + StringUtils.stringifyException((Throwable)e));
        }
    }

    private int incrementReferenceCounter(JobID jobID) {
        AtomicInteger ai;
        while ((ai = (AtomicInteger)this.libraryReferenceCounter.get(jobID)) == null) {
            ai = new AtomicInteger(1);
            if (this.libraryReferenceCounter.putIfAbsent(jobID, ai) != null) continue;
            return 1;
        }
        return ai.incrementAndGet();
    }

    private int decrementReferenceCounter(JobID jobID) {
        AtomicInteger ai = (AtomicInteger)this.libraryReferenceCounter.get(jobID);
        if (ai == null) {
            throw new IllegalStateException("Cannot find reference counter entry for job " + jobID);
        }
        int retVal = ai.decrementAndGet();
        if (retVal == 0) {
            this.libraryReferenceCounter.remove(jobID);
        }
        return retVal;
    }

    public static void register(JobID id, Path[] clientPaths) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.registerInternal(id, clientPaths);
    }

    private void registerInternal(JobID id, Path[] clientPaths) throws IOException {
        String[] cacheNames = new String[clientPaths.length];
        for (int i = 0; i < clientPaths.length; ++i) {
            LibraryTranslationKey key = new LibraryTranslationKey(id, clientPaths[i]);
            cacheNames[i] = (String)this.clientPathToCacheName.get(key);
            if (cacheNames[i] != null) continue;
            throw new IOException("Cannot map" + clientPaths[i].toString() + " to cache name");
        }
        this.registerInternal(id, cacheNames);
    }

    public static void register(JobID id, String[] requiredJarFiles) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.registerInternal(id, requiredJarFiles);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerInternal(JobID id, String[] requiredJarFiles) throws IOException {
        while (this.lockMap.putIfAbsent(id, LOCK_OBJECT) != null) {
        }
        try {
            if (this.incrementReferenceCounter(id) > 1) {
                return;
            }
            if (this.libraryManagerEntries.containsKey(id)) {
                throw new IllegalStateException("Library cache manager already contains entry for job ID " + id);
            }
            URL[] urls = null;
            if (requiredJarFiles != null) {
                urls = new URL[requiredJarFiles.length];
                for (int i = 0; i < requiredJarFiles.length; ++i) {
                    Path p = LibraryCacheManager.contains(requiredJarFiles[i]);
                    if (p == null) {
                        throw new IOException(requiredJarFiles[i] + " does not exist in the library cache");
                    }
                    try {
                        urls[i] = p.toUri().toURL();
                        continue;
                    }
                    catch (MalformedURLException e) {
                        throw new IOException(StringUtils.stringifyException((Throwable)e));
                    }
                }
            }
            LibraryManagerEntry entry = new LibraryManagerEntry(id, requiredJarFiles, urls);
            this.libraryManagerEntries.put(id, entry);
        }
        finally {
            this.lockMap.remove(id);
        }
    }

    public static void unregister(JobID id) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.unregisterInternal(id);
    }

    private void unregisterInternal(JobID id) {
        while (this.lockMap.putIfAbsent(id, LOCK_OBJECT) != null) {
        }
        if (this.decrementReferenceCounter(id) == 0) {
            this.libraryManagerEntries.remove(id);
        }
        this.lockMap.remove(id);
    }

    public static Path contains(String cacheName) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        return lib.containsInternal(cacheName);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Path containsInternal(String cacheName) throws IOException {
        Path p = new Path(this.libraryCachePath + File.separator + cacheName);
        FileSystem fileSystem = this.fs;
        synchronized (fileSystem) {
            if (this.fs.exists(p)) {
                return p;
            }
        }
        return null;
    }

    public static ClassLoader getClassLoader(JobID id) throws IOException {
        if (id == null) {
            return null;
        }
        LibraryCacheManager lib = LibraryCacheManager.get();
        return lib.getClassLoaderInternal(id);
    }

    private ClassLoader getClassLoaderInternal(JobID id) {
        LibraryManagerEntry entry = (LibraryManagerEntry)this.libraryManagerEntries.get(id);
        if (entry == null) {
            return null;
        }
        return entry.getClassLoader();
    }

    public static String[] getRequiredJarFiles(JobID id) throws IOException {
        if (id == null) {
            return new String[0];
        }
        LibraryCacheManager lib = LibraryCacheManager.get();
        return lib.getRequiredJarFilesInternal(id);
    }

    private String[] getRequiredJarFilesInternal(JobID id) {
        LibraryManagerEntry entry = (LibraryManagerEntry)this.libraryManagerEntries.get(id);
        if (entry == null) {
            return null;
        }
        return entry.getRequiredJarFiles();
    }

    public static void writeLibraryToStream(String libraryFileName, DataOutput out) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.writeLibraryToStreamInternal(libraryFileName, out);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void writeLibraryToStreamInternal(String libraryFileName, DataOutput out) throws IOException {
        if (libraryFileName == null) {
            throw new IOException("libraryName is null!");
        }
        Path storePath = new Path(this.libraryCachePath + "/" + libraryFileName);
        FileSystem fileSystem = this.fs;
        synchronized (fileSystem) {
            if (!this.fs.exists(storePath)) {
                throw new IOException(storePath + " does not exist!");
            }
            FileStatus status = this.fs.getFileStatus(storePath);
            StringRecord.writeString((DataOutput)out, (String)libraryFileName);
            out.writeLong(status.getLen());
            FSDataInputStream inStream = this.fs.open(storePath);
            byte[] buf = new byte[8192];
            int read = inStream.read(buf, 0, buf.length);
            while (read > 0) {
                out.write(buf, 0, read);
                read = inStream.read(buf, 0, buf.length);
            }
            inStream.close();
        }
    }

    public static void readLibraryFromStream(DataInput in) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.readLibraryFromStreamInternal(in);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void readLibraryFromStreamInternal(DataInput in) throws IOException {
        String libraryFileName = StringRecord.readString((DataInput)in);
        if (libraryFileName == null) {
            throw new IOException("libraryFileName is null!");
        }
        long length = in.readLong();
        if (length > Integer.MAX_VALUE) {
            throw new IOException("Submitted jar file " + libraryFileName + " is too large");
        }
        byte[] buf = new byte[(int)length];
        in.readFully(buf);
        Path storePath = new Path(this.libraryCachePath + "/" + libraryFileName);
        FileSystem fileSystem = this.fs;
        synchronized (fileSystem) {
            if (!this.fs.exists(storePath)) {
                FSDataOutputStream fos = this.fs.create(storePath, false);
                fos.write(buf, 0, buf.length);
                fos.close();
            }
        }
    }

    public static void addLibrary(JobID jobID, Path name, long size, DataInput in) throws IOException {
        LibraryCacheManager lib = LibraryCacheManager.get();
        lib.addLibraryInternal(jobID, name, size, in);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addLibraryInternal(JobID jobID, Path name, long size, DataInput in) throws IOException {
        if (size > Integer.MAX_VALUE) {
            throw new IOException("Submitted jar file " + name + " is too large");
        }
        byte[] buf = new byte[(int)size];
        in.readFully(buf);
        this.md.reset();
        this.md.update(buf);
        String cacheName = StringUtils.byteToHexString((byte[])this.md.digest()) + ".jar";
        Path storePath = new Path(this.libraryCachePath + "/" + cacheName);
        FileSystem fileSystem = this.fs;
        synchronized (fileSystem) {
            if (!this.fs.exists(storePath)) {
                FSDataOutputStream fos = this.fs.create(storePath, false);
                fos.write(buf, 0, buf.length);
                fos.close();
            }
        }
        LibraryTranslationKey key = new LibraryTranslationKey(jobID, name);
        this.clientPathToCacheName.putIfAbsent(key, cacheName);
    }

    private static class LibraryTranslationKey {
        private final JobID jobID;
        private final Path clientPath;

        public LibraryTranslationKey(JobID jobID, Path clientPath) {
            this.jobID = jobID;
            this.clientPath = clientPath;
        }

        public int hashCode() {
            long temp = (this.jobID.hashCode() + this.clientPath.hashCode()) % Integer.MAX_VALUE;
            return (int)temp;
        }

        public boolean equals(Object obj) {
            LibraryTranslationKey key;
            if (obj == null) {
                return false;
            }
            if (this.jobID == null) {
                return false;
            }
            if (this.clientPath == null) {
                return false;
            }
            return obj instanceof LibraryTranslationKey && this.jobID.equals((key = (LibraryTranslationKey)obj).getJobID()) && this.clientPath.equals((Object)key.getClientPath());
        }

        public Path getClientPath() {
            return this.clientPath;
        }

        public JobID getJobID() {
            return this.jobID;
        }
    }

    private static class LibraryManagerEntry {
        private final ClassLoader classLoader;
        private final String[] requiredJarFiles;

        public LibraryManagerEntry(JobID id, String[] requiredJarFiles, URL[] urls) {
            String[] temp = requiredJarFiles;
            if (temp == null) {
                temp = new String[]{};
            }
            this.requiredJarFiles = temp;
            if (urls == null) {
                urls = new URL[]{};
            }
            this.classLoader = new URLClassLoader(urls, ClassLoader.getSystemClassLoader());
        }

        public ClassLoader getClassLoader() {
            return this.classLoader;
        }

        public String[] getRequiredJarFiles() {
            return this.requiredJarFiles;
        }
    }
}

