/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.jobmaster.slotpool;

import java.io.Serializable;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.clusterframework.types.ResourceProfile;
import org.apache.flink.runtime.jobmanager.slots.TaskManagerGateway;
import org.apache.flink.runtime.jobmaster.JobMasterId;
import org.apache.flink.runtime.jobmaster.RpcTaskManagerGateway;
import org.apache.flink.runtime.jobmaster.SlotInfo;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlot;
import org.apache.flink.runtime.jobmaster.slotpool.AllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.DefaultAllocatedSlotPool;
import org.apache.flink.runtime.jobmaster.slotpool.PhysicalSlot;
import org.apache.flink.runtime.taskexecutor.TaskExecutorGateway;
import org.apache.flink.runtime.taskexecutor.TestingTaskExecutorGatewayBuilder;
import org.apache.flink.runtime.taskmanager.LocalTaskManagerLocation;
import org.apache.flink.runtime.taskmanager.TaskManagerLocation;
import org.apache.flink.shaded.guava18.com.google.common.collect.Iterables;
import org.apache.flink.util.TestLogger;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.hamcrest.TypeSafeMatcher;
import org.junit.Assert;
import org.junit.Test;

public class DefaultAllocatedSlotPoolTest
extends TestLogger {
    @Test
    public void testAddSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        slotPool.addSlots(slots, 0L);
        this.assertSlotPoolContainsSlots(slotPool, slots);
        this.assertSlotPoolContainsFreeSlots(slotPool, slots);
    }

    @Test
    public void testRemoveSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        slotPool.addSlots(slots, 0L);
        Iterator<AllocatedSlot> iterator = slots.iterator();
        AllocatedSlot removedSlot = iterator.next();
        iterator.remove();
        slotPool.removeSlot(removedSlot.getAllocationId());
        this.assertSlotPoolContainsSlots(slotPool, slots);
    }

    @Test
    public void testRemoveSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        Collection<AllocatedSlot> slots = this.createAllocatedSlotsWithOwner(owner);
        AllocatedSlot otherSlot = this.createAllocatedSlot(ResourceID.generate());
        slots.add(otherSlot);
        slotPool.addSlots(slots, 0L);
        slotPool.removeSlots(owner);
        this.assertSlotPoolContainsSlots(slotPool, Collections.singleton(otherSlot));
    }

    @Test
    public void testRemoveSlotsReturnValue() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        AllocatedSlot slot1 = this.createAllocatedSlot(owner);
        AllocatedSlot slot2 = this.createAllocatedSlot(owner);
        slotPool.addSlots(Arrays.asList(slot1, slot2), 0L);
        slotPool.reserveFreeSlot(slot1.getAllocationId());
        AllocatedSlotPool.AllocatedSlotsAndReservationStatus allocatedSlotsAndReservationStatus = slotPool.removeSlots(owner);
        Assert.assertThat((Object)allocatedSlotsAndReservationStatus.getAllocatedSlots(), (Matcher)CoreMatchers.hasItems((Object[])new AllocatedSlot[]{slot1, slot2}));
        Assert.assertThat((Object)allocatedSlotsAndReservationStatus.wasFree(slot1.getAllocationId()), (Matcher)CoreMatchers.is((Object)false));
        Assert.assertThat((Object)allocatedSlotsAndReservationStatus.wasFree(slot2.getAllocationId()), (Matcher)CoreMatchers.is((Object)true));
    }

    @Test
    public void testContainsSlots() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(owner);
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assert.assertTrue((boolean)slotPool.containsSlots(owner));
        Assert.assertFalse((boolean)slotPool.containsSlots(ResourceID.generate()));
    }

    @Test
    public void testContainsSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assert.assertTrue((boolean)slotPool.containsSlot(allocatedSlot.getAllocationId()));
        Assert.assertFalse((boolean)slotPool.containsSlot(new AllocationID()));
    }

    @Test
    public void testReserveFreeSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> allSlots = this.createAllocatedSlots();
        ArrayList<AllocatedSlot> freeSlots = new ArrayList<AllocatedSlot>(allSlots);
        Iterator iterator = freeSlots.iterator();
        AllocatedSlot allocatedSlot = (AllocatedSlot)iterator.next();
        iterator.remove();
        slotPool.addSlots(allSlots, 0L);
        Assert.assertThat((Object)slotPool.reserveFreeSlot(allocatedSlot.getAllocationId()), (Matcher)Matchers.sameInstance((Object)allocatedSlot));
        this.assertSlotPoolContainsFreeSlots(slotPool, freeSlots);
        this.assertSlotPoolContainsSlots(slotPool, allSlots);
    }

    @Test(expected=IllegalStateException.class)
    public void testReserveNonFreeSlotFails() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot slot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(slot), 0L);
        slotPool.reserveFreeSlot(slot.getAllocationId());
        slotPool.reserveFreeSlot(slot.getAllocationId());
    }

    @Test
    public void testFreeingOfReservedSlot() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Collection<AllocatedSlot> slots = this.createAllocatedSlots();
        boolean initialTime = false;
        slotPool.addSlots(slots, 0L);
        AllocatedSlot slot = slots.iterator().next();
        slotPool.reserveFreeSlot(slot.getAllocationId());
        boolean releaseTime = true;
        Assert.assertTrue((boolean)slotPool.freeReservedSlot(slot.getAllocationId(), 1L).isPresent());
        this.assertSlotPoolContainsFreeSlots(slotPool, slots);
        for (AllocatedSlotPool.FreeSlotInfo freeSlotInfo : slotPool.getFreeSlotsInformation()) {
            long time = freeSlotInfo.getAllocationId().equals((Object)slot.getAllocationId()) ? 1L : 0L;
            Assert.assertThat((Object)freeSlotInfo.getFreeSince(), (Matcher)CoreMatchers.is((Object)time));
        }
    }

    @Test
    public void testFreeingOfFreeSlotIsIgnored() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot slot = this.createAllocatedSlot(null);
        slotPool.addSlots(Collections.singleton(slot), 0L);
        Assert.assertFalse((boolean)slotPool.freeReservedSlot(slot.getAllocationId(), 1L).isPresent());
        AllocatedSlotPool.FreeSlotInfo freeSlotInfo = (AllocatedSlotPool.FreeSlotInfo)Iterables.getOnlyElement((Iterable)slotPool.getFreeSlotsInformation());
        Assert.assertThat((Object)freeSlotInfo.getFreeSince(), (Matcher)CoreMatchers.is((Object)0L));
    }

    @Test
    public void testSlotUtilizationCalculation() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        ResourceID owner = ResourceID.generate();
        Collection<AllocatedSlot> slots = this.createAllocatedSlotsWithOwner(owner);
        slotPool.addSlots(slots, 0L);
        for (AllocatedSlotPool.FreeSlotInfo freeSlotInfo : slotPool.getFreeSlotsInformation()) {
            Assert.assertThat((Object)freeSlotInfo.asSlotInfo().getTaskExecutorUtilization(), (Matcher)Matchers.closeTo((double)0.0, (double)0.1));
        }
        int numAllocatedSlots = 0;
        for (AllocatedSlot slot : slots) {
            Assert.assertThat((Object)slotPool.reserveFreeSlot(slot.getAllocationId()), (Matcher)Matchers.sameInstance((Object)slot));
            ++numAllocatedSlots;
            for (AllocatedSlotPool.FreeSlotInfo freeSlotInfo : slotPool.getFreeSlotsInformation()) {
                double utilization = (double)numAllocatedSlots / (double)slots.size();
                Assert.assertThat((Object)freeSlotInfo.asSlotInfo().getTaskExecutorUtilization(), (Matcher)Matchers.closeTo((double)utilization, (double)0.1));
            }
        }
    }

    @Test
    public void testRemoveSlotsOfUnknownOwnerIsIgnored() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        slotPool.removeSlots(ResourceID.generate());
    }

    @Test
    public void testContainsFreeSlotReturnsTrueIfSlotIsFree() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(ResourceID.generate());
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        Assert.assertTrue((boolean)slotPool.containsFreeSlot(allocatedSlot.getAllocationId()));
    }

    @Test
    public void testContainsFreeSlotReturnsFalseIfSlotDoesNotExist() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        Assert.assertFalse((boolean)slotPool.containsFreeSlot(new AllocationID()));
    }

    @Test
    public void testContainsFreeSlotReturnsFalseIfSlotIsReserved() {
        DefaultAllocatedSlotPool slotPool = new DefaultAllocatedSlotPool();
        AllocatedSlot allocatedSlot = this.createAllocatedSlot(ResourceID.generate());
        slotPool.addSlots(Collections.singleton(allocatedSlot), 0L);
        slotPool.reserveFreeSlot(allocatedSlot.getAllocationId());
        Assert.assertFalse((boolean)slotPool.containsFreeSlot(allocatedSlot.getAllocationId()));
    }

    private void assertSlotPoolContainsSlots(DefaultAllocatedSlotPool slotPool, Collection<AllocatedSlot> slots) {
        Assert.assertThat((Object)slotPool.getAllSlotsInformation(), (Matcher)Matchers.hasSize((int)slots.size()));
        Map slotsPerAllocationId = slots.stream().collect(Collectors.toMap(AllocatedSlot::getAllocationId, Function.identity()));
        for (SlotInfo slotInfo : slotPool.getAllSlotsInformation()) {
            Assert.assertTrue((boolean)slotsPerAllocationId.containsKey(slotInfo.getAllocationId()));
            AllocatedSlot allocatedSlot = (AllocatedSlot)slotsPerAllocationId.get(slotInfo.getAllocationId());
            Assert.assertThat((Object)slotInfo, DefaultAllocatedSlotPoolTest.matchesPhysicalSlot((PhysicalSlot)allocatedSlot));
        }
    }

    private void assertSlotPoolContainsFreeSlots(DefaultAllocatedSlotPool slotPool, Collection<AllocatedSlot> allocatedSlots) {
        Collection freeSlotsInformation = slotPool.getFreeSlotsInformation();
        Assert.assertThat((Object)freeSlotsInformation, (Matcher)Matchers.hasSize((int)allocatedSlots.size()));
        Map allocatedSlotMap = allocatedSlots.stream().collect(Collectors.toMap(AllocatedSlot::getAllocationId, Function.identity()));
        for (AllocatedSlotPool.FreeSlotInfo freeSlotInfo : freeSlotsInformation) {
            Assert.assertTrue((boolean)allocatedSlotMap.containsKey(freeSlotInfo.getAllocationId()));
            Assert.assertThat((Object)freeSlotInfo.asSlotInfo(), DefaultAllocatedSlotPoolTest.matchesPhysicalSlot((PhysicalSlot)allocatedSlotMap.get(freeSlotInfo.getAllocationId())));
        }
    }

    static Matcher<SlotInfo> matchesPhysicalSlot(PhysicalSlot allocatedSlot) {
        return new SlotInfoMatcher(allocatedSlot);
    }

    private Collection<AllocatedSlot> createAllocatedSlots() {
        return new ArrayList<AllocatedSlot>(Arrays.asList(this.createAllocatedSlot(null), this.createAllocatedSlot(null), this.createAllocatedSlot(null)));
    }

    private Collection<AllocatedSlot> createAllocatedSlotsWithOwner(ResourceID owner) {
        return new ArrayList<AllocatedSlot>(Arrays.asList(this.createAllocatedSlot(owner), this.createAllocatedSlot(owner), this.createAllocatedSlot(owner)));
    }

    @Nonnull
    private AllocatedSlot createAllocatedSlot(@Nullable ResourceID owner) {
        return new AllocatedSlot(new AllocationID(), (TaskManagerLocation)(owner == null ? new LocalTaskManagerLocation() : new TaskManagerLocation(owner, InetAddress.getLoopbackAddress(), 41)), 0, ResourceProfile.UNKNOWN, (TaskManagerGateway)new RpcTaskManagerGateway((TaskExecutorGateway)new TestingTaskExecutorGatewayBuilder().createTestingTaskExecutorGateway(), JobMasterId.generate()));
    }

    static class SlotInfoMatcher
    extends TypeSafeMatcher<SlotInfo> {
        private final PhysicalSlot physicalSlot;

        SlotInfoMatcher(PhysicalSlot physicalSlot) {
            this.physicalSlot = physicalSlot;
        }

        public void describeTo(Description description) {
            description.appendText("SlotInfo with values: ");
            description.appendValueList("{", ",", "}", (Object[])new Serializable[]{this.physicalSlot.getAllocationId(), Integer.valueOf(this.physicalSlot.getPhysicalSlotNumber()), this.physicalSlot.getResourceProfile(), this.physicalSlot.getTaskManagerLocation()});
        }

        protected boolean matchesSafely(SlotInfo item) {
            return item.getAllocationId().equals((Object)this.physicalSlot.getAllocationId()) && item.getPhysicalSlotNumber() == this.physicalSlot.getPhysicalSlotNumber() && item.getResourceProfile().equals((Object)this.physicalSlot.getResourceProfile()) && item.getTaskManagerLocation().equals((Object)this.physicalSlot.getTaskManagerLocation());
        }
    }
}

