/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.ConcurrentModificationException;
import java.util.List;
import java.util.concurrent.Executor;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.jobgraph.JobStatus;
import org.apache.flink.runtime.state.RetrievableStateHandle;
import org.apache.flink.runtime.zookeeper.ZooKeeperStateHandleStore;
import org.apache.flink.util.FlinkException;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.function.ThrowingConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ZooKeeperCompletedCheckpointStore
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(ZooKeeperCompletedCheckpointStore.class);
    private static final Comparator<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> STRING_COMPARATOR = Comparator.comparing(o -> (String)o.f1);
    private final ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper;
    private final int maxNumberOfCheckpointsToRetain;
    private final ArrayDeque<CompletedCheckpoint> completedCheckpoints;
    private final Executor executor;

    public ZooKeeperCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, ZooKeeperStateHandleStore<CompletedCheckpoint> checkpointsInZooKeeper, Executor executor) {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        this.checkpointsInZooKeeper = (ZooKeeperStateHandleStore)Preconditions.checkNotNull(checkpointsInZooKeeper);
        this.completedCheckpoints = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        this.executor = (Executor)Preconditions.checkNotNull((Object)executor);
    }

    @Override
    public boolean requiresExternalizedCheckpoints() {
        return true;
    }

    @Override
    public void recover() throws Exception {
        List<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> initialCheckpoints;
        LOG.info("Recovering checkpoints from ZooKeeper.");
        while (true) {
            try {
                initialCheckpoints = this.checkpointsInZooKeeper.getAllAndLock();
            }
            catch (ConcurrentModificationException e) {
                LOG.warn("Concurrent modification while reading from ZooKeeper. Retrying.");
                continue;
            }
            break;
        }
        Collections.sort(initialCheckpoints, STRING_COMPARATOR);
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in ZooKeeper.", (Object)numberOfInitialCheckpoints);
        ArrayList<CompletedCheckpoint> lastTryRetrievedCheckpoints = new ArrayList<CompletedCheckpoint>(numberOfInitialCheckpoints);
        ArrayList<CompletedCheckpoint> retrievedCheckpoints = new ArrayList<CompletedCheckpoint>(numberOfInitialCheckpoints);
        do {
            LOG.info("Trying to fetch {} checkpoints from storage.", (Object)numberOfInitialCheckpoints);
            lastTryRetrievedCheckpoints.clear();
            lastTryRetrievedCheckpoints.addAll(retrievedCheckpoints);
            retrievedCheckpoints.clear();
            for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpointStateHandle : initialCheckpoints) {
                CompletedCheckpoint completedCheckpoint = null;
                try {
                    completedCheckpoint = ZooKeeperCompletedCheckpointStore.retrieveCompletedCheckpoint(checkpointStateHandle);
                    if (completedCheckpoint == null) continue;
                    retrievedCheckpoints.add(completedCheckpoint);
                }
                catch (Exception e) {
                    LOG.warn("Could not retrieve checkpoint, not adding to list of recovered checkpoints.", (Throwable)e);
                }
            }
        } while (retrievedCheckpoints.size() != numberOfInitialCheckpoints && !CompletedCheckpoint.checkpointsMatch(lastTryRetrievedCheckpoints, retrievedCheckpoints));
        this.completedCheckpoints.clear();
        this.completedCheckpoints.addAll(retrievedCheckpoints);
        if (this.completedCheckpoints.isEmpty() && numberOfInitialCheckpoints > 0) {
            throw new FlinkException("Could not read any of the " + numberOfInitialCheckpoints + " checkpoints from storage.");
        }
        if (this.completedCheckpoints.size() != numberOfInitialCheckpoints) {
            LOG.warn("Could only fetch {} of {} checkpoints from storage.", (Object)this.completedCheckpoints.size(), (Object)numberOfInitialCheckpoints);
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint) throws Exception {
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = ZooKeeperCompletedCheckpointStore.checkpointIdToPath(checkpoint.getCheckpointID());
        this.checkpointsInZooKeeper.addAndLock(path, checkpoint);
        this.completedCheckpoints.addLast(checkpoint);
        while (this.completedCheckpoints.size() > this.maxNumberOfCheckpointsToRetain) {
            CompletedCheckpoint completedCheckpoint = this.completedCheckpoints.removeFirst();
            this.tryRemoveCompletedCheckpoint(completedCheckpoint, (ThrowingConsumer<CompletedCheckpoint, Exception>)((ThrowingConsumer)CompletedCheckpoint::discardOnSubsume));
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    private void tryRemoveCompletedCheckpoint(CompletedCheckpoint completedCheckpoint, ThrowingConsumer<CompletedCheckpoint, Exception> discardCallback) {
        try {
            if (this.tryRemove(completedCheckpoint.getCheckpointID())) {
                this.executor.execute(() -> {
                    try {
                        discardCallback.accept((Object)completedCheckpoint);
                    }
                    catch (Exception e) {
                        LOG.warn("Could not discard completed checkpoint {}.", (Object)completedCheckpoint.getCheckpointID(), (Object)e);
                    }
                });
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to subsume the old checkpoint", (Throwable)e);
        }
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() throws Exception {
        ArrayList<CompletedCheckpoint> checkpoints = new ArrayList<CompletedCheckpoint>(this.completedCheckpoints);
        return checkpoints;
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.completedCheckpoints.size();
    }

    @Override
    public int getMaxNumberOfRetainedCheckpoints() {
        return this.maxNumberOfCheckpointsToRetain;
    }

    @Override
    public void shutdown(JobStatus jobStatus) throws Exception {
        if (jobStatus.isGloballyTerminalState()) {
            LOG.info("Shutting down");
            for (CompletedCheckpoint checkpoint : this.completedCheckpoints) {
                this.tryRemoveCompletedCheckpoint(checkpoint, (ThrowingConsumer<CompletedCheckpoint, Exception>)((ThrowingConsumer)completedCheckpoint -> completedCheckpoint.discardOnShutdown(jobStatus)));
            }
            this.completedCheckpoints.clear();
            this.checkpointsInZooKeeper.deleteChildren();
        } else {
            LOG.info("Suspending");
            this.completedCheckpoints.clear();
            this.checkpointsInZooKeeper.releaseAll();
        }
    }

    private boolean tryRemove(long checkpointId) throws Exception {
        return this.checkpointsInZooKeeper.releaseAndTryRemove(ZooKeeperCompletedCheckpointStore.checkpointIdToPath(checkpointId));
    }

    public static String checkpointIdToPath(long checkpointId) {
        return String.format("/%019d", checkpointId);
    }

    public static long pathToCheckpointId(String path) {
        try {
            String numberString = '/' == path.charAt(0) ? path.substring(1) : path;
            return Long.parseLong(numberString);
        }
        catch (NumberFormatException e) {
            LOG.warn("Could not parse checkpoint id from {}. This indicates that the checkpoint id to path conversion has changed.", (Object)path);
            return -1L;
        }
    }

    private static CompletedCheckpoint retrieveCompletedCheckpoint(Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandlePath) throws FlinkException {
        long checkpointId = ZooKeeperCompletedCheckpointStore.pathToCheckpointId((String)stateHandlePath.f1);
        LOG.info("Trying to retrieve checkpoint {}.", (Object)checkpointId);
        try {
            return (CompletedCheckpoint)((RetrievableStateHandle)stateHandlePath.f0).retrieveState();
        }
        catch (ClassNotFoundException cnfe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandlePath.f1 + ". This indicates that you are trying to recover from state written by an older Flink version which is not compatible. Try cleaning the state handle store.", (Throwable)cnfe);
        }
        catch (IOException ioe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandlePath.f1 + ". This indicates that the retrieved state handle is broken. Try cleaning the state handle store.", (Throwable)ioe);
        }
    }
}

