/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.hadoop.shaded.org.apache.curator;

import java.io.Closeable;
import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.flink.hadoop.shaded.org.apache.curator.CuratorConnectionLossException;
import org.apache.flink.hadoop.shaded.org.apache.curator.HandleHolder;
import org.apache.flink.hadoop.shaded.org.apache.curator.SessionFailRetryLoop;
import org.apache.flink.hadoop.shaded.org.apache.curator.TimeTrace;
import org.apache.flink.hadoop.shaded.org.apache.curator.drivers.TracerDriver;
import org.apache.flink.hadoop.shaded.org.apache.curator.ensemble.EnsembleProvider;
import org.apache.flink.hadoop.shaded.org.apache.curator.utils.CloseableUtils;
import org.apache.flink.hadoop.shaded.org.apache.curator.utils.ZookeeperFactory;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConnectionState
implements Watcher,
Closeable {
    private static final int MAX_BACKGROUND_EXCEPTIONS = 10;
    private static final boolean LOG_EVENTS = Boolean.getBoolean("curator-log-events");
    private final Logger log = LoggerFactory.getLogger(this.getClass());
    private final HandleHolder zooKeeper;
    private final AtomicBoolean isConnected = new AtomicBoolean(false);
    private final EnsembleProvider ensembleProvider;
    private final int sessionTimeoutMs;
    private final int connectionTimeoutMs;
    private final AtomicReference<TracerDriver> tracer;
    private final Queue<Exception> backgroundExceptions = new ConcurrentLinkedQueue<Exception>();
    private final Queue<Watcher> parentWatchers = new ConcurrentLinkedQueue<Watcher>();
    private final AtomicLong instanceIndex = new AtomicLong();
    private volatile long connectionStartMs = 0L;

    ConnectionState(ZookeeperFactory zookeeperFactory, EnsembleProvider ensembleProvider, int sessionTimeoutMs, int connectionTimeoutMs, Watcher parentWatcher, AtomicReference<TracerDriver> tracer, boolean canBeReadOnly) {
        this.ensembleProvider = ensembleProvider;
        this.sessionTimeoutMs = sessionTimeoutMs;
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.tracer = tracer;
        if (parentWatcher != null) {
            this.parentWatchers.offer(parentWatcher);
        }
        this.zooKeeper = new HandleHolder(zookeeperFactory, this, ensembleProvider, sessionTimeoutMs, canBeReadOnly);
    }

    ZooKeeper getZooKeeper() throws Exception {
        if (SessionFailRetryLoop.sessionForThreadHasFailed()) {
            throw new SessionFailRetryLoop.SessionFailedException();
        }
        Exception exception = this.backgroundExceptions.poll();
        if (exception != null) {
            this.tracer.get().addCount("background-exceptions", 1);
            throw exception;
        }
        boolean localIsConnected = this.isConnected.get();
        if (!localIsConnected) {
            this.checkTimeouts();
        }
        return this.zooKeeper.getZooKeeper();
    }

    boolean isConnected() {
        return this.isConnected.get();
    }

    void start() throws Exception {
        this.log.debug("Starting");
        this.ensembleProvider.start();
        this.reset();
    }

    @Override
    public void close() throws IOException {
        this.log.debug("Closing");
        CloseableUtils.closeQuietly(this.ensembleProvider);
        try {
            this.zooKeeper.closeAndClear();
        }
        catch (Exception e) {
            throw new IOException(e);
        }
        finally {
            this.isConnected.set(false);
        }
    }

    void addParentWatcher(Watcher watcher) {
        this.parentWatchers.offer(watcher);
    }

    void removeParentWatcher(Watcher watcher) {
        this.parentWatchers.remove(watcher);
    }

    long getInstanceIndex() {
        return this.instanceIndex.get();
    }

    @Override
    public void process(WatchedEvent event) {
        boolean wasConnected;
        if (LOG_EVENTS) {
            this.log.debug("ConnectState watcher: " + event);
        }
        for (Watcher parentWatcher : this.parentWatchers) {
            TimeTrace timeTrace = new TimeTrace("connection-state-parent-process", this.tracer.get());
            parentWatcher.process(event);
            timeTrace.commit();
        }
        boolean newIsConnected = wasConnected = this.isConnected.get();
        if (event.getType() == Watcher.Event.EventType.None) {
            newIsConnected = this.checkState(event.getState(), wasConnected);
        }
        if (newIsConnected != wasConnected) {
            this.isConnected.set(newIsConnected);
            this.connectionStartMs = System.currentTimeMillis();
        }
    }

    EnsembleProvider getEnsembleProvider() {
        return this.ensembleProvider;
    }

    private synchronized void checkTimeouts() throws Exception {
        int minTimeout = Math.min(this.sessionTimeoutMs, this.connectionTimeoutMs);
        long elapsed = System.currentTimeMillis() - this.connectionStartMs;
        if (elapsed >= (long)minTimeout) {
            if (this.zooKeeper.hasNewConnectionString()) {
                this.handleNewConnectionString();
            } else {
                int maxTimeout = Math.max(this.sessionTimeoutMs, this.connectionTimeoutMs);
                if (elapsed > (long)maxTimeout) {
                    if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                        this.log.warn(String.format("Connection attempt unsuccessful after %d (greater than max timeout of %d). Resetting connection and trying again with a new connection.", elapsed, maxTimeout));
                    }
                    this.reset();
                } else {
                    CuratorConnectionLossException connectionLossException = new CuratorConnectionLossException();
                    if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                        this.log.error(String.format("Connection timed out for connection string (%s) and timeout (%d) / elapsed (%d)", this.zooKeeper.getConnectionString(), this.connectionTimeoutMs, elapsed), (Throwable)connectionLossException);
                    }
                    this.tracer.get().addCount("connections-timed-out", 1);
                    throw connectionLossException;
                }
            }
        }
    }

    private synchronized void reset() throws Exception {
        this.log.debug("reset");
        this.instanceIndex.incrementAndGet();
        this.isConnected.set(false);
        this.connectionStartMs = System.currentTimeMillis();
        this.zooKeeper.closeAndReset();
        this.zooKeeper.getZooKeeper();
    }

    private boolean checkState(Watcher.Event.KeeperState state, boolean wasConnected) {
        boolean isConnected = wasConnected;
        boolean checkNewConnectionString = true;
        switch (state) {
            default: {
                isConnected = false;
                break;
            }
            case SyncConnected: 
            case ConnectedReadOnly: {
                isConnected = true;
                break;
            }
            case AuthFailed: {
                isConnected = false;
                this.log.error("Authentication failed");
                break;
            }
            case Expired: {
                isConnected = false;
                checkNewConnectionString = false;
                this.handleExpiredSession();
            }
            case SaslAuthenticated: 
        }
        if (checkNewConnectionString && this.zooKeeper.hasNewConnectionString()) {
            this.handleNewConnectionString();
        }
        return isConnected;
    }

    private void handleNewConnectionString() {
        this.log.info("Connection string changed");
        this.tracer.get().addCount("connection-string-changed", 1);
        try {
            this.reset();
        }
        catch (Exception e) {
            this.queueBackgroundException(e);
        }
    }

    private void handleExpiredSession() {
        this.log.warn("Session expired event received");
        this.tracer.get().addCount("session-expired", 1);
        try {
            this.reset();
        }
        catch (Exception e) {
            this.queueBackgroundException(e);
        }
    }

    private void queueBackgroundException(Exception e) {
        while (this.backgroundExceptions.size() >= 10) {
            this.backgroundExceptions.poll();
        }
        this.backgroundExceptions.offer(e);
    }
}

