/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.streaming.api.functions.sink.filesystem;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import javax.annotation.Nullable;
import org.apache.flink.annotation.Internal;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.fs.RecoverableWriter;
import org.apache.flink.streaming.api.functions.sink.filesystem.BucketState;
import org.apache.flink.streaming.api.functions.sink.filesystem.PartFileWriter;
import org.apache.flink.streaming.api.functions.sink.filesystem.RollingPolicy;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Internal
public class Bucket<IN, BucketID> {
    private static final Logger LOG = LoggerFactory.getLogger(Bucket.class);
    private static final String PART_PREFIX = "part";
    private final BucketID bucketId;
    private final Path bucketPath;
    private final int subtaskIndex;
    private final PartFileWriter.PartFileFactory<IN, BucketID> partFileFactory;
    private final RecoverableWriter fsWriter;
    private final RollingPolicy<IN, BucketID> rollingPolicy;
    private final Map<Long, List<RecoverableWriter.CommitRecoverable>> pendingPartsPerCheckpoint;
    private long partCounter;
    @Nullable
    private PartFileWriter<IN, BucketID> inProgressPart;
    private List<RecoverableWriter.CommitRecoverable> pendingPartsForCurrentCheckpoint;

    private Bucket(RecoverableWriter fsWriter, int subtaskIndex, BucketID bucketId, Path bucketPath, long initialPartCounter, PartFileWriter.PartFileFactory<IN, BucketID> partFileFactory, RollingPolicy<IN, BucketID> rollingPolicy) {
        this.fsWriter = (RecoverableWriter)Preconditions.checkNotNull((Object)fsWriter);
        this.subtaskIndex = subtaskIndex;
        this.bucketId = Preconditions.checkNotNull(bucketId);
        this.bucketPath = (Path)Preconditions.checkNotNull((Object)bucketPath);
        this.partCounter = initialPartCounter;
        this.partFileFactory = (PartFileWriter.PartFileFactory)Preconditions.checkNotNull(partFileFactory);
        this.rollingPolicy = (RollingPolicy)Preconditions.checkNotNull(rollingPolicy);
        this.pendingPartsForCurrentCheckpoint = new ArrayList<RecoverableWriter.CommitRecoverable>();
        this.pendingPartsPerCheckpoint = new HashMap<Long, List<RecoverableWriter.CommitRecoverable>>();
    }

    private Bucket(RecoverableWriter fsWriter, int subtaskIndex, long initialPartCounter, PartFileWriter.PartFileFactory<IN, BucketID> partFileFactory, RollingPolicy<IN, BucketID> rollingPolicy, BucketState<BucketID> bucketState) throws IOException {
        this(fsWriter, subtaskIndex, bucketState.getBucketId(), bucketState.getBucketPath(), initialPartCounter, partFileFactory, rollingPolicy);
        super.restoreInProgressFile(bucketState);
        super.commitRecoveredPendingFiles(bucketState);
    }

    private void restoreInProgressFile(BucketState<BucketID> state) throws IOException {
        if (state.hasInProgressResumableFile()) {
            RecoverableWriter.ResumeRecoverable resumable = state.getInProgressResumableFile();
            this.inProgressPart = this.partFileFactory.resumeFrom(this.bucketId, this.fsWriter, resumable, state.getInProgressFileCreationTime());
        }
    }

    private void commitRecoveredPendingFiles(BucketState<BucketID> state) throws IOException {
        for (List<RecoverableWriter.CommitRecoverable> committables : state.getCommittableFilesPerCheckpoint().values()) {
            for (RecoverableWriter.CommitRecoverable committable : committables) {
                this.fsWriter.recoverForCommit(committable).commitAfterRecovery();
            }
        }
    }

    BucketID getBucketId() {
        return this.bucketId;
    }

    Path getBucketPath() {
        return this.bucketPath;
    }

    long getPartCounter() {
        return this.partCounter;
    }

    boolean isActive() {
        return this.inProgressPart != null || !this.pendingPartsForCurrentCheckpoint.isEmpty() || !this.pendingPartsPerCheckpoint.isEmpty();
    }

    void merge(Bucket<IN, BucketID> bucket) throws IOException {
        Preconditions.checkNotNull(bucket);
        Preconditions.checkState((boolean)Objects.equals(bucket.bucketPath, this.bucketPath));
        Preconditions.checkState((boolean)bucket.pendingPartsForCurrentCheckpoint.isEmpty());
        Preconditions.checkState((boolean)bucket.pendingPartsPerCheckpoint.isEmpty());
        RecoverableWriter.CommitRecoverable committable = super.closePartFile();
        if (committable != null) {
            this.pendingPartsForCurrentCheckpoint.add(committable);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Subtask {} merging buckets for bucket id={}", (Object)this.subtaskIndex, this.bucketId);
        }
    }

    void write(IN element, long currentTime) throws IOException {
        if (this.inProgressPart == null || this.rollingPolicy.shouldRollOnEvent(this.inProgressPart, element)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Subtask {} closing in-progress part file for bucket id={} due to element {}.", new Object[]{this.subtaskIndex, this.bucketId, element});
            }
            this.rollPartFile(currentTime);
        }
        this.inProgressPart.write(element, currentTime);
    }

    private void rollPartFile(long currentTime) throws IOException {
        this.closePartFile();
        Path partFilePath = this.assembleNewPartPath();
        this.inProgressPart = this.partFileFactory.openNew(this.bucketId, this.fsWriter, partFilePath, currentTime);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Subtask {} opening new part file \"{}\" for bucket id={}.", new Object[]{this.subtaskIndex, partFilePath.getName(), this.bucketId});
        }
        ++this.partCounter;
    }

    private Path assembleNewPartPath() {
        return new Path(this.bucketPath, "part-" + this.subtaskIndex + '-' + this.partCounter);
    }

    private RecoverableWriter.CommitRecoverable closePartFile() throws IOException {
        RecoverableWriter.CommitRecoverable committable = null;
        if (this.inProgressPart != null) {
            committable = this.inProgressPart.closeForCommit();
            this.pendingPartsForCurrentCheckpoint.add(committable);
            this.inProgressPart = null;
        }
        return committable;
    }

    void disposePartFile() {
        if (this.inProgressPart != null) {
            this.inProgressPart.dispose();
        }
    }

    BucketState<BucketID> onReceptionOfCheckpoint(long checkpointId) throws IOException {
        this.prepareBucketForCheckpointing(checkpointId);
        RecoverableWriter.ResumeRecoverable inProgressResumable = null;
        long inProgressFileCreationTime = Long.MAX_VALUE;
        if (this.inProgressPart != null) {
            inProgressResumable = this.inProgressPart.persist();
            inProgressFileCreationTime = this.inProgressPart.getCreationTime();
        }
        return new BucketState<BucketID>(this.bucketId, this.bucketPath, inProgressFileCreationTime, inProgressResumable, this.pendingPartsPerCheckpoint);
    }

    private void prepareBucketForCheckpointing(long checkpointId) throws IOException {
        if (this.inProgressPart != null && this.rollingPolicy.shouldRollOnCheckpoint(this.inProgressPart)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Subtask {} closing in-progress part file for bucket id={} on checkpoint.", (Object)this.subtaskIndex, this.bucketId);
            }
            this.closePartFile();
        }
        if (!this.pendingPartsForCurrentCheckpoint.isEmpty()) {
            this.pendingPartsPerCheckpoint.put(checkpointId, this.pendingPartsForCurrentCheckpoint);
            this.pendingPartsForCurrentCheckpoint = new ArrayList<RecoverableWriter.CommitRecoverable>();
        }
    }

    void onSuccessfulCompletionOfCheckpoint(long checkpointId) throws IOException {
        Preconditions.checkNotNull((Object)this.fsWriter);
        Iterator<Map.Entry<Long, List<RecoverableWriter.CommitRecoverable>>> it = this.pendingPartsPerCheckpoint.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<Long, List<RecoverableWriter.CommitRecoverable>> entry = it.next();
            if (entry.getKey() > checkpointId) continue;
            for (RecoverableWriter.CommitRecoverable committable : entry.getValue()) {
                this.fsWriter.recoverForCommit(committable).commit();
            }
            it.remove();
        }
    }

    void onProcessingTime(long timestamp) throws IOException {
        if (this.inProgressPart != null && this.rollingPolicy.shouldRollOnProcessingTime(this.inProgressPart, timestamp)) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Subtask {} closing in-progress part file for bucket id={} due to processing time rolling policy (in-progress file created @ {}, last updated @ {} and current time is {}).", new Object[]{this.subtaskIndex, this.bucketId, this.inProgressPart.getCreationTime(), this.inProgressPart.getLastUpdateTime(), timestamp});
            }
            this.closePartFile();
        }
    }

    @VisibleForTesting
    Map<Long, List<RecoverableWriter.CommitRecoverable>> getPendingPartsPerCheckpoint() {
        return this.pendingPartsPerCheckpoint;
    }

    @Nullable
    @VisibleForTesting
    PartFileWriter<IN, BucketID> getInProgressPart() {
        return this.inProgressPart;
    }

    @VisibleForTesting
    List<RecoverableWriter.CommitRecoverable> getPendingPartsForCurrentCheckpoint() {
        return this.pendingPartsForCurrentCheckpoint;
    }

    static <IN, BucketID> Bucket<IN, BucketID> getNew(RecoverableWriter fsWriter, int subtaskIndex, BucketID bucketId, Path bucketPath, long initialPartCounter, PartFileWriter.PartFileFactory<IN, BucketID> partFileFactory, RollingPolicy<IN, BucketID> rollingPolicy) {
        return new Bucket<IN, BucketID>(fsWriter, subtaskIndex, bucketId, bucketPath, initialPartCounter, partFileFactory, rollingPolicy);
    }

    static <IN, BucketID> Bucket<IN, BucketID> restore(RecoverableWriter fsWriter, int subtaskIndex, long initialPartCounter, PartFileWriter.PartFileFactory<IN, BucketID> partFileFactory, RollingPolicy<IN, BucketID> rollingPolicy, BucketState<BucketID> bucketState) throws IOException {
        return new Bucket<IN, BucketID>(fsWriter, subtaskIndex, initialPartCounter, partFileFactory, rollingPolicy, bucketState);
    }
}

