/**
 *  Licensed to the Apache Software Foundation (ASF) under one or more
 *  contributor license agreements.  See the NOTICE file distributed with
 *  this work for additional information regarding copyright ownership.
 *  The ASF licenses this file to You under the Apache License, Version 2.0
 *  (the "License"); you may not use this file except in compliance with
 *  the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.apache.geronimo.security.jaas.server;

import java.security.Principal;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import javax.security.auth.callback.Callback;
import javax.security.auth.login.LoginException;

import org.apache.geronimo.common.GeronimoSecurityException;
import org.apache.geronimo.management.geronimo.LoginService;


/**
 * Interface used to connect to the JaasLoginService via remoting, etc.  This
 * may no longer be necessary?
 *
 * @version $Rev: 487175 $ $Date: 2006-12-14 03:10:31 -0800 (Thu, 14 Dec 2006) $
 */
public interface JaasLoginServiceMBean extends LoginService {
    /**
     * Starts a new authentication process on behalf of an end user.  The
     * returned session id will identify that user throughout the user's interaction
     * with the server.  On the server side, that means maintaining the
     * Subject and Principals for the user.
     *
     * @return The <code>JaasSessionId</code> used as an argument for the rest of the
     *         methods in this class.
     */
    public JaasSessionId connectToRealm(String realmName);

    /**
     * Gets the login module configuration for the specified realm.  The
     * caller needs that in order to perform the authentication process.
     */
    public JaasLoginModuleConfiguration[] getLoginConfiguration(JaasSessionId sessionHandle) throws LoginException;

    /**
     * Retrieves callbacks for a server side login module.  When the client
     * is going through the configured login modules, if a specific login
     * module is client-side, it will be handled directly.  If it is
     * server-side, the client gets the callbacks (using this method),
     * populates them, and sends them back to the server.
     */
    public Callback[] getServerLoginCallbacks(JaasSessionId sessionHandle, int loginModuleIndex) throws LoginException;

    /**
     * Returns populated callbacks for a server side login module.  When the
     * client is going through the configured login modules, if a specific
     * login module is client-side, it will be handled directly.  If it is
     * server-side, the client gets the callbacks, populates them, and sends
     * them back to the server (using this method).
     */
    public boolean performLogin(JaasSessionId sessionHandle, int loginModuleIndex, Callback[] results) throws LoginException;

    /**
     * Indicates that the overall login succeeded, and some principals were
     * generated by a client-side login module.  This method needs to be called
     * once for each client-side login module, to specify Principals for each
     * module.
     */
    public boolean performCommit(JaasSessionId sessionHandle, int loginModuleIndex) throws LoginException;

    /**
     * Indicates that the overall login succeeded.  All login modules that were
     * touched should have been logged in and committed before calling this.
     *
     * @param sessionHandle the handle to the login session
     * @return the identifier principal
     * @throws LoginException if the handle is no longer valid.
     */
    public Principal loginSucceeded(JaasSessionId sessionHandle) throws LoginException;

    /**
     * Indicates that the overall login failed, and the server should release
     * any resources associated with the user ID.
     */
    public void loginFailed(JaasSessionId sessionHandle);

    /**
     * Indicates that the client has logged out, and the server should release
     * any resources associated with the user ID.
     */
    public void logout(JaasSessionId sessionHandle) throws LoginException;

    /**
     * Syncs the shared state that's on the client with the shared state that
     * is on the server.
     *
     * @param sessionHandle
     * @param sharedState the shared state that is on the client
     * @return the sync'd shared state that is on the server
     */
    public Map syncShareState(JaasSessionId sessionHandle, Map sharedState) throws LoginException;

    /**
     * Syncs the set of principals that are on the client with the set of principals that
     * are on the server.
     *
     * @param sessionHandle
     * @param principals the set of principals that are on the client side
     * @return the sync'd set of principals that are on the server
     */
    public Set syncPrincipals(JaasSessionId sessionHandle, Set principals) throws LoginException;

    /**
     * Indicates that the overall login failed.  This method needs to be called
     * once for each client-side login module.
     */
    public boolean performAbort(JaasSessionId sessionHandle, int lmIndex) throws LoginException;
}
