/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs.s3a.s3guard;

import com.amazonaws.AmazonClientException;
import com.amazonaws.SdkBaseException;
import com.amazonaws.services.dynamodbv2.AmazonDynamoDB;
import com.amazonaws.services.dynamodbv2.document.BatchWriteItemOutcome;
import com.amazonaws.services.dynamodbv2.document.DynamoDB;
import com.amazonaws.services.dynamodbv2.document.Item;
import com.amazonaws.services.dynamodbv2.document.ItemCollection;
import com.amazonaws.services.dynamodbv2.document.PrimaryKey;
import com.amazonaws.services.dynamodbv2.document.PutItemOutcome;
import com.amazonaws.services.dynamodbv2.document.ScanOutcome;
import com.amazonaws.services.dynamodbv2.document.Table;
import com.amazonaws.services.dynamodbv2.document.TableWriteItems;
import com.amazonaws.services.dynamodbv2.document.spec.GetItemSpec;
import com.amazonaws.services.dynamodbv2.document.spec.QuerySpec;
import com.amazonaws.services.dynamodbv2.document.utils.ValueMap;
import com.amazonaws.services.dynamodbv2.model.CreateTableRequest;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughput;
import com.amazonaws.services.dynamodbv2.model.ProvisionedThroughputDescription;
import com.amazonaws.services.dynamodbv2.model.ResourceInUseException;
import com.amazonaws.services.dynamodbv2.model.ResourceNotFoundException;
import com.amazonaws.services.dynamodbv2.model.TableDescription;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.s3a.Invoker;
import org.apache.hadoop.fs.s3a.S3AFileSystem;
import org.apache.hadoop.fs.s3a.S3AInstrumentation;
import org.apache.hadoop.fs.s3a.S3ARetryPolicy;
import org.apache.hadoop.fs.s3a.S3AUtils;
import org.apache.hadoop.fs.s3a.Tristate;
import org.apache.hadoop.fs.s3a.s3guard.DescendantsIterator;
import org.apache.hadoop.fs.s3a.s3guard.DirListingMetadata;
import org.apache.hadoop.fs.s3a.s3guard.DynamoDBClientFactory;
import org.apache.hadoop.fs.s3a.s3guard.MetadataStore;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadata;
import org.apache.hadoop.fs.s3a.s3guard.PathMetadataDynamoDBTranslation;
import org.apache.hadoop.fs.s3a.s3guard.S3Guard;
import org.apache.hadoop.io.retry.RetryPolicies;
import org.apache.hadoop.io.retry.RetryPolicy;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.ReflectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
@InterfaceStability.Evolving
public class DynamoDBMetadataStore
implements MetadataStore {
    public static final Logger LOG = LoggerFactory.getLogger(DynamoDBMetadataStore.class);
    public static final String VERSION_MARKER = "../VERSION";
    public static final int VERSION = 100;
    public static final String E_NO_VERSION_MARKER = "S3Guard table lacks version marker.";
    public static final String E_INCOMPATIBLE_VERSION = "Database table is from an incompatible S3Guard version.";
    public static final long MIN_RETRY_SLEEP_MSEC = 100L;
    @VisibleForTesting
    static final String DESCRIPTION = "S3Guard metadata store in DynamoDB";
    @VisibleForTesting
    static final String READ_CAPACITY = "read-capacity";
    @VisibleForTesting
    static final String WRITE_CAPACITY = "write-capacity";
    @VisibleForTesting
    static final String STATUS = "status";
    @VisibleForTesting
    static final String TABLE = "table";
    private static ValueMap deleteTrackingValueMap = new ValueMap().withBoolean(":false", false);
    private DynamoDB dynamoDB;
    private String region;
    private Table table;
    private String tableName;
    private Configuration conf;
    private String username;
    private RetryPolicy dataAccessRetryPolicy;
    private S3AInstrumentation.S3GuardInstrumentation instrumentation;
    private S3AFileSystem owner;
    private Invoker invoker = new Invoker(RetryPolicies.TRY_ONCE_THEN_FAIL, Invoker.NO_OP);
    private Invoker dataAccess;
    private static final int THROTTLE_EVENT_LOG_LIMIT = 100;
    private AtomicInteger throttleEventCount = new AtomicInteger(0);

    private static DynamoDB createDynamoDB(Configuration conf, String s3Region) throws IOException {
        Preconditions.checkNotNull((Object)conf);
        Class cls = conf.getClass("fs.s3a.s3guard.ddb.client.factory.impl", S3Guard.S3GUARD_DDB_CLIENT_FACTORY_IMPL_DEFAULT, DynamoDBClientFactory.class);
        LOG.debug("Creating DynamoDB client {} with S3 region {}", (Object)cls, (Object)s3Region);
        AmazonDynamoDB dynamoDBClient = ((DynamoDBClientFactory)ReflectionUtils.newInstance((Class)cls, (Configuration)conf)).createDynamoDBClient(s3Region);
        return new DynamoDB(dynamoDBClient);
    }

    @Override
    public void initialize(FileSystem fs) throws IOException {
        Preconditions.checkNotNull((Object)fs, (Object)"Null filesystem");
        Preconditions.checkArgument((boolean)(fs instanceof S3AFileSystem), (Object)"DynamoDBMetadataStore only supports S3A filesystem.");
        this.owner = (S3AFileSystem)fs;
        this.instrumentation = this.owner.getInstrumentation().getS3GuardInstrumentation();
        String bucket = this.owner.getBucket();
        this.conf = this.owner.getConf();
        String confRegion = this.conf.getTrimmed("fs.s3a.s3guard.ddb.region");
        if (!StringUtils.isEmpty((String)confRegion)) {
            this.region = confRegion;
            LOG.debug("Overriding S3 region with configured DynamoDB region: {}", (Object)this.region);
        } else {
            this.region = this.owner.getBucketLocation();
            LOG.debug("Inferring DynamoDB region from S3 bucket: {}", (Object)this.region);
        }
        this.username = this.owner.getUsername();
        this.dynamoDB = DynamoDBMetadataStore.createDynamoDB(this.conf, this.region);
        this.tableName = this.conf.getTrimmed("fs.s3a.s3guard.ddb.table", bucket);
        this.initDataAccessRetries(this.conf);
        this.invoker = new Invoker(new S3ARetryPolicy(this.conf), this::retryEvent);
        this.initTable();
        this.instrumentation.initialized();
    }

    @Override
    public void initialize(Configuration config) throws IOException {
        this.conf = config;
        this.tableName = this.conf.getTrimmed("fs.s3a.s3guard.ddb.table");
        Preconditions.checkArgument((!StringUtils.isEmpty((String)this.tableName) ? 1 : 0) != 0, (Object)"No DynamoDB table name configured");
        this.region = this.conf.getTrimmed("fs.s3a.s3guard.ddb.region");
        Preconditions.checkArgument((!StringUtils.isEmpty((String)this.region) ? 1 : 0) != 0, (Object)"No DynamoDB region configured");
        this.dynamoDB = DynamoDBMetadataStore.createDynamoDB(this.conf, this.region);
        this.username = UserGroupInformation.getCurrentUser().getShortUserName();
        this.initDataAccessRetries(this.conf);
        this.initTable();
    }

    private void initDataAccessRetries(Configuration config) {
        int maxRetries = config.getInt("fs.s3a.s3guard.ddb.max.retries", 9);
        this.dataAccessRetryPolicy = RetryPolicies.exponentialBackoffRetry((int)maxRetries, (long)100L, (TimeUnit)TimeUnit.MILLISECONDS);
        this.dataAccess = new Invoker(this.dataAccessRetryPolicy, this::retryEvent);
    }

    @Override
    public void delete(Path path) throws IOException {
        this.innerDelete(path, true);
    }

    @Override
    public void forgetMetadata(Path path) throws IOException {
        this.innerDelete(path, false);
    }

    private void innerDelete(Path path, boolean tombstone) throws IOException {
        this.checkPath(path);
        LOG.debug("Deleting from table {} in region {}: {}", new Object[]{this.tableName, this.region, path});
        if (path.isRoot()) {
            LOG.debug("Skip deleting root directory as it does not exist in table");
            return;
        }
        boolean idempotent = true;
        if (tombstone) {
            Item item = PathMetadataDynamoDBTranslation.pathMetadataToItem(PathMetadata.tombstone(path));
            this.invoker.retry("Put tombstone", path.toString(), idempotent, () -> this.table.putItem(item));
        } else {
            PrimaryKey key = PathMetadataDynamoDBTranslation.pathToKey(path);
            this.invoker.retry("Delete key", path.toString(), idempotent, () -> this.table.deleteItem(key));
        }
    }

    @Override
    public void deleteSubtree(Path path) throws IOException {
        this.checkPath(path);
        LOG.debug("Deleting subtree from table {} in region {}: {}", new Object[]{this.tableName, this.region, path});
        PathMetadata meta = this.get(path);
        if (meta == null || meta.isDeleted()) {
            LOG.debug("Subtree path {} does not exist; this will be a no-op", (Object)path);
            return;
        }
        DescendantsIterator desc = new DescendantsIterator(this, meta);
        while (desc.hasNext()) {
            this.innerDelete(desc.next().getPath(), true);
        }
    }

    private Item getConsistentItem(PrimaryKey key) {
        GetItemSpec spec = new GetItemSpec().withPrimaryKey(key).withConsistentRead(true);
        return this.table.getItem(spec);
    }

    @Override
    public PathMetadata get(Path path) throws IOException {
        return this.get(path, false);
    }

    @Override
    public PathMetadata get(Path path, boolean wantEmptyDirectoryFlag) throws IOException {
        this.checkPath(path);
        LOG.debug("Get from table {} in region {}: {}", new Object[]{this.tableName, this.region, path});
        return Invoker.once("get", path.toString(), () -> this.innerGet(path, wantEmptyDirectoryFlag));
    }

    private PathMetadata innerGet(Path path, boolean wantEmptyDirectoryFlag) throws IOException {
        FileStatus status;
        PathMetadata meta;
        if (path.isRoot()) {
            meta = new PathMetadata(this.makeDirStatus(this.username, path));
        } else {
            Item item = this.getConsistentItem(PathMetadataDynamoDBTranslation.pathToKey(path));
            meta = PathMetadataDynamoDBTranslation.itemToPathMetadata(item, this.username);
            LOG.debug("Get from table {} in region {} returning for {}: {}", new Object[]{this.tableName, this.region, path, meta});
        }
        if (wantEmptyDirectoryFlag && meta != null && (status = meta.getFileStatus()).isDirectory()) {
            QuerySpec spec = new QuerySpec().withHashKey(PathMetadataDynamoDBTranslation.pathToParentKeyAttribute(path)).withConsistentRead(true).withFilterExpression("is_deleted = :false").withValueMap((Map)deleteTrackingValueMap);
            ItemCollection items = this.table.query(spec);
            boolean hasChildren = items.iterator().hasNext();
            meta.setIsEmptyDirectory(hasChildren ? Tristate.FALSE : Tristate.UNKNOWN);
        }
        return meta;
    }

    private FileStatus makeDirStatus(String owner, Path path) {
        return new FileStatus(0L, true, 1, 0L, 0L, 0L, null, owner, null, path);
    }

    @Override
    public DirListingMetadata listChildren(Path path) throws IOException {
        this.checkPath(path);
        LOG.debug("Listing table {} in region {}: {}", new Object[]{this.tableName, this.region, path});
        return Invoker.once("listChildren", path.toString(), () -> {
            QuerySpec spec = new QuerySpec().withHashKey(PathMetadataDynamoDBTranslation.pathToParentKeyAttribute(path)).withConsistentRead(true);
            ItemCollection items = this.table.query(spec);
            ArrayList<PathMetadata> metas = new ArrayList<PathMetadata>();
            for (Item item : items) {
                PathMetadata meta = PathMetadataDynamoDBTranslation.itemToPathMetadata(item, this.username);
                metas.add(meta);
            }
            LOG.trace("Listing table {} in region {} for {} returning {}", new Object[]{this.tableName, this.region, path, metas});
            return metas.isEmpty() && this.get(path) == null ? null : new DirListingMetadata(path, metas, false);
        });
    }

    Collection<PathMetadata> completeAncestry(Collection<PathMetadata> pathsToCreate) {
        HashMap<Path, PathMetadata> ancestry = new HashMap<Path, PathMetadata>();
        for (PathMetadata meta : pathsToCreate) {
            Preconditions.checkArgument((meta != null ? 1 : 0) != 0);
            Path path = meta.getFileStatus().getPath();
            if (path.isRoot()) break;
            ancestry.put(path, meta);
            Path parent = path.getParent();
            while (!parent.isRoot() && !ancestry.containsKey(parent)) {
                LOG.debug("auto-create ancestor path {} for child path {}", (Object)parent, (Object)path);
                FileStatus status = DynamoDBMetadataStore.makeDirStatus(parent, this.username);
                ancestry.put(parent, new PathMetadata(status, Tristate.FALSE, false));
                parent = parent.getParent();
            }
        }
        return ancestry.values();
    }

    @Override
    public void move(Collection<Path> pathsToDelete, Collection<PathMetadata> pathsToCreate) throws IOException {
        if (pathsToDelete == null && pathsToCreate == null) {
            return;
        }
        LOG.debug("Moving paths of table {} in region {}: {} paths to delete and {} paths to create", new Object[]{this.tableName, this.region, pathsToDelete == null ? 0 : pathsToDelete.size(), pathsToCreate == null ? 0 : pathsToCreate.size()});
        LOG.trace("move: pathsToDelete = {}, pathsToCreate = {}", pathsToDelete, pathsToCreate);
        ArrayList<PathMetadata> newItems = new ArrayList<PathMetadata>();
        if (pathsToCreate != null) {
            newItems.addAll(this.completeAncestry(pathsToCreate));
        }
        if (pathsToDelete != null) {
            for (Path meta : pathsToDelete) {
                newItems.add(PathMetadata.tombstone(meta));
            }
        }
        Invoker.once("move", this.tableName, () -> this.processBatchWriteRequest(null, PathMetadataDynamoDBTranslation.pathMetadataToItem(newItems)));
    }

    private void processBatchWriteRequest(PrimaryKey[] keysToDelete, Item[] itemsToPut) throws IOException {
        int totalToDelete = keysToDelete == null ? 0 : keysToDelete.length;
        int totalToPut = itemsToPut == null ? 0 : itemsToPut.length;
        int count = 0;
        while (count < totalToDelete + totalToPut) {
            TableWriteItems writeItems = new TableWriteItems(this.tableName);
            int numToDelete = 0;
            if (keysToDelete != null && count < totalToDelete) {
                numToDelete = Math.min(25, totalToDelete - count);
                writeItems.withPrimaryKeysToDelete(Arrays.copyOfRange(keysToDelete, count, count + numToDelete));
                count += numToDelete;
            }
            if (numToDelete < 25 && itemsToPut != null && count < totalToDelete + totalToPut) {
                int numToPut = Math.min(25 - numToDelete, totalToDelete + totalToPut - count);
                int index = count - totalToDelete;
                writeItems.withItemsToPut(Arrays.copyOfRange(itemsToPut, index, index + numToPut));
                count += numToPut;
            }
            BatchWriteItemOutcome res = this.dynamoDB.batchWriteItem(new TableWriteItems[]{writeItems});
            Map unprocessed = res.getUnprocessedItems();
            int retryCount = 0;
            while (!unprocessed.isEmpty()) {
                this.retryBackoff(retryCount++);
                res = this.dynamoDB.batchWriteItemUnprocessed(unprocessed);
                unprocessed = res.getUnprocessedItems();
            }
        }
    }

    private void retryBackoff(int retryCount) throws IOException {
        try {
            RetryPolicy.RetryAction action = this.dataAccessRetryPolicy.shouldRetry(null, retryCount, 0, true);
            if (action.action == RetryPolicy.RetryAction.RetryDecision.FAIL) {
                throw new IOException(String.format("Max retries exceeded (%d) for DynamoDB. This may be because write threshold of DynamoDB is set too low.", retryCount));
            }
            LOG.debug("Sleeping {} msec before next retry", (Object)action.delayMillis);
            Thread.sleep(action.delayMillis);
        }
        catch (InterruptedException e) {
            throw (IOException)new InterruptedIOException(e.toString()).initCause(e);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException("Unexpected exception", e);
        }
    }

    @Override
    public void put(PathMetadata meta) throws IOException {
        LOG.debug("Saving to table {} in region {}: {}", new Object[]{this.tableName, this.region, meta});
        ArrayList<PathMetadata> wrapper = new ArrayList<PathMetadata>(1);
        wrapper.add(meta);
        this.put(wrapper);
    }

    @Override
    public void put(Collection<PathMetadata> metas) throws IOException {
        Item[] items = PathMetadataDynamoDBTranslation.pathMetadataToItem(this.completeAncestry(metas));
        LOG.debug("Saving batch of {} items to table {}, region {}", new Object[]{items.length, this.tableName, this.region});
        this.processBatchWriteRequest(null, items);
    }

    private Collection<PathMetadata> fullPathsToPut(PathMetadata meta) throws IOException {
        Item item;
        DynamoDBMetadataStore.checkPathMetadata(meta);
        ArrayList<PathMetadata> metasToPut = new ArrayList<PathMetadata>();
        if (!meta.getFileStatus().getPath().isRoot()) {
            metasToPut.add(meta);
        }
        for (Path path = meta.getFileStatus().getPath().getParent(); path != null && !path.isRoot() && !this.itemExists(item = this.getConsistentItem(PathMetadataDynamoDBTranslation.pathToKey(path))); path = path.getParent()) {
            FileStatus status = DynamoDBMetadataStore.makeDirStatus(path, this.username);
            metasToPut.add(new PathMetadata(status, Tristate.FALSE, false));
        }
        return metasToPut;
    }

    private boolean itemExists(Item item) {
        if (item == null) {
            return false;
        }
        return !item.hasAttribute("is_deleted") || !item.getBoolean("is_deleted");
    }

    static FileStatus makeDirStatus(Path f, String owner) {
        return new FileStatus(0L, true, 1, 0L, System.currentTimeMillis(), 0L, null, owner, owner, f);
    }

    @Override
    public void put(DirListingMetadata meta) throws IOException {
        LOG.debug("Saving to table {} in region {}: {}", new Object[]{this.tableName, this.region, meta});
        Path path = meta.getPath();
        PathMetadata p = new PathMetadata(DynamoDBMetadataStore.makeDirStatus(path, this.username), meta.isEmpty(), false);
        Collection metasToPut = this.invoker.retry("paths to put", path.toString(), true, () -> this.fullPathsToPut(p));
        metasToPut.addAll(meta.getListing());
        Invoker.once("put", path.toString(), () -> this.processBatchWriteRequest(null, PathMetadataDynamoDBTranslation.pathMetadataToItem(metasToPut)));
    }

    @Override
    public synchronized void close() {
        if (this.instrumentation != null) {
            this.instrumentation.storeClosed();
        }
        if (this.dynamoDB != null) {
            LOG.debug("Shutting down {}", (Object)this);
            this.dynamoDB.shutdown();
            this.dynamoDB = null;
        }
    }

    @Override
    public void destroy() throws IOException {
        if (this.table == null) {
            LOG.info("In destroy(): no table to delete");
            return;
        }
        LOG.info("Deleting DynamoDB table {} in region {}", (Object)this.tableName, (Object)this.region);
        Preconditions.checkNotNull((Object)this.dynamoDB, (Object)"Not connected to DynamoDB");
        try {
            this.table.delete();
            this.table.waitForDelete();
        }
        catch (ResourceNotFoundException rnfe) {
            LOG.info("ResourceNotFoundException while deleting DynamoDB table {} in region {}.  This may indicate that the table does not exist, or has been deleted by another concurrent thread or process.", (Object)this.tableName, (Object)this.region);
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted while waiting for DynamoDB table {} being deleted", (Object)this.tableName, (Object)ie);
            throw new InterruptedIOException("Table " + this.tableName + " in region " + this.region + " has not been deleted");
        }
        catch (AmazonClientException e) {
            throw S3AUtils.translateException("destroy", this.tableName, (SdkBaseException)e);
        }
    }

    private ItemCollection<ScanOutcome> expiredFiles(long modTime, String keyPrefix) {
        String filterExpression = "mod_time < :mod_time and begins_with(parent, :parent)";
        String projectionExpression = "parent,child";
        ValueMap map = new ValueMap().withLong(":mod_time", modTime).withString(":parent", keyPrefix);
        return this.table.scan(filterExpression, projectionExpression, null, (Map)map);
    }

    @Override
    public void prune(long modTime) throws IOException {
        this.prune(modTime, "/");
    }

    @Override
    public void prune(long modTime, String keyPrefix) throws IOException {
        int itemCount = 0;
        try {
            ArrayList<Path> deletionBatch = new ArrayList<Path>(25);
            int delay = this.conf.getInt("fs.s3a.s3guard.ddb.background.sleep", 25);
            for (Item item : this.expiredFiles(modTime, keyPrefix)) {
                PathMetadata md = PathMetadataDynamoDBTranslation.itemToPathMetadata(item, this.username);
                Path path = md.getFileStatus().getPath();
                deletionBatch.add(path);
                ++itemCount;
                if (deletionBatch.size() != 25) continue;
                Thread.sleep(delay);
                this.processBatchWriteRequest(PathMetadataDynamoDBTranslation.pathToKey(deletionBatch), null);
                deletionBatch.clear();
            }
            if (deletionBatch.size() > 0) {
                Thread.sleep(delay);
                this.processBatchWriteRequest(PathMetadataDynamoDBTranslation.pathToKey(deletionBatch), null);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new InterruptedIOException("Pruning was interrupted");
        }
        LOG.info("Finished pruning {} items in batches of {}", (Object)itemCount, (Object)25);
    }

    public String toString() {
        return this.getClass().getSimpleName() + '{' + "region=" + this.region + ", tableName=" + this.tableName + '}';
    }

    @VisibleForTesting
    void initTable() throws IOException {
        block16: {
            this.table = this.dynamoDB.getTable(this.tableName);
            try {
                try {
                    String status;
                    LOG.debug("Binding to table {}", (Object)this.tableName);
                    TableDescription description = this.table.describe();
                    LOG.debug("Table state: {}", (Object)description);
                    switch (status = description.getTableStatus()) {
                        case "CREATING": 
                        case "UPDATING": {
                            LOG.debug("Table {} in region {} is being created/updated. This may indicate that the table is being operated by another concurrent thread or process. Waiting for active...", (Object)this.tableName, (Object)this.region);
                            this.waitForTableActive(this.table);
                            break;
                        }
                        case "DELETING": {
                            throw new FileNotFoundException("DynamoDB table '" + this.tableName + "' is being deleted in region " + this.region);
                        }
                        case "ACTIVE": {
                            break;
                        }
                        default: {
                            throw new IOException("Unknown DynamoDB table status " + status + ": tableName='" + this.tableName + "', region=" + this.region);
                        }
                    }
                    Item versionMarker = this.getVersionMarkerItem();
                    DynamoDBMetadataStore.verifyVersionCompatibility(this.tableName, versionMarker);
                    Long created = PathMetadataDynamoDBTranslation.extractCreationTimeFromMarker(versionMarker);
                    LOG.debug("Using existing DynamoDB table {} in region {} created {}", new Object[]{this.tableName, this.region, created != null ? new Date(created) : null});
                }
                catch (ResourceNotFoundException rnfe) {
                    if (this.conf.getBoolean("fs.s3a.s3guard.ddb.table.create", false)) {
                        ProvisionedThroughput capacity = new ProvisionedThroughput(Long.valueOf(this.conf.getLong("fs.s3a.s3guard.ddb.table.capacity.read", 500L)), Long.valueOf(this.conf.getLong("fs.s3a.s3guard.ddb.table.capacity.write", 100L)));
                        this.createTable(capacity);
                        break block16;
                    }
                    throw (FileNotFoundException)new FileNotFoundException("DynamoDB table '" + this.tableName + "' does not exist in region " + this.region + "; auto-creation is turned off").initCause(rnfe);
                }
            }
            catch (AmazonClientException e) {
                throw S3AUtils.translateException("initTable", this.tableName, (SdkBaseException)e);
            }
        }
    }

    private Item getVersionMarkerItem() throws IOException {
        PrimaryKey versionMarkerKey = PathMetadataDynamoDBTranslation.createVersionMarkerPrimaryKey(VERSION_MARKER);
        int retryCount = 0;
        Item versionMarker = this.table.getItem(versionMarkerKey);
        while (versionMarker == null) {
            try {
                RetryPolicy.RetryAction action = this.dataAccessRetryPolicy.shouldRetry(null, retryCount, 0, true);
                if (action.action == RetryPolicy.RetryAction.RetryDecision.FAIL) break;
                LOG.debug("Sleeping {} ms before next retry", (Object)action.delayMillis);
                Thread.sleep(action.delayMillis);
            }
            catch (Exception e) {
                throw new IOException("initTable: Unexpected exception", e);
            }
            ++retryCount;
            versionMarker = this.table.getItem(versionMarkerKey);
        }
        return versionMarker;
    }

    @VisibleForTesting
    static void verifyVersionCompatibility(String tableName, Item versionMarker) throws IOException {
        if (versionMarker == null) {
            LOG.warn("Table {} contains no version marker", (Object)tableName);
            throw new IOException("S3Guard table lacks version marker. Table: " + tableName);
        }
        int version = PathMetadataDynamoDBTranslation.extractVersionFromMarker(versionMarker);
        if (100 != version) {
            throw new IOException("Database table is from an incompatible S3Guard version. Table " + tableName + " Expected version " + 100 + " actual " + version);
        }
    }

    private void waitForTableActive(Table t) throws InterruptedIOException {
        try {
            t.waitForActive();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while waiting for table {} in region {} active", new Object[]{this.tableName, this.region, e});
            Thread.currentThread().interrupt();
            throw (InterruptedIOException)new InterruptedIOException("DynamoDB table '" + this.tableName + "' is not active yet in region " + this.region).initCause(e);
        }
    }

    private void createTable(ProvisionedThroughput capacity) throws IOException {
        try {
            LOG.info("Creating non-existent DynamoDB table {} in region {}", (Object)this.tableName, (Object)this.region);
            this.table = this.dynamoDB.createTable(new CreateTableRequest().withTableName(this.tableName).withKeySchema(PathMetadataDynamoDBTranslation.keySchema()).withAttributeDefinitions(PathMetadataDynamoDBTranslation.attributeDefinitions()).withProvisionedThroughput(capacity));
            LOG.debug("Awaiting table becoming active");
        }
        catch (ResourceInUseException e) {
            LOG.warn("ResourceInUseException while creating DynamoDB table {} in region {}.  This may indicate that the table was created by another concurrent thread or process.", (Object)this.tableName, (Object)this.region);
        }
        this.waitForTableActive(this.table);
        Item marker = PathMetadataDynamoDBTranslation.createVersionMarker(VERSION_MARKER, 100, System.currentTimeMillis());
        this.putItem(marker);
    }

    PutItemOutcome putItem(Item item) {
        LOG.debug("Putting item {}", (Object)item);
        return this.table.putItem(item);
    }

    void provisionTable(Long readCapacity, Long writeCapacity) throws IOException {
        ProvisionedThroughput toProvision = new ProvisionedThroughput().withReadCapacityUnits(readCapacity).withWriteCapacityUnits(writeCapacity);
        this.invoker.retry("ProvisionTable", this.tableName, true, () -> {
            ProvisionedThroughputDescription p = this.table.updateTable(toProvision).getProvisionedThroughput();
            LOG.info("Provision table {} in region {}: readCapacityUnits={}, writeCapacityUnits={}", new Object[]{this.tableName, this.region, p.getReadCapacityUnits(), p.getWriteCapacityUnits()});
        });
    }

    @VisibleForTesting
    void provisionTableBlocking(Long readCapacity, Long writeCapacity) throws IOException {
        this.provisionTable(readCapacity, writeCapacity);
        this.waitForTableActive(this.table);
    }

    @VisibleForTesting
    Table getTable() {
        return this.table;
    }

    String getRegion() {
        return this.region;
    }

    @VisibleForTesting
    DynamoDB getDynamoDB() {
        return this.dynamoDB;
    }

    private Path checkPath(Path path) {
        Preconditions.checkNotNull((Object)path);
        Preconditions.checkArgument((boolean)path.isAbsolute(), (String)"Path %s is not absolute", (Object[])new Object[]{path});
        URI uri = path.toUri();
        Preconditions.checkNotNull((Object)uri.getScheme(), (String)"Path %s missing scheme", (Object[])new Object[]{path});
        Preconditions.checkArgument((boolean)uri.getScheme().equals("s3a"), (String)"Path %s scheme must be %s", (Object[])new Object[]{path, "s3a"});
        Preconditions.checkArgument((!StringUtils.isEmpty((String)uri.getHost()) ? 1 : 0) != 0, (String)"Path %s is missing bucket.", (Object[])new Object[]{path});
        return path;
    }

    private static void checkPathMetadata(PathMetadata meta) {
        Preconditions.checkNotNull((Object)meta);
        Preconditions.checkNotNull((Object)meta.getFileStatus());
        Preconditions.checkNotNull((Object)meta.getFileStatus().getPath());
    }

    @Override
    public Map<String, String> getDiagnostics() throws IOException {
        TreeMap<String, String> map = new TreeMap<String, String>();
        if (this.table != null) {
            TableDescription desc = this.getTableDescription(true);
            map.put("name", desc.getTableName());
            map.put(STATUS, desc.getTableStatus());
            map.put("ARN", desc.getTableArn());
            map.put("size", desc.getTableSizeBytes().toString());
            map.put(TABLE, desc.toString());
            ProvisionedThroughputDescription throughput = desc.getProvisionedThroughput();
            map.put(READ_CAPACITY, throughput.getReadCapacityUnits().toString());
            map.put(WRITE_CAPACITY, throughput.getWriteCapacityUnits().toString());
            map.put(TABLE, desc.toString());
            map.put("persist.authoritative.bit", Boolean.toString(false));
        } else {
            map.put("name", "DynamoDB Metadata Store");
            map.put(TABLE, "none");
            map.put(STATUS, "undefined");
        }
        map.put("description", DESCRIPTION);
        map.put("region", this.region);
        if (this.dataAccessRetryPolicy != null) {
            map.put("retryPolicy", this.dataAccessRetryPolicy.toString());
        }
        return map;
    }

    private TableDescription getTableDescription(boolean forceUpdate) {
        TableDescription desc = this.table.getDescription();
        if (desc == null || forceUpdate) {
            desc = this.table.describe();
        }
        return desc;
    }

    @Override
    public void updateParameters(Map<String, String> parameters) throws IOException {
        Preconditions.checkNotNull((Object)this.table, (Object)"Not initialized");
        TableDescription desc = this.getTableDescription(true);
        ProvisionedThroughputDescription current = desc.getProvisionedThroughput();
        long currentRead = current.getReadCapacityUnits();
        long newRead = this.getLongParam(parameters, "fs.s3a.s3guard.ddb.table.capacity.read", currentRead);
        long currentWrite = current.getWriteCapacityUnits();
        long newWrite = this.getLongParam(parameters, "fs.s3a.s3guard.ddb.table.capacity.write", currentWrite);
        if (newRead != currentRead || newWrite != currentWrite) {
            LOG.info("Current table capacity is read: {}, write: {}", (Object)currentRead, (Object)currentWrite);
            LOG.info("Changing capacity of table to read: {}, write: {}", (Object)newRead, (Object)newWrite);
            this.provisionTableBlocking(newRead, newWrite);
        } else {
            LOG.info("Table capacity unchanged at read: {}, write: {}", (Object)newRead, (Object)newWrite);
        }
    }

    private long getLongParam(Map<String, String> parameters, String key, long defVal) {
        String k = parameters.get(key);
        if (k != null) {
            return Long.parseLong(k);
        }
        return defVal;
    }

    void retryEvent(String text, IOException ex, int attempts, boolean idempotent) {
        if (S3AUtils.isThrottleException(ex)) {
            if (this.instrumentation != null) {
                this.instrumentation.throttled();
            }
            int eventCount = this.throttleEventCount.addAndGet(1);
            if (attempts == 1 && eventCount < 100) {
                LOG.warn("DynamoDB IO limits reached in {}; consider increasing capacity: {}", (Object)text, (Object)ex.toString());
                LOG.debug("Throttled", (Throwable)ex);
            } else {
                LOG.debug("DynamoDB IO limits reached in {}; consider increasing capacity: {}", (Object)text, (Object)ex.toString());
            }
        } else if (attempts == 1) {
            LOG.info("Retrying {}: {}", (Object)text, (Object)ex.toString());
            LOG.debug("Retrying {}", (Object)text, (Object)ex);
        }
        if (this.instrumentation != null) {
            this.instrumentation.retrying();
        }
        if (this.owner != null) {
            this.owner.metastoreOperationRetried(ex, attempts, idempotent);
        }
    }
}

