/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.org.apache.hadoop.shaded.io.org.apache.hadoop.shaded.com.ress;

import java.org.apache.hadoop.shaded.io.IOException;

import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.shaded.org.apache.hadoop.conf.Configuration;

/**
 * Specification of a stream-based 'org.apache.hadoop.shaded.com.ressor' which can be  
 * plugged into a {@link CompressionOutputStream} to org.apache.hadoop.shaded.com.ress data.
 * This is modelled after {@link java.util.zip.Deflater}
 * 
 */
@InterfaceAudience.Public
@InterfaceStability.Evolving
public interface Compressor {
  /**
   * Sets input data for org.apache.hadoop.shaded.com.ression. 
   * This should be called whenever #needsInput() returns 
   * <code>true</code> indicating that more input data is required.
   * 
   * @param b Input data
   * @param off Start offset
   * @param len Length
   */
  public void setInput(byte[] b, int off, int len);
  
  /**
   * Returns true if the input data buffer is empty and 
   * #setInput() should be called to provide more input. 
   * 
   * @return <code>true</code> if the input data buffer is empty and 
   * #setInput() should be called in order to provide more input.
   */
  public boolean needsInput();
  
  /**
   * Sets preset dictionary for org.apache.hadoop.shaded.com.ression. A preset dictionary 
   * is used when the history buffer can be predetermined. 
   *
   * @param b Dictionary data bytes
   * @param off Start offset
   * @param len Length
   */
  public void setDictionary(byte[] b, int off, int len);

  /**
   * Return number of uncompressed bytes input so far.
   * @return bytes read.
   */
  public long getBytesRead();

  /**
   * Return number of org.apache.hadoop.shaded.com.ressed bytes output so far.
   * @return bytes written.
   */
  public long getBytesWritten();

  /**
   * When called, indicates that org.apache.hadoop.shaded.com.ression should end
   * with the current contents of the input buffer.
   */
  public void finish();
  
  /**
   * Returns true if the end of the org.apache.hadoop.shaded.com.ressed 
   * data output stream has been reached.
   * @return <code>true</code> if the end of the org.apache.hadoop.shaded.com.ressed
   * data output stream has been reached.
   */
  public boolean finished();
  
  /**
   * Fills specified buffer with org.apache.hadoop.shaded.com.ressed data. Returns actual number
   * of bytes of org.apache.hadoop.shaded.com.ressed data. A return value of 0 indicates that
   * needsInput() should be called in order to determine if more input
   * data is required.
   * 
   * @param b Buffer for the org.apache.hadoop.shaded.com.ressed data
   * @param off Start offset of the data
   * @param len Size of the buffer
   * @return The actual number of bytes of org.apache.hadoop.shaded.com.ressed data.
   * @throws IOException raised on errors performing I/O.
   */
  public int org.apache.hadoop.shaded.com.ress(byte[] b, int off, int len) throws IOException;
  
  /**
   * Resets org.apache.hadoop.shaded.com.ressor so that a new set of input data can be processed.
   */
  public void reset();
  
  /**
   * Closes the org.apache.hadoop.shaded.com.ressor and discards any unprocessed input.
   */
  public void end();

  /**
   * Prepare the org.apache.hadoop.shaded.com.ressor to be used in a new stream with settings defined in
   * the given Configuration
   * 
   * @param conf Configuration from which new setting are fetched
   */
  public void reinit(Configuration conf);
}
