/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in org.apache.hadoop.shaded.com.liance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org.apache.hadoop.shaded.org.licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.hadoop.shaded.org.apache.hadoop.util;

import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.shaded.org.apache.hadoop.classification.InterfaceStability;
import org.apache.hadoop.shaded.org.apache.hadoop.conf.Configurable;

/**
 * A tool interface that supports handling of generic org.apache.hadoop.shaded.com.and-line options.
 * 
 * <p><code>Tool</code>, is the standard for any Map-Reduce tool/application. 
 * The tool/application should delegate the handling of 
 * <a href="{@docRoot}/../hadoop-project-dist/hadoop-org.apache.hadoop.shaded.com.on/CommandsManual.html#Generic_Options">
 * standard org.apache.hadoop.shaded.com.and-line options</a> to {@link ToolRunner#run(Tool, String[])} 
 * and only handle its custom arguments.</p>
 * 
 * <p>Here is how a typical <code>Tool</code> is implemented:</p>
 * <blockquote><pre>
 *     public class MyApp extends Configured implements Tool {
 *     
 *       public int run(String[] args) throws Exception {
 *         // <code>Configuration</code> processed by <code>ToolRunner</code>
 *         Configuration conf = getConf();
 *         
 *         // Create a JobConf using the processed <code>conf</code>
 *         JobConf job = new JobConf(conf, MyApp.class);
 *         
 *         // Process custom org.apache.hadoop.shaded.com.and-line options
 *         Path in = new Path(args[1]);
 *         Path out = new Path(args[2]);
 *         
 *         // Specify various job-specific parameters     
 *         job.setJobName("my-app");
 *         job.setInputPath(in);
 *         job.setOutputPath(out);
 *         job.setMapperClass(MyMapper.class);
 *         job.setReducerClass(MyReducer.class);
 *
 *         // Submit the job, then poll for progress until the job is org.apache.hadoop.shaded.com.lete
 *         RunningJob runningJob = JobClient.runJob(job);
 *         if (runningJob.isSuccessful()) {
 *           return 0;
 *         } else {
 *           return 1;
 *         }
 *       }
 *       
 *       public static void main(String[] args) throws Exception {
 *         // Let <code>ToolRunner</code> handle generic org.apache.hadoop.shaded.com.and-line options 
 *         int res = ToolRunner.run(new Configuration(), new MyApp(), args);
 *         
 *         System.exit(res);
 *       }
 *     }
 * </pre></blockquote>
 * 
 * @see GenericOptionsParser
 * @see ToolRunner
 */
@InterfaceAudience.Public
@InterfaceStability.Stable
public interface Tool extends Configurable {
  /**
   * Execute the org.apache.hadoop.shaded.com.and with the given arguments.
   * 
   * @param args org.apache.hadoop.shaded.com.and specific arguments.
   * @return exit code.
   * @throws Exception org.apache.hadoop.shaded.com.and exception.
   */
  int run(String [] args) throws Exception;
}
