/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.security;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.Charsets;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.IdMappingServiceProvider;
import org.apache.hadoop.util.Time;

public class ShellBasedIdMapping
implements IdMappingServiceProvider {
    private static final Log LOG = LogFactory.getLog(ShellBasedIdMapping.class);
    private static final String OS = System.getProperty("os.name");
    static final String GET_ALL_USERS_CMD = "getent passwd | cut -d: -f1,3";
    static final String GET_ALL_GROUPS_CMD = "getent group | cut -d: -f1,3";
    static final String MAC_GET_ALL_USERS_CMD = "dscl . -list /Users UniqueID";
    static final String MAC_GET_ALL_GROUPS_CMD = "dscl . -list /Groups PrimaryGroupID";
    private final File staticMappingFile;
    private StaticMapping staticMapping = null;
    private long lastModificationTimeStaticMap = 0L;
    private boolean constructFullMapAtInit = false;
    private static final Pattern EMPTY_LINE = Pattern.compile("^\\s*$");
    private static final Pattern COMMENT_LINE = Pattern.compile("^\\s*#.*$");
    private static final Pattern MAPPING_LINE = Pattern.compile("^(uid|gid)\\s+(\\d+)\\s+(\\d+)\\s*(#.*)?$");
    private final long timeout;
    private BiMap<Integer, String> uidNameMap = HashBiMap.create();
    private BiMap<Integer, String> gidNameMap = HashBiMap.create();
    private long lastUpdateTime = 0L;
    private static final String DUPLICATE_NAME_ID_DEBUG_INFO = "NFS gateway could have problem starting with duplicate name or id on the host system.\nThis is because HDFS (non-kerberos cluster) uses name as the only way to identify a user or group.\nThe host system with duplicated user/group name or id might work fine most of the time by itself.\nHowever when NFS gateway talks to HDFS, HDFS accepts only user and group name.\nTherefore, same name means the same user or same group. To find the duplicated names/ids, one can do:\n<getent passwd | cut -d: -f1,3> and <getent group | cut -d: -f1,3> on Linux systems,\n<dscl . -list /Users UniqueID> and <dscl . -list /Groups PrimaryGroupID> on MacOS.";

    @VisibleForTesting
    public ShellBasedIdMapping(Configuration conf, boolean constructFullMapAtInit) throws IOException {
        this.constructFullMapAtInit = constructFullMapAtInit;
        long updateTime = conf.getLong("usergroupid.update.millis", 900000L);
        if (updateTime < 60000L) {
            LOG.info((Object)"User configured user account update time is less than 1 minute. Use 1 minute instead.");
            this.timeout = 60000L;
        } else {
            this.timeout = updateTime;
        }
        String staticFilePath = conf.get("static.id.mapping.file", "/etc/nfs.map");
        this.staticMappingFile = new File(staticFilePath);
        this.updateStaticMapping();
        this.updateMaps();
    }

    public ShellBasedIdMapping(Configuration conf) throws IOException {
        this(conf, false);
    }

    @VisibleForTesting
    public long getTimeout() {
        return this.timeout;
    }

    @VisibleForTesting
    public BiMap<Integer, String> getUidNameMap() {
        return this.uidNameMap;
    }

    @VisibleForTesting
    public BiMap<Integer, String> getGidNameMap() {
        return this.gidNameMap;
    }

    @VisibleForTesting
    public synchronized void clearNameMaps() {
        this.uidNameMap.clear();
        this.gidNameMap.clear();
        this.lastUpdateTime = Time.monotonicNow();
    }

    private synchronized boolean isExpired() {
        return Time.monotonicNow() - this.lastUpdateTime > this.timeout;
    }

    private void checkAndUpdateMaps() {
        if (this.isExpired()) {
            LOG.info((Object)"Update cache now");
            try {
                this.updateMaps();
            }
            catch (IOException e) {
                LOG.error((Object)"Can't update the maps. Will use the old ones, which can potentially cause problem.", (Throwable)e);
            }
        }
    }

    private static void reportDuplicateEntry(String header, Integer key, String value, Integer ekey, String evalue) {
        LOG.warn((Object)("\n" + header + String.format("new entry (%d, %s), existing entry: (%d, %s).%n%s%n%s", key, value, ekey, evalue, "The new entry is to be ignored for the following reason.", DUPLICATE_NAME_ID_DEBUG_INFO)));
    }

    private static Integer parseId(String idStr) {
        Long longVal = Long.parseLong(idStr);
        int intVal = longVal.intValue();
        return intVal;
    }

    @VisibleForTesting
    public static boolean updateMapInternal(BiMap<Integer, String> map, String mapName, String command, String regex, Map<Integer, Integer> staticMapping) throws IOException {
        boolean updated = false;
        BufferedReader br = null;
        try {
            Process process = Runtime.getRuntime().exec(new String[]{"bash", "-c", command});
            br = new BufferedReader(new InputStreamReader(process.getInputStream(), Charset.defaultCharset()));
            String line = null;
            while ((line = br.readLine()) != null) {
                String[] nameId = line.split(regex);
                if (nameId == null || nameId.length != 2) {
                    throw new IOException("Can't parse " + mapName + " list entry:" + line);
                }
                LOG.debug((Object)("add to " + mapName + "map:" + nameId[0] + " id:" + nameId[1]));
                Integer key = staticMapping.get(ShellBasedIdMapping.parseId(nameId[1]));
                String value = nameId[0];
                if (map.containsKey((Object)key)) {
                    String prevValue = (String)map.get((Object)key);
                    if (value.equals(prevValue)) continue;
                    ShellBasedIdMapping.reportDuplicateEntry("Got multiple names associated with the same id: ", key, value, key, prevValue);
                    continue;
                }
                if (map.containsValue((Object)value)) {
                    Integer prevKey = (Integer)map.inverse().get((Object)value);
                    ShellBasedIdMapping.reportDuplicateEntry("Got multiple ids associated with the same name: ", key, value, prevKey, value);
                    continue;
                }
                map.put((Object)key, (Object)value);
                updated = true;
            }
            LOG.debug((Object)("Updated " + mapName + " map size: " + map.size()));
        }
        catch (IOException e) {
            LOG.error((Object)("Can't update " + mapName + " map"));
            throw e;
        }
        finally {
            if (br != null) {
                try {
                    br.close();
                }
                catch (IOException e1) {
                    LOG.error((Object)"Can't close BufferedReader of command result", (Throwable)e1);
                }
            }
        }
        return updated;
    }

    private boolean checkSupportedPlatform() {
        if (!OS.startsWith("Linux") && !OS.startsWith("Mac")) {
            LOG.error((Object)("Platform is not supported:" + OS + ". Can't update user map and group map and" + " 'nobody' will be used for any user and group."));
            return false;
        }
        return true;
    }

    private static boolean isInteger(String s) {
        try {
            Integer.parseInt(s);
        }
        catch (NumberFormatException e) {
            return false;
        }
        return true;
    }

    private synchronized void updateStaticMapping() throws IOException {
        boolean init;
        boolean bl = init = this.staticMapping == null;
        if (this.staticMappingFile.exists()) {
            long lmTime = this.staticMappingFile.lastModified();
            if (lmTime != this.lastModificationTimeStaticMap) {
                LOG.info((Object)(init ? "Using " : "Reloading '" + this.staticMappingFile + "' for static UID/GID mapping..."));
                this.lastModificationTimeStaticMap = lmTime;
                this.staticMapping = ShellBasedIdMapping.parseStaticMap(this.staticMappingFile);
            }
        } else {
            if (init) {
                this.staticMapping = new StaticMapping(new HashMap<Integer, Integer>(), new HashMap<Integer, Integer>());
            }
            if (this.lastModificationTimeStaticMap != 0L || init) {
                LOG.info((Object)("Not doing static UID/GID mapping because '" + this.staticMappingFile + "' does not exist."));
            }
            this.lastModificationTimeStaticMap = 0L;
            this.staticMapping.clear();
        }
    }

    public synchronized void updateMaps() throws IOException {
        if (!this.checkSupportedPlatform()) {
            return;
        }
        if (this.constructFullMapAtInit) {
            this.loadFullMaps();
            this.constructFullMapAtInit = false;
        } else {
            this.updateStaticMapping();
            this.clearNameMaps();
        }
    }

    private synchronized void loadFullUserMap() throws IOException {
        HashBiMap uMap = HashBiMap.create();
        if (OS.startsWith("Mac")) {
            ShellBasedIdMapping.updateMapInternal((BiMap<Integer, String>)uMap, "user", MAC_GET_ALL_USERS_CMD, "\\s+", this.staticMapping.uidMapping);
        } else {
            ShellBasedIdMapping.updateMapInternal((BiMap<Integer, String>)uMap, "user", GET_ALL_USERS_CMD, ":", this.staticMapping.uidMapping);
        }
        this.uidNameMap = uMap;
        this.lastUpdateTime = Time.monotonicNow();
    }

    private synchronized void loadFullGroupMap() throws IOException {
        HashBiMap gMap = HashBiMap.create();
        if (OS.startsWith("Mac")) {
            ShellBasedIdMapping.updateMapInternal((BiMap<Integer, String>)gMap, "group", MAC_GET_ALL_GROUPS_CMD, "\\s+", this.staticMapping.gidMapping);
        } else {
            ShellBasedIdMapping.updateMapInternal((BiMap<Integer, String>)gMap, "group", GET_ALL_GROUPS_CMD, ":", this.staticMapping.gidMapping);
        }
        this.gidNameMap = gMap;
        this.lastUpdateTime = Time.monotonicNow();
    }

    private synchronized void loadFullMaps() throws IOException {
        this.loadFullUserMap();
        this.loadFullGroupMap();
    }

    private String getName2IdCmdLinux(String name, boolean isGrp) {
        String cmd = isGrp ? "getent group " + name + " | cut -d: -f1,3" : "id -u " + name + " | awk '{print \"" + name + ":\"$1 }'";
        return cmd;
    }

    private String getId2NameCmdLinux(int id, boolean isGrp) {
        String cmd = "getent ";
        cmd = cmd + (isGrp ? "group " : "passwd ");
        cmd = cmd + String.valueOf(id) + " | cut -d: -f1,3";
        return cmd;
    }

    private String getName2IdCmdMac(String name, boolean isGrp) {
        String cmd;
        if (isGrp) {
            cmd = "dscl . -read /Groups/" + name;
            cmd = cmd + " | grep PrimaryGroupID | awk '($1 == \"PrimaryGroupID:\") ";
            cmd = cmd + "{ print \"" + name + "  \" $2 }'";
        } else {
            cmd = "id -u " + name + " | awk '{print \"" + name + "  \"$1 }'";
        }
        return cmd;
    }

    private String getId2NameCmdMac(int id, boolean isGrp) {
        String cmd = "dscl . -search /";
        cmd = cmd + (isGrp ? "Groups PrimaryGroupID " : "Users UniqueID ");
        cmd = cmd + String.valueOf(id);
        cmd = cmd + " | sed 'N;s/\\n//g;N;s/\\n//g' | sed 's/";
        cmd = cmd + (isGrp ? "PrimaryGroupID" : "UniqueID");
        cmd = cmd + " = (//g' | sed 's/)//g' | sed 's/\\\"//g'";
        return cmd;
    }

    private synchronized void updateMapIncr(String name, boolean isGrp) throws IOException {
        if (!this.checkSupportedPlatform()) {
            return;
        }
        if (ShellBasedIdMapping.isInteger(name) && isGrp) {
            this.loadFullGroupMap();
            return;
        }
        boolean updated = false;
        this.updateStaticMapping();
        updated = OS.startsWith("Linux") ? (isGrp ? ShellBasedIdMapping.updateMapInternal(this.gidNameMap, "group", this.getName2IdCmdLinux(name, true), ":", this.staticMapping.gidMapping) : ShellBasedIdMapping.updateMapInternal(this.uidNameMap, "user", this.getName2IdCmdLinux(name, false), ":", this.staticMapping.uidMapping)) : (isGrp ? ShellBasedIdMapping.updateMapInternal(this.gidNameMap, "group", this.getName2IdCmdMac(name, true), "\\s+", this.staticMapping.gidMapping) : ShellBasedIdMapping.updateMapInternal(this.uidNameMap, "user", this.getName2IdCmdMac(name, false), "\\s+", this.staticMapping.uidMapping));
        if (updated) {
            this.lastUpdateTime = Time.monotonicNow();
        }
    }

    private synchronized void updateMapIncr(int id, boolean isGrp) throws IOException {
        if (!this.checkSupportedPlatform()) {
            return;
        }
        boolean updated = false;
        this.updateStaticMapping();
        updated = OS.startsWith("Linux") ? (isGrp ? ShellBasedIdMapping.updateMapInternal(this.gidNameMap, "group", this.getId2NameCmdLinux(id, true), ":", this.staticMapping.gidMapping) : ShellBasedIdMapping.updateMapInternal(this.uidNameMap, "user", this.getId2NameCmdLinux(id, false), ":", this.staticMapping.uidMapping)) : (isGrp ? ShellBasedIdMapping.updateMapInternal(this.gidNameMap, "group", this.getId2NameCmdMac(id, true), "\\s+", this.staticMapping.gidMapping) : ShellBasedIdMapping.updateMapInternal(this.uidNameMap, "user", this.getId2NameCmdMac(id, false), "\\s+", this.staticMapping.uidMapping));
        if (updated) {
            this.lastUpdateTime = Time.monotonicNow();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static StaticMapping parseStaticMap(File staticMapFile) throws IOException {
        HashMap<Integer, Integer> uidMapping = new HashMap<Integer, Integer>();
        HashMap<Integer, Integer> gidMapping = new HashMap<Integer, Integer>();
        try (BufferedReader in = new BufferedReader(new InputStreamReader((InputStream)new FileInputStream(staticMapFile), Charsets.UTF_8));){
            String line = null;
            while ((line = in.readLine()) != null) {
                if (EMPTY_LINE.matcher(line).matches() || COMMENT_LINE.matcher(line).matches()) continue;
                Matcher lineMatcher = MAPPING_LINE.matcher(line);
                if (!lineMatcher.matches()) {
                    LOG.warn((Object)("Could not parse line '" + line + "'. Lines should be of " + "the form '[uid|gid] [remote id] [local id]'. Blank lines and " + "everything following a '#' on a line will be ignored."));
                    continue;
                }
                String firstComponent = lineMatcher.group(1);
                int remoteId = ShellBasedIdMapping.parseId(lineMatcher.group(2));
                int localId = ShellBasedIdMapping.parseId(lineMatcher.group(3));
                if (firstComponent.equals("uid")) {
                    uidMapping.put(localId, remoteId);
                    continue;
                }
                gidMapping.put(localId, remoteId);
            }
        }
        return new StaticMapping(uidMapping, gidMapping);
    }

    @Override
    public synchronized int getUid(String user) throws IOException {
        this.checkAndUpdateMaps();
        Integer id = (Integer)this.uidNameMap.inverse().get((Object)user);
        if (id == null) {
            this.updateMapIncr(user, false);
            id = (Integer)this.uidNameMap.inverse().get((Object)user);
            if (id == null) {
                throw new IOException("User just deleted?:" + user);
            }
        }
        return id;
    }

    @Override
    public synchronized int getGid(String group) throws IOException {
        this.checkAndUpdateMaps();
        Integer id = (Integer)this.gidNameMap.inverse().get((Object)group);
        if (id == null) {
            this.updateMapIncr(group, true);
            id = (Integer)this.gidNameMap.inverse().get((Object)group);
            if (id == null) {
                throw new IOException("No such group:" + group);
            }
        }
        return id;
    }

    @Override
    public synchronized String getUserName(int uid, String unknown) {
        this.checkAndUpdateMaps();
        String uname = (String)this.uidNameMap.get((Object)uid);
        if (uname == null) {
            try {
                this.updateMapIncr(uid, false);
            }
            catch (Exception e) {
                // empty catch block
            }
            uname = (String)this.uidNameMap.get((Object)uid);
            if (uname == null) {
                LOG.warn((Object)("Can't find user name for uid " + uid + ". Use default user name " + unknown));
                uname = unknown;
            }
        }
        return uname;
    }

    @Override
    public synchronized String getGroupName(int gid, String unknown) {
        this.checkAndUpdateMaps();
        String gname = (String)this.gidNameMap.get((Object)gid);
        if (gname == null) {
            try {
                this.updateMapIncr(gid, true);
            }
            catch (Exception e) {
                // empty catch block
            }
            gname = (String)this.gidNameMap.get((Object)gid);
            if (gname == null) {
                LOG.warn((Object)("Can't find group name for gid " + gid + ". Use default group name " + unknown));
                gname = unknown;
            }
        }
        return gname;
    }

    @Override
    public int getUidAllowingUnknown(String user) {
        int uid;
        this.checkAndUpdateMaps();
        try {
            uid = this.getUid(user);
        }
        catch (IOException e) {
            uid = user.hashCode();
            LOG.info((Object)("Can't map user " + user + ". Use its string hashcode:" + uid));
        }
        return uid;
    }

    @Override
    public int getGidAllowingUnknown(String group) {
        int gid;
        this.checkAndUpdateMaps();
        try {
            gid = this.getGid(group);
        }
        catch (IOException e) {
            gid = group.hashCode();
            LOG.info((Object)("Can't map group " + group + ". Use its string hashcode:" + gid));
        }
        return gid;
    }

    @VisibleForTesting
    static final class StaticMapping {
        final Map<Integer, Integer> uidMapping;
        final Map<Integer, Integer> gidMapping;

        public StaticMapping(Map<Integer, Integer> uidMapping, Map<Integer, Integer> gidMapping) {
            this.uidMapping = new PassThroughMap<Integer>(uidMapping);
            this.gidMapping = new PassThroughMap<Integer>(gidMapping);
        }

        public void clear() {
            this.uidMapping.clear();
            this.gidMapping.clear();
        }

        public boolean isNonEmpty() {
            return this.uidMapping.size() > 0 || this.gidMapping.size() > 0;
        }
    }

    static final class PassThroughMap<K>
    extends HashMap<K, K> {
        public PassThroughMap() {
            this(new HashMap());
        }

        public PassThroughMap(Map<K, K> mapping) {
            for (Map.Entry<K, K> entry : mapping.entrySet()) {
                super.put(entry.getKey(), entry.getValue());
            }
        }

        @Override
        public K get(Object key) {
            if (super.containsKey(key)) {
                return (K)super.get(key);
            }
            return (K)key;
        }
    }
}

