/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Supplier;
import java.io.IOException;
import java.util.List;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSNNTopology;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.balancer.NameNodeConnector;
import org.apache.hadoop.hdfs.server.common.HdfsServerConstants;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.INode;
import org.apache.hadoop.hdfs.server.namenode.INodeAttributes;
import org.apache.hadoop.hdfs.server.namenode.Namesystem;
import org.apache.hadoop.hdfs.server.namenode.SecondaryNameNode;
import org.apache.hadoop.hdfs.server.namenode.XAttrFeature;
import org.apache.hadoop.hdfs.server.namenode.XAttrStorage;
import org.apache.hadoop.hdfs.server.namenode.ha.HATestUtil;
import org.apache.hadoop.hdfs.server.namenode.sps.Context;
import org.apache.hadoop.hdfs.server.namenode.sps.SPSService;
import org.apache.hadoop.hdfs.server.namenode.sps.StoragePolicySatisfier;
import org.apache.hadoop.hdfs.server.sps.ExternalSPSContext;
import org.apache.hadoop.test.GenericTestUtils;
import org.junit.Assert;
import org.junit.Test;

public class TestPersistentStoragePolicySatisfier {
    private static Configuration conf;
    private static MiniDFSCluster cluster;
    private static DistributedFileSystem fs;
    private NameNodeConnector nnc;
    private StoragePolicySatisfier sps;
    private ExternalSPSContext ctxt;
    private static Path testFile;
    private static String testFileName;
    private static Path parentDir;
    private static Path parentFile;
    private static String parentFileName;
    private static Path childDir;
    private static Path childFile;
    private static String childFileName;
    private static final String COLD = "COLD";
    private static final String WARM = "WARM";
    private static final String ONE_SSD = "ONE_SSD";
    private static StorageType[][] storageTypes;
    private final int timeout = 90000;

    public void clusterSetUp() throws Exception {
        this.clusterSetUp(false, (Configuration)new HdfsConfiguration());
    }

    public void clusterSetUp(Configuration hdfsConf) throws Exception {
        this.clusterSetUp(false, hdfsConf);
    }

    private void clusterSetUp(boolean isHAEnabled, Configuration newConf) throws Exception {
        conf = newConf;
        conf.set("dfs.storage.policy.satisfier.recheck.timeout.millis", "3000");
        conf.set("dfs.storage.policy.satisfier.mode", HdfsConstants.StoragePolicySatisfierMode.EXTERNAL.toString());
        conf.setLong("dfs.storage.policy.satisfier.datanode.cache.refresh.interval.ms", 1000L);
        conf.setInt("dfs.storage.policy.satisfier.retry.max.attempts", 20);
        int dnNumber = storageTypes.length;
        int replication = 3;
        MiniDFSCluster.Builder clusterBuilder = new MiniDFSCluster.Builder(conf).storageTypes(storageTypes).storagesPerDatanode(3).numDataNodes(dnNumber);
        if (isHAEnabled) {
            clusterBuilder.nnTopology(MiniDFSNNTopology.simpleHATopology());
        }
        cluster = clusterBuilder.build();
        cluster.waitActive();
        if (isHAEnabled) {
            cluster.transitionToActive(0);
            fs = HATestUtil.configureFailoverFs(cluster, conf);
        } else {
            fs = cluster.getFileSystem();
        }
        this.nnc = DFSTestUtil.getNameNodeConnector(conf, HdfsServerConstants.MOVER_ID_PATH, 1, false);
        this.sps = new StoragePolicySatisfier(conf);
        this.ctxt = new ExternalSPSContext((SPSService)this.sps, this.nnc);
        this.sps.init((Context)this.ctxt);
        this.sps.start(HdfsConstants.StoragePolicySatisfierMode.EXTERNAL);
        this.createTestFiles(fs, (short)3);
    }

    private void createTestFiles(DistributedFileSystem dfs, short replication) throws Exception {
        DFSTestUtil.createFile((FileSystem)dfs, testFile, 1024L, replication, 0L);
        DFSTestUtil.createFile((FileSystem)dfs, parentFile, 1024L, replication, 0L);
        DFSTestUtil.createFile((FileSystem)dfs, childFile, 1024L, replication, 0L);
        DFSTestUtil.waitReplication((FileSystem)dfs, testFile, replication);
        DFSTestUtil.waitReplication((FileSystem)dfs, parentFile, replication);
        DFSTestUtil.waitReplication((FileSystem)dfs, childFile, replication);
    }

    private void clusterShutdown() throws IOException {
        if (fs != null) {
            fs.close();
            fs = null;
        }
        if (cluster != null) {
            cluster.shutdown(true);
            cluster = null;
        }
        if (this.sps != null) {
            this.sps.stopGracefully();
        }
    }

    @Test(timeout=300000L)
    public void testWithCheckpoint() throws Exception {
        SecondaryNameNode secondary = null;
        try {
            this.clusterSetUp();
            fs.setStoragePolicy(testFile, WARM);
            fs.satisfyStoragePolicy(testFile);
            conf.set("dfs.namenode.secondary.http-address", "0.0.0.0:0");
            secondary = new SecondaryNameNode(conf);
            secondary.doCheckpoint();
            this.restartCluster();
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.DISK, 1, 90000, fs);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.ARCHIVE, 2, 90000, fs);
            fs.setStoragePolicy(parentDir, COLD);
            fs.satisfyStoragePolicy(parentDir);
            DFSTestUtil.waitExpectedStorageType(parentFileName, StorageType.ARCHIVE, 3, 90000, fs);
            DFSTestUtil.waitExpectedStorageType(childFileName, StorageType.ARCHIVE, 3, 90000, fs);
        }
        finally {
            if (secondary != null) {
                secondary.shutdown();
            }
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000L)
    public void testWithRestarts() throws Exception {
        try {
            this.clusterSetUp();
            fs.setStoragePolicy(testFile, ONE_SSD);
            fs.satisfyStoragePolicy(testFile);
            this.restartCluster();
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.SSD, 1, 90000, fs);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.DISK, 2, 90000, fs);
            fs.setStoragePolicy(parentDir, COLD);
            fs.satisfyStoragePolicy(parentDir);
            this.restartCluster();
            DFSTestUtil.waitExpectedStorageType(parentFileName, StorageType.ARCHIVE, 3, 90000, fs);
            DFSTestUtil.waitExpectedStorageType(childFileName, StorageType.ARCHIVE, 3, 90000, fs);
        }
        finally {
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000L)
    public void testMultipleSatisfyStoragePolicy() throws Exception {
        try {
            conf = new HdfsConfiguration();
            long minCheckTimeout = 500L;
            conf.setLong("dfs.storage.policy.satisfier.recheck.timeout.millis", 500L);
            this.clusterSetUp(conf);
            fs.setStoragePolicy(testFile, ONE_SSD);
            fs.satisfyStoragePolicy(testFile);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.SSD, 1, 90000, fs);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.DISK, 2, 90000, fs);
            DFSTestUtil.waitForXattrRemoved(testFileName, "user.hdfs.sps", (Namesystem)cluster.getNamesystem(), 30000);
            fs.setStoragePolicy(testFile, COLD);
            fs.satisfyStoragePolicy(testFile);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.ARCHIVE, 3, 90000, fs);
        }
        finally {
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000000L)
    public void testDropSPS() throws Exception {
        try {
            this.clusterSetUp();
            fs.setStoragePolicy(testFile, ONE_SSD);
            fs.satisfyStoragePolicy(testFile);
            cluster.getNamesystem().getBlockManager().getSPSManager().changeModeEvent(HdfsConstants.StoragePolicySatisfierMode.NONE);
            DFSTestUtil.waitForXattrRemoved(testFileName, "user.hdfs.sps", (Namesystem)cluster.getNamesystem(), 30000);
        }
        finally {
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000L)
    public void testSPSShouldNotLeakXattrIfStorageAlreadySatisfied() throws Exception {
        try {
            this.clusterSetUp();
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.DISK, 3, 90000, fs);
            fs.satisfyStoragePolicy(testFile);
            DFSTestUtil.waitExpectedStorageType(testFileName, StorageType.DISK, 3, 90000, fs);
            DFSTestUtil.waitForXattrRemoved(testFileName, "user.hdfs.sps", (Namesystem)cluster.getNamesystem(), 30000);
        }
        finally {
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000L)
    public void testNameNodeRestartWhenSPSCalledOnChildFileAndParentDir() throws Exception {
        try {
            this.clusterSetUp();
            fs.setStoragePolicy(childFile, COLD);
            fs.satisfyStoragePolicy(childFile);
            DFSTestUtil.waitExpectedStorageType(childFile.toUri().getPath(), StorageType.ARCHIVE, 3, 30000, cluster.getFileSystem());
            Thread.sleep(30000L);
            fs.setStoragePolicy(childDir, COLD);
            fs.satisfyStoragePolicy(childDir);
            try {
                cluster.restartNameNodes();
            }
            catch (Exception e) {
                Assert.assertFalse((boolean)e.getMessage().contains("Cannot request to call satisfy storage policy"));
            }
        }
        finally {
            this.clusterShutdown();
        }
    }

    @Test(timeout=300000L)
    public void testSPSOnChildAndParentDirectory() throws Exception {
        try {
            this.clusterSetUp();
            fs.setStoragePolicy(parentDir, COLD);
            fs.satisfyStoragePolicy(childDir);
            DFSTestUtil.waitExpectedStorageType(childFileName, StorageType.ARCHIVE, 3, 30000, cluster.getFileSystem());
            fs.satisfyStoragePolicy(parentDir);
            DFSTestUtil.waitExpectedStorageType(parentFileName, StorageType.ARCHIVE, 3, 30000, cluster.getFileSystem());
        }
        finally {
            this.clusterShutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=300000L)
    public void testSPSxAttrWhenSpsCalledForDir() throws Exception {
        try {
            int i;
            this.clusterSetUp();
            Path parent = new Path("/parent");
            fs.mkdirs(parent);
            for (int i2 = 0; i2 < 5; ++i2) {
                DFSTestUtil.createFile((FileSystem)fs, new Path(parent, "f" + i2), 1024L, (short)3, 0L);
            }
            fs.setStoragePolicy(parent, COLD);
            MiniDFSCluster.DataNodeProperties stopDataNode = cluster.stopDataNode(0);
            fs.satisfyStoragePolicy(parent);
            FSNamesystem namesystem = cluster.getNamesystem();
            INode inode = namesystem.getFSDirectory().getINode("/parent");
            XAttrFeature f = inode.getXAttrFeature();
            Assert.assertTrue((String)"SPS xAttr should be exist", (f.getXAttr("user.hdfs.sps") != null ? 1 : 0) != 0);
            for (i = 0; i < 5; ++i) {
                inode = namesystem.getFSDirectory().getINode("/parent/f" + i);
                f = inode.getXAttrFeature();
                Assert.assertTrue((f == null ? 1 : 0) != 0);
            }
            cluster.restartDataNode(stopDataNode, false);
            for (i = 0; i < 5; ++i) {
                DFSTestUtil.waitExpectedStorageType("/parent/f" + i, StorageType.ARCHIVE, 3, 30000, cluster.getFileSystem());
            }
            DFSTestUtil.waitForXattrRemoved("/parent", "user.hdfs.sps", (Namesystem)namesystem, 10000);
        }
        finally {
            this.clusterShutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=300000L)
    public void testSPSxAttrWhenSpsCalledForFile() throws Exception {
        try {
            this.clusterSetUp();
            Path file = new Path("/file");
            DFSTestUtil.createFile((FileSystem)fs, file, 1024L, (short)3, 0L);
            fs.setStoragePolicy(file, COLD);
            MiniDFSCluster.DataNodeProperties stopDataNode = cluster.stopDataNode(0);
            fs.satisfyStoragePolicy(file);
            FSNamesystem namesystem = cluster.getNamesystem();
            final INode inode = namesystem.getFSDirectory().getINode("/file");
            XAttrFeature f = inode.getXAttrFeature();
            Assert.assertTrue((String)"SPS xAttr should be exist", (f.getXAttr("user.hdfs.sps") != null ? 1 : 0) != 0);
            cluster.restartDataNode(stopDataNode, false);
            DFSTestUtil.waitExpectedStorageType("/file", StorageType.ARCHIVE, 3, 30000, cluster.getFileSystem());
            GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

                public Boolean get() {
                    List existingXAttrs = XAttrStorage.readINodeXAttrs((INodeAttributes)inode);
                    return !existingXAttrs.contains("user.hdfs.sps");
                }
            }, (int)100, (int)10000);
        }
        finally {
            this.clusterShutdown();
        }
    }

    private void restartCluster() throws Exception {
        cluster.restartDataNodes();
        cluster.restartNameNodes();
        cluster.waitActive();
        cluster.triggerHeartbeats();
    }

    static {
        testFile = new Path("/testFile");
        testFileName = testFile.toString();
        parentDir = new Path("/parentDir");
        parentFile = new Path(parentDir, "parentFile");
        parentFileName = parentFile.toString();
        childDir = new Path(parentDir, "childDir");
        childFile = new Path(childDir, "childFile");
        childFileName = childFile.toString();
        storageTypes = new StorageType[][]{{StorageType.DISK, StorageType.ARCHIVE, StorageType.SSD}, {StorageType.DISK, StorageType.ARCHIVE, StorageType.SSD}, {StorageType.DISK, StorageType.ARCHIVE, StorageType.SSD}};
    }
}

