/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs;

import com.google.common.base.Preconditions;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Random;
import java.util.UUID;
import java.util.concurrent.TimeoutException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.StorageType;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DFSUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.MiniDFSNNTopology;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.impl.FsVolumeImpl;
import org.apache.hadoop.test.PathUtils;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;

public class TestMiniDFSCluster {
    private static final String CLUSTER_1 = "cluster1";
    private static final String CLUSTER_2 = "cluster2";
    private static final String CLUSTER_3 = "cluster3";
    private static final String CLUSTER_4 = "cluster4";
    private static final String CLUSTER_5 = "cluster5";
    protected File testDataPath;

    @Before
    public void setUp() {
        this.testDataPath = new File(PathUtils.getTestDir(this.getClass()), "miniclusters");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=100000L)
    public void testClusterWithoutSystemProperties() throws Throwable {
        String oldPrp = System.getProperty("test.build.data");
        System.clearProperty("test.build.data");
        HdfsConfiguration conf = new HdfsConfiguration();
        File testDataCluster1 = new File(this.testDataPath, CLUSTER_1);
        String c1Path = testDataCluster1.getAbsolutePath();
        conf.set("hdfs.minidfs.basedir", c1Path);
        try (MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).build();){
            Assert.assertEquals((Object)new File(c1Path + "/data"), (Object)new File(cluster.getDataDirectory()));
        }
        finally {
            if (oldPrp != null) {
                System.setProperty("test.build.data", oldPrp);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=100000L)
    public void testClusterSetStorageCapacity() throws Throwable {
        HdfsConfiguration conf = new HdfsConfiguration();
        boolean numDatanodes = true;
        int defaultBlockSize = 1024;
        int blocks = 100;
        int blocksSize = 1024;
        int fileLen = 102400;
        long capcacity = 0xC800000L;
        long[] capacities = new long[]{0xC800000L, 0x19000000L};
        MiniDFSCluster cluster = this.newCluster((Configuration)conf, 1, capacities, 1024, 102400);
        try {
            this.verifyStorageCapacity(cluster, capacities);
            cluster.restartDataNodes();
            cluster.waitActive();
            this.verifyStorageCapacity(cluster, capacities);
            cluster.restartNameNodes();
            cluster.waitActive();
            this.verifyStorageCapacity(cluster, capacities);
            cluster.restartNameNodes();
            cluster.restartDataNodes();
            cluster.waitActive();
            this.verifyStorageCapacity(cluster, capacities);
            cluster.restartDataNodes();
            cluster.restartNameNodes();
            cluster.waitActive();
            this.verifyStorageCapacity(cluster, capacities);
        }
        finally {
            if (cluster != null) {
                cluster.shutdown();
            }
        }
    }

    private void verifyStorageCapacity(MiniDFSCluster cluster, long[] capacities) throws IOException {
        FsVolumeImpl source = null;
        FsVolumeImpl dest = null;
        for (int i = 0; i < cluster.getDataNodes().size(); ++i) {
            DataNode dnNode = cluster.getDataNodes().get(i);
            try (FsDatasetSpi.FsVolumeReferences refs = dnNode.getFSDataset().getFsVolumeReferences();){
                source = (FsVolumeImpl)refs.get(0);
                dest = (FsVolumeImpl)refs.get(1);
                Assert.assertEquals((long)capacities[0], (long)source.getCapacity());
                Assert.assertEquals((long)capacities[1], (long)dest.getCapacity());
                continue;
            }
        }
    }

    private MiniDFSCluster newCluster(Configuration conf, int numDatanodes, long[] storageCapacities, int defaultBlockSize, int fileLen) throws IOException, InterruptedException, TimeoutException {
        conf.setBoolean("dfs.disk.balancer.enabled", true);
        conf.setLong("dfs.blocksize", (long)defaultBlockSize);
        conf.setInt("dfs.bytes-per-checksum", defaultBlockSize);
        conf.setLong("dfs.heartbeat.interval", 1L);
        String fileName = "/" + UUID.randomUUID().toString();
        Path filePath = new Path(fileName);
        Preconditions.checkNotNull((Object)storageCapacities);
        Preconditions.checkArgument((storageCapacities.length == 2 ? 1 : 0) != 0, (Object)"need to specify capacities for two storages.");
        MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).numDataNodes(numDatanodes).storageCapacities(storageCapacities).storageTypes(new StorageType[]{StorageType.DISK, StorageType.DISK}).storagesPerDatanode(2).build();
        cluster.waitActive();
        boolean replicationFactor = true;
        Random r = new Random();
        DistributedFileSystem fs = cluster.getFileSystem(0);
        DFSTestUtil.createFile((FileSystem)fs, filePath, fileLen, (short)1, r.nextLong());
        DFSTestUtil.waitReplication((FileSystem)fs, filePath, (short)1);
        return cluster;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=100000L)
    public void testIsClusterUpAfterShutdown() throws Throwable {
        HdfsConfiguration conf = new HdfsConfiguration();
        File testDataCluster4 = new File(this.testDataPath, CLUSTER_4);
        MiniDFSCluster cluster4 = new MiniDFSCluster.Builder((Configuration)conf, testDataCluster4).build();
        try {
            DistributedFileSystem dfs = cluster4.getFileSystem();
            dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_ENTER);
            cluster4.shutdown();
        }
        finally {
            while (cluster4.isClusterUp()) {
                Thread.sleep(1000L);
            }
        }
    }

    @Test(timeout=100000L)
    public void testClusterSetDatanodeHostname() throws Throwable {
        Assume.assumeTrue((boolean)System.getProperty("os.name").startsWith("Linux"));
        HdfsConfiguration conf = new HdfsConfiguration();
        conf.set("dfs.datanode.hostname", "MYHOST");
        File testDataCluster5 = new File(this.testDataPath, CLUSTER_5);
        try (MiniDFSCluster cluster5 = new MiniDFSCluster.Builder((Configuration)conf, testDataCluster5).numDataNodes(1).checkDataNodeHostConfig(true).build();){
            Assert.assertEquals((String)"DataNode hostname config not respected", (Object)"MYHOST", (Object)cluster5.getDataNodes().get(0).getDatanodeId().getHostName());
        }
    }

    @Test
    public void testClusterSetDatanodeDifferentStorageType() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        StorageType[][] storageType = new StorageType[][]{{StorageType.DISK, StorageType.ARCHIVE}, {StorageType.DISK}, {StorageType.ARCHIVE}};
        try (MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).storageTypes(storageType).build();){
            cluster.waitActive();
            ArrayList<DataNode> dataNodes = cluster.getDataNodes();
            for (int i = 0; i < storageType.length; ++i) {
                Assert.assertEquals((long)DataNode.getStorageLocations((Configuration)dataNodes.get(i).getConf()).size(), (long)storageType[i].length);
            }
        }
    }

    @Test
    public void testClusterNoStorageTypeSetForDatanodes() throws IOException {
        HdfsConfiguration conf = new HdfsConfiguration();
        try (MiniDFSCluster cluster = new MiniDFSCluster.Builder((Configuration)conf).numDataNodes(3).build();){
            cluster.waitActive();
            ArrayList<DataNode> dataNodes = cluster.getDataNodes();
            for (DataNode datanode : dataNodes) {
                Assert.assertEquals((long)DataNode.getStorageLocations((Configuration)datanode.getConf()).size(), (long)2L);
            }
        }
    }

    @Test
    public void testSetUpFederatedCluster() throws Exception {
        Configuration conf = new Configuration();
        try (MiniDFSCluster cluster = new MiniDFSCluster.Builder(conf).nnTopology(MiniDFSNNTopology.simpleHAFederatedTopology(2)).numDataNodes(2).build();){
            cluster.waitActive();
            cluster.transitionToActive(1);
            cluster.transitionToActive(3);
            Assert.assertEquals((Object)"standby", (Object)cluster.getNamesystem(0).getHAState());
            Assert.assertEquals((Object)"active", (Object)cluster.getNamesystem(1).getHAState());
            Assert.assertEquals((Object)"standby", (Object)cluster.getNamesystem(2).getHAState());
            Assert.assertEquals((Object)"active", (Object)cluster.getNamesystem(3).getHAState());
            String ns0nn0 = conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns0", "nn0"}));
            String ns0nn1 = conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns0", "nn1"}));
            String ns1nn0 = conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns1", "nn0"}));
            String ns1nn1 = conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns1", "nn1"}));
            for (MiniDFSCluster.NameNodeInfo nnInfo : cluster.getNameNodeInfos()) {
                Assert.assertEquals((Object)ns0nn0, (Object)nnInfo.conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns0", "nn0"})));
                Assert.assertEquals((Object)ns0nn1, (Object)nnInfo.conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns0", "nn1"})));
                Assert.assertEquals((Object)ns1nn0, (Object)nnInfo.conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns1", "nn0"})));
                Assert.assertEquals((Object)ns1nn1, (Object)nnInfo.conf.get(DFSUtil.addKeySuffixes((String)"dfs.namenode.http-address", (String[])new String[]{"ns1", "nn1"})));
            }
            cluster.shutdownNameNode(0);
            cluster.shutdownNameNode(1);
            cluster.restartNameNode(0);
            cluster.restartNameNode(1);
        }
    }
}

