/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URI;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.crypto.key.KeyProvider;
import org.apache.hadoop.fs.FileContext;
import org.apache.hadoop.fs.FileContextTestWrapper;
import org.apache.hadoop.fs.FileEncryptionInfo;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileSystemTestHelper;
import org.apache.hadoop.fs.FileSystemTestWrapper;
import org.apache.hadoop.fs.Options;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.fs.SafeModeAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hdfs.DFSTestUtil;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.HdfsConfiguration;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.apache.hadoop.hdfs.client.CreateEncryptionZoneFlag;
import org.apache.hadoop.hdfs.client.HdfsAdmin;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.hdfs.protocol.ReencryptionStatus;
import org.apache.hadoop.hdfs.protocol.SnapshotAccessControlException;
import org.apache.hadoop.hdfs.protocol.ZoneReencryptionStatus;
import org.apache.hadoop.hdfs.server.namenode.EncryptionFaultInjector;
import org.apache.hadoop.hdfs.server.namenode.EncryptionZoneManager;
import org.apache.hadoop.hdfs.server.namenode.FSNamesystem;
import org.apache.hadoop.hdfs.server.namenode.ReencryptionHandler;
import org.apache.hadoop.hdfs.server.namenode.ReencryptionUpdater;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.ipc.RetriableException;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.test.Whitebox;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class TestReencryption {
    protected static final Logger LOG = LoggerFactory.getLogger(TestReencryption.class);
    private Configuration conf;
    private FileSystemTestHelper fsHelper;
    private MiniDFSCluster cluster;
    protected HdfsAdmin dfsAdmin;
    private DistributedFileSystem fs;
    private FSNamesystem fsn;
    private File testRootDir;
    private static final String TEST_KEY = "test_key";
    private FileSystemTestWrapper fsWrapper;
    private FileContextTestWrapper fcWrapper;
    private static final EnumSet<CreateEncryptionZoneFlag> NO_TRASH = EnumSet.of(CreateEncryptionZoneFlag.NO_TRASH);
    @Rule
    public Timeout globalTimeout = new Timeout(180000);

    protected String getKeyProviderURI() {
        return "jceks://file" + new Path(this.testRootDir.toString(), "test.jks").toUri();
    }

    @Before
    public void setup() throws Exception {
        this.conf = new HdfsConfiguration();
        this.fsHelper = new FileSystemTestHelper();
        String testRoot = this.fsHelper.getTestRootDir();
        this.testRootDir = new File(testRoot).getAbsoluteFile();
        this.conf.set("hadoop.security.key.provider.path", this.getKeyProviderURI());
        this.conf.setBoolean("dfs.namenode.delegation.token.always-use", true);
        this.conf.setInt("dfs.namenode.list.encryption.zones.num.responses", 2);
        this.conf.setInt("dfs.ls.limit", 3);
        this.conf.setInt("dfs.namenode.reencrypt.batch.size", 5);
        this.cluster = new MiniDFSCluster.Builder(this.conf).numDataNodes(1).build();
        this.cluster.waitActive();
        this.cluster.waitClusterUp();
        this.fs = this.cluster.getFileSystem();
        this.fsn = this.cluster.getNamesystem();
        this.fsWrapper = new FileSystemTestWrapper((FileSystem)this.fs);
        this.fcWrapper = new FileContextTestWrapper(FileContext.getFileContext((URI)this.cluster.getURI(), (Configuration)this.conf));
        this.dfsAdmin = new HdfsAdmin(this.cluster.getURI(), this.conf);
        this.setProvider();
        DFSTestUtil.createKey(TEST_KEY, this.cluster, this.conf);
        GenericTestUtils.setLogLevel((Logger)EncryptionZoneManager.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)ReencryptionHandler.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)ReencryptionStatus.LOG, (Level)Level.TRACE);
        GenericTestUtils.setLogLevel((Logger)ReencryptionUpdater.LOG, (Level)Level.TRACE);
    }

    protected void setProvider() {
        this.fs.getClient().setKeyProvider((KeyProvider)this.cluster.getNameNode().getNamesystem().getProvider());
    }

    @After
    public void teardown() {
        if (this.cluster != null) {
            this.cluster.shutdown();
            this.cluster = null;
        }
        EncryptionFaultInjector.instance = new EncryptionFaultInjector();
    }

    private FileEncryptionInfo getFileEncryptionInfo(Path path) throws Exception {
        return this.fsn.getFileInfo(path.toString(), false, false, false).getFileEncryptionInfo();
    }

    @Test
    public void testReencryptionBasic() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path("/dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path encFile1 = new Path(zone, "0");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile1);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.assertKeyVersionEquals(encFile1, fei0);
        this.verifyZoneStatus(zone, null, 0L);
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(2);
        FileEncryptionInfo fei1 = this.getFileEncryptionInfo(encFile1);
        this.assertKeyVersionChanged(encFile1, fei0);
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertNotEquals((Object)fei0.getEzKeyVersionName(), (Object)zs.getEzKeyVersionName());
        Assert.assertEquals((Object)fei1.getEzKeyVersionName(), (Object)zs.getEzKeyVersionName());
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(3);
        this.assertKeyVersionEquals(encFile1, fei1);
        try {
            this.dfsAdmin.reencryptEncryptionZone(subdir, HdfsConstants.ReencryptAction.START);
            Assert.fail((String)"Re-encrypting non-EZ should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"not the root of an encryption zone", (Throwable)expected);
        }
        try {
            this.dfsAdmin.reencryptEncryptionZone(new Path(zone, "notexist"), HdfsConstants.ReencryptAction.START);
            Assert.fail((String)"Re-encrypting non-existing dir should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            Assert.assertTrue((boolean)(expected.unwrapRemoteException() instanceof FileNotFoundException));
        }
        try {
            this.dfsAdmin.reencryptEncryptionZone(encFile1, HdfsConstants.ReencryptAction.START);
            Assert.fail((String)"Re-encrypting on a file should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"not the root of an encryption zone", (Throwable)expected);
        }
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        try {
            this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"already submitted", (Throwable)expected);
        }
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedZones(4);
        Path emptyZone = new Path("/emptyZone");
        this.fsWrapper.mkdir(emptyZone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(emptyZone, TEST_KEY, NO_TRASH);
        this.dfsAdmin.reencryptEncryptionZone(emptyZone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(5);
        this.dfsAdmin.reencryptEncryptionZone(emptyZone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(6);
        Path renamedZone = new Path("/renamedZone");
        this.fsWrapper.rename(zone, renamedZone, new Options.Rename[0]);
        it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)renamedZone.toString(), (Object)zs.getZoneName());
    }

    @Test
    public void testReencryptOrdering() throws Exception {
        int i;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        Path subdir = new Path(zone, "dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        for (i = 0; i < 4; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        for (i = 0; i < 5; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "f" + Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path lastReencryptedFile = new Path(subdir, "f");
        Path notReencrypted = new Path(zone, "f0");
        FileEncryptionInfo fei = this.getFileEncryptionInfo(lastReencryptedFile);
        FileEncryptionInfo feiLast = this.getFileEncryptionInfo(notReencrypted);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.assertKeyVersionChanged(lastReencryptedFile, fei);
        this.assertKeyVersionEquals(notReencrypted, feiLast);
    }

    @Test
    public void testDeleteDuringReencrypt() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.getEzManager().pauseReencryptForTesting();
        this.getEzManager().resetMetricsForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.fs.delete(zone, true);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForTotalZones(0);
        Assert.assertNull((Object)this.getZoneStatus(zone.toString()));
    }

    @Test
    public void testZoneDeleteDuringReencrypt() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resetMetricsForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.fs.delete(zoneParent, true);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForTotalZones(0);
        Assert.assertNull((Object)this.getEzManager().getZoneStatus(zone.toString()));
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertFalse((boolean)it.hasNext());
    }

    @Test
    public void testRestartAfterReencrypt() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path encFile0 = new Path(zone, "0");
        Path encFile9 = new Path(zone, "9");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9 = this.getFileEncryptionInfo(encFile9);
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.assertKeyVersionChanged(encFile0, fei0);
        this.assertKeyVersionChanged(encFile9, fei9);
        FileEncryptionInfo fei0new = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9new = this.getFileEncryptionInfo(encFile9);
        this.restartClusterDisableReencrypt();
        this.assertKeyVersionEquals(encFile0, fei0new);
        this.assertKeyVersionEquals(encFile9, fei9new);
        Assert.assertNull((String)"Re-encrypt queue should be empty after restart", (Object)this.getReencryptionStatus().getNextUnprocessedZone());
    }

    @Test
    public void testRestartWithRenames() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "f"), 8196L, (short)1, 65261L);
        this.fsWrapper.rename(new Path(zone, "f"), new Path(zone, "f1"), new Options.Rename[0]);
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.cluster.restartNameNodes();
        this.cluster.waitActive();
        this.waitForReencryptedZones(1);
    }

    @Test
    public void testRestartDuringReencrypt() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        this.fsWrapper.mkdir(new Path(zone, "dir_empty"), FsPermission.getDirDefault(), true);
        Path subdir = new Path(zone, "dir2");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        this.fsWrapper.mkdir(new Path(subdir, "dir_empty2"), FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        subdir = new Path(zone, "dir1");
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.fsWrapper.mkdir(new Path(subdir, "dir_empty1"), FsPermission.getDirDefault(), true);
        Path encFile0 = new Path(subdir, "0");
        Path encFile9 = new Path(subdir, "9");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9 = this.getFileEncryptionInfo(encFile9);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.restartClusterDisableReencrypt();
        Long zoneId = this.fsn.getFSDirectory().getINode(zone.toString()).getId();
        Assert.assertEquals((String)"Re-encrypt should restore to the last checkpoint zone", (Object)zoneId, (Object)this.getReencryptionStatus().getNextUnprocessedZone());
        Assert.assertEquals((String)"Re-encrypt should restore to the last checkpoint file", (Object)new Path(subdir, "4").toString(), (Object)this.getEzManager().getZoneStatus(zone.toString()).getLastCheckpointFile());
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedZones(1);
        this.assertKeyVersionChanged(encFile0, fei0);
        this.assertKeyVersionChanged(encFile9, fei9);
        Assert.assertNull((String)"Re-encrypt queue should be empty after restart", (Object)this.getReencryptionStatus().getNextUnprocessedZone());
        Assert.assertEquals((long)11L, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRestartAfterReencryptAndCheckpoint() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path encFile0 = new Path(zone, "0");
        Path encFile9 = new Path(zone, "9");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9 = this.getFileEncryptionInfo(encFile9);
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.assertKeyVersionChanged(encFile0, fei0);
        this.assertKeyVersionChanged(encFile9, fei9);
        FileEncryptionInfo fei0new = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9new = this.getFileEncryptionInfo(encFile9);
        this.fs.setSafeMode(SafeModeAction.ENTER);
        this.fs.saveNamespace();
        this.fs.setSafeMode(SafeModeAction.LEAVE);
        this.restartClusterDisableReencrypt();
        this.assertKeyVersionEquals(encFile0, fei0new);
        this.assertKeyVersionEquals(encFile9, fei9new);
        Assert.assertNull((String)"Re-encrypt queue should be empty after restart", (Object)this.getReencryptionStatus().getNextUnprocessedZone());
    }

    @Test
    public void testReencryptLoadedFromEdits() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path encFile0 = new Path(zone, "0");
        Path encFile9 = new Path(zone, "9");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9 = this.getFileEncryptionInfo(encFile9);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.restartClusterDisableReencrypt();
        this.waitForQueuedZones(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedZones(1);
        this.assertKeyVersionChanged(encFile0, fei0);
        this.assertKeyVersionChanged(encFile9, fei9);
        this.verifyZoneStatus(zone, fei0, 11L);
    }

    private void verifyZoneStatus(Path zone, FileEncryptionInfo fei, long expectedFiles) throws IOException {
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        if (fei != null) {
            Assert.assertNotEquals((Object)fei.getEzKeyVersionName(), (Object)zs.getEzKeyVersionName());
        }
        Assert.assertEquals((long)expectedFiles, (long)zs.getFilesReencrypted());
    }

    private void verifyZoneCompletionTime(ZoneReencryptionStatus zs) {
        Assert.assertNotNull((Object)zs);
        Assert.assertTrue((String)("Completion time should be positive. " + zs.getCompletionTime()), (zs.getCompletionTime() > 0L ? 1 : 0) != 0);
        Assert.assertTrue((String)("Completion time " + zs.getCompletionTime() + " should be no less than submission time " + zs.getSubmissionTime()), (zs.getCompletionTime() >= zs.getSubmissionTime() ? 1 : 0) != 0);
    }

    @Test
    public void testReencryptLoadedFromFsimage() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path encFile0 = new Path(zone, "0");
        Path encFile9 = new Path(zone, "9");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile0);
        FileEncryptionInfo fei9 = this.getFileEncryptionInfo(encFile9);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.fs.setSafeMode(SafeModeAction.ENTER);
        this.fs.saveNamespace();
        this.fs.setSafeMode(SafeModeAction.LEAVE);
        this.restartClusterDisableReencrypt();
        this.waitForQueuedZones(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedZones(1);
        this.assertKeyVersionChanged(encFile0, fei0);
        this.assertKeyVersionChanged(encFile9, fei9);
        this.verifyZoneStatus(zone, fei0, 11L);
    }

    @Test
    public void testReencryptCommandsQueuedOrdering() throws Exception {
        int i;
        Object zoneId;
        int i2;
        Path zoneParent = new Path("/zones");
        String zoneBaseName = zoneParent.toString() + "/zone";
        int numZones = 10;
        for (i2 = 0; i2 < 10; ++i2) {
            Path zone = new Path(zoneBaseName + i2);
            this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
            this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        }
        this.getEzManager().pauseReencryptForTesting();
        for (i2 = 0; i2 < 10; ++i2) {
            this.dfsAdmin.reencryptEncryptionZone(new Path(zoneBaseName + i2), HdfsConstants.ReencryptAction.START);
        }
        this.waitForQueuedZones(10);
        ReencryptionStatus rzs = new ReencryptionStatus(this.getReencryptionStatus());
        for (int i3 = 0; i3 < 10; ++i3) {
            zoneId = this.fsn.getFSDirectory().getINode(zoneBaseName + i3).getId();
            Assert.assertEquals((Object)zoneId, (Object)rzs.getNextUnprocessedZone());
            rzs.removeZone((Long)zoneId);
        }
        HashSet<Integer> cancelled = new HashSet<Integer>(Arrays.asList(0, 3, 4));
        zoneId = cancelled.iterator();
        while (zoneId.hasNext()) {
            int cancel = (Integer)zoneId.next();
            this.dfsAdmin.reencryptEncryptionZone(new Path(zoneBaseName + cancel), HdfsConstants.ReencryptAction.CANCEL);
        }
        this.restartClusterDisableReencrypt();
        this.waitForQueuedZones(10 - cancelled.size());
        rzs = new ReencryptionStatus(this.getReencryptionStatus());
        for (i = 0; i < 10; ++i) {
            if (cancelled.contains(i)) continue;
            Long zoneId2 = this.fsn.getFSDirectory().getINode(zoneBaseName + i).getId();
            Assert.assertEquals((Object)zoneId2, (Object)rzs.getNextUnprocessedZone());
            rzs.removeZone(zoneId2);
        }
        this.fs.setSafeMode(SafeModeAction.ENTER);
        this.fs.saveNamespace();
        this.fs.setSafeMode(SafeModeAction.LEAVE);
        this.restartClusterDisableReencrypt();
        this.waitForQueuedZones(10 - cancelled.size());
        rzs = new ReencryptionStatus(this.getReencryptionStatus());
        for (i = 0; i < 10; ++i) {
            if (cancelled.contains(i)) continue;
            Long zoneId3 = this.fsn.getFSDirectory().getINode(zoneBaseName + i).getId();
            Assert.assertEquals((Object)zoneId3, (Object)rzs.getNextUnprocessedZone());
            rzs.removeZone(zoneId3);
        }
    }

    @Test
    public void testReencryptNestedZones() throws Exception {
        int i;
        int len = 8196;
        Path zoneRoot = new Path("/");
        Path zoneL1 = new Path(zoneRoot, "level1");
        Path zoneL2 = new Path(zoneL1, "level2");
        Path nonzoneDir = new Path(zoneRoot, "dir");
        this.dfsAdmin.createEncryptionZone(zoneRoot, TEST_KEY, NO_TRASH);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(zoneRoot, "file"), 8196L, (short)1, 65261L);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(nonzoneDir, "dfile"), 8196L, (short)1, 65261L);
        this.fsWrapper.mkdir(zoneL1, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zoneL1, TEST_KEY, NO_TRASH);
        for (i = 0; i < 3; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zoneL1, "fileL1-" + i), 8196L, (short)1, 65261L);
        }
        this.fsWrapper.mkdir(zoneL2, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zoneL2, TEST_KEY, NO_TRASH);
        for (i = 0; i < 4; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zoneL2, "fileL2-" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zoneRoot, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        ReencryptionStatus rzs = this.getReencryptionStatus();
        Assert.assertEquals((Object)this.fsn.getFSDirectory().getINode(zoneRoot.toString()).getId(), (Object)rzs.getNextUnprocessedZone());
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zoneRoot.toString());
        Assert.assertEquals((long)2L, (long)this.getZoneStatus(zoneRoot.toString()).getFilesReencrypted());
        this.getEzManager().resetMetricsForTesting();
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zoneL1, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        rzs = this.getReencryptionStatus();
        Assert.assertEquals((Object)this.fsn.getFSDirectory().getINode(zoneL1.toString()).getId(), (Object)rzs.getNextUnprocessedZone());
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zoneL1.toString());
        Assert.assertEquals((long)3L, (long)this.getZoneStatus(zoneL1.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceCreateHandler() throws Exception {
        int i;
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        int expected = 10;
        for (i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        for (i = 0; i < 6; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file8" + i), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "dirsub");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i2 = 10; i2 < 15; ++i2) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i2), 8196L, (short)1, 65261L);
        }
        Path sub = new Path(zone, "sub");
        this.fsWrapper.mkdir(sub, FsPermission.getDirDefault(), true);
        for (int i3 = 15; i3 < 20; ++i3) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(sub, "file" + i3), 8196L, (short)1, 65261L);
        }
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)expected, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteHandler() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        int expected = 15;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "subdir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i = 10; i < 15; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.fsWrapper.delete(new Path(zone, "file5"), true);
        this.fsWrapper.delete(new Path(zone, "file8"), true);
        expected -= 2;
        this.fsWrapper.delete(subdir, true);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(expected -= 5), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteUpdater() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        int expected = 15;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "subdir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i = 10; i < 15; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthCheckpoint(zone.toString(), 1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.getEzManager().resumeReencryptForTesting();
        Thread.sleep(3000L);
        this.fsWrapper.delete(new Path(zone, "file5"), true);
        this.fsWrapper.delete(new Path(zone, "file8"), true);
        expected -= 2;
        this.fsWrapper.delete(subdir, true);
        this.getEzManager().resumeReencryptUpdaterForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(expected -= 5), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteCurrentDirHandler() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        Path subdir = new Path(zone, "subdir");
        int expected = 15;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i), 8196L, (short)1, 65261L);
        }
        Path subdir2 = new Path(zone, "subdir2");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i = 10; i < 15; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir2, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.fsWrapper.delete(subdir, true);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(expected -= 5), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteCurrentDirUpdater() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        Path subdir = new Path(zone, "subdir");
        int expected = 15;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i), 8196L, (short)1, 65261L);
        }
        Path subdir2 = new Path(zone, "subdir2");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i = 10; i < 15; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir2, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthCheckpoint(zone.toString(), 1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.getEzManager().resumeReencryptForTesting();
        Thread.sleep(3000L);
        this.fsWrapper.delete(subdir, true);
        this.getEzManager().resumeReencryptUpdaterForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(expected -= 5), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteZoneHandler() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 11; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().pauseForTestingAfterNthCheckpoint(zone.toString(), 1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        Thread.sleep(3000L);
        Map tasks = (Map)Whitebox.getInternalState((Object)this.getHandler(), (String)"submissions");
        LinkedList<Future> futures = new LinkedList<Future>();
        for (ReencryptionUpdater.ZoneSubmissionTracker zst : tasks.values()) {
            for (Future f : zst.getTasks()) {
                futures.add(f);
            }
        }
        this.fsWrapper.delete(zone, true);
        this.getEzManager().resumeReencryptForTesting();
        for (Future f : futures) {
            Assert.assertTrue((boolean)f.isDone());
        }
        this.waitForTotalZones(0);
    }

    @Test
    public void testRaceDeleteCreateHandler() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        int expected = 10;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        Path recreated = new Path(zone, "file9");
        this.fsWrapper.delete(recreated, true);
        DFSTestUtil.createFile((FileSystem)this.fs, recreated, 8196L, (short)2, 65261L);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(--expected), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testRaceDeleteCreateUpdater() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        int expected = 10;
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthCheckpoint(zone.toString(), 1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.getEzManager().resumeReencryptForTesting();
        Thread.sleep(3000L);
        Path recreated = new Path(zone, "file9");
        FileEncryptionInfo feiOrig = this.getFileEncryptionInfo(recreated);
        String contentOrig = DFSTestUtil.readFile((FileSystem)this.fs, recreated);
        this.fsWrapper.delete(recreated, true);
        DFSTestUtil.createFile((FileSystem)this.fs, recreated, 8196L, (short)2, 65261L);
        this.getEzManager().resumeReencryptUpdaterForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)(--expected), (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
        this.assertKeyVersionChanged(recreated, feiOrig);
        String content = DFSTestUtil.readFile((FileSystem)this.fs, recreated);
        Assert.assertEquals((Object)contentOrig, (Object)content);
    }

    @Test
    public void testReencryptRaceRename() throws Exception {
        int len = 8196;
        Path zone = new Path("/dir");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, "file" + i), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path(zone, "subdir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        for (int i = 10; i < 15; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "file" + i), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForReencryptedFiles(zone.toString(), 5);
        try {
            this.fsWrapper.rename(new Path(zone, "file8"), new Path(zone, "file08"), new Options.Rename[0]);
            Assert.fail((String)"rename a file in an EZ should be disabled");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains((String)"under re-encryption", (Throwable)e);
        }
        this.getEzManager().pauseReencryptUpdaterForTesting();
        this.getEzManager().resumeReencryptForTesting();
        try {
            this.fsWrapper.rename(new Path(zone, "file8"), new Path(zone, "file08"), new Options.Rename[0]);
            Assert.fail((String)"rename a file in an EZ should be disabled");
        }
        catch (IOException e) {
            GenericTestUtils.assertExceptionContains((String)"under re-encryption", (Throwable)e);
        }
    }

    @Test
    public void testReencryptSnapshots() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.allowSnapshot(zone);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path("/dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        Path zoneSnap = this.fs.createSnapshot(zone);
        this.fsWrapper.rename(new Path(zone, "5"), new Path(zone, "5new"), new Options.Rename[0]);
        this.fsWrapper.rename(new Path(zone, "6"), new Path(zone, "6new"), new Options.Rename[0]);
        this.fsWrapper.delete(new Path(zone, "6new"), true);
        Path encFile1 = new Path(zone, "0");
        FileEncryptionInfo fei0 = this.getFileEncryptionInfo(encFile1);
        this.rollKey(TEST_KEY);
        try {
            this.dfsAdmin.reencryptEncryptionZone(zoneSnap, HdfsConstants.ReencryptAction.START);
            Assert.fail((String)"Reencrypt command on snapshot path should fail.");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception", (Throwable)expected);
            Assert.assertTrue((boolean)(expected.unwrapRemoteException() instanceof SnapshotAccessControlException));
        }
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.waitForReencryptedFiles(zone.toString(), 9);
        this.assertKeyVersionChanged(encFile1, fei0);
    }

    private void restartClusterDisableReencrypt() throws Exception {
        this.cluster.restartNameNode(false);
        this.fsn = this.cluster.getNamesystem();
        this.getEzManager().pauseReencryptForTesting();
        this.cluster.waitActive();
        this.cluster.waitClusterUp();
    }

    private void waitForReencryptedZones(final int expected) throws TimeoutException, InterruptedException {
        LOG.info("Waiting for re-encrypted zones to be {}", (Object)expected);
        try {
            GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

                @Override
                public Boolean get() {
                    return TestReencryption.this.getReencryptionStatus().getNumZonesReencrypted() == (long)expected;
                }
            }, (long)100L, (long)10000L);
        }
        finally {
            LOG.info("Re-encrypted zones = {} ", (Object)this.getReencryptionStatus().getNumZonesReencrypted());
        }
    }

    private void waitForQueuedZones(final int expected) throws TimeoutException, InterruptedException {
        LOG.info("Waiting for queued zones for re-encryption to be {}", (Object)expected);
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                return TestReencryption.this.getReencryptionStatus().zonesQueued() == expected;
            }
        }, (long)100L, (long)10000L);
    }

    private void waitForTotalZones(final int expected) throws TimeoutException, InterruptedException {
        LOG.info("Waiting for queued zones for re-encryption to be {}", (Object)expected);
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                return TestReencryption.this.getReencryptionStatus().zonesTotal() == expected;
            }
        }, (long)100L, (long)10000L);
    }

    private void waitForZoneCompletes(final String zone) throws TimeoutException, InterruptedException {
        LOG.info("Waiting for re-encryption zone {} to complete.", (Object)zone);
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                try {
                    return TestReencryption.this.getZoneStatus(zone).getState() == ZoneReencryptionStatus.State.Completed;
                }
                catch (Exception ex) {
                    LOG.error("Exception caught", (Throwable)ex);
                    return false;
                }
            }
        }, (long)100L, (long)10000L);
    }

    private EncryptionZoneManager getEzManager() {
        return this.fsn.getFSDirectory().ezManager;
    }

    private ReencryptionStatus getReencryptionStatus() {
        return this.getEzManager().getReencryptionStatus();
    }

    private ZoneReencryptionStatus getZoneStatus(String zone) throws IOException {
        return this.getEzManager().getZoneStatus(zone);
    }

    private void waitForReencryptedFiles(final String zone, final int expected) throws TimeoutException, InterruptedException {
        LOG.info("Waiting for total re-encrypted file count to be {}", (Object)expected);
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                try {
                    return TestReencryption.this.getZoneStatus(zone).getFilesReencrypted() == (long)expected;
                }
                catch (IOException e) {
                    return false;
                }
            }
        }, (long)100L, (long)10000L);
    }

    private void assertKeyVersionChanged(Path file, FileEncryptionInfo original) throws Exception {
        FileEncryptionInfo actual = this.getFileEncryptionInfo(file);
        Assert.assertNotEquals((String)"KeyVersion should be different", (Object)original.getEzKeyVersionName(), (Object)actual.getEzKeyVersionName());
    }

    private void assertKeyVersionEquals(Path file, FileEncryptionInfo expected) throws Exception {
        FileEncryptionInfo actual = this.getFileEncryptionInfo(file);
        Assert.assertEquals((String)"KeyVersion should be the same", (Object)expected.getEzKeyVersionName(), (Object)actual.getEzKeyVersionName());
    }

    @Test
    public void testReencryptCancel() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path("/dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)0L, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
        try {
            this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
        }
        catch (RemoteException expected) {
            GenericTestUtils.assertExceptionContains((String)"not under re-encryption", (Throwable)expected);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.getEzManager().resumeReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
        this.getEzManager().resumeReencryptForTesting();
        this.waitForZoneCompletes(zone.toString());
        Assert.assertEquals((long)5L, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
        Assert.assertNull((Object)this.getEzManager().getZoneStatus(zone.toString()).getLastCheckpointFile());
        Assert.assertNull((Object)this.getReencryptionStatus().getNextUnprocessedZone());
        try {
            this.dfsAdmin.reencryptEncryptionZone(subdir, HdfsConstants.ReencryptAction.CANCEL);
            Assert.fail((String)"Re-encrypting non-EZ should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"not the root of an encryption zone", (Throwable)expected);
        }
        try {
            this.dfsAdmin.reencryptEncryptionZone(new Path(zone, "notexist"), HdfsConstants.ReencryptAction.CANCEL);
            Assert.fail((String)"Re-encrypting non-existing dir should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            Assert.assertTrue((boolean)(expected.unwrapRemoteException() instanceof FileNotFoundException));
        }
        Path encFile = new Path(zone, "0");
        try {
            this.dfsAdmin.reencryptEncryptionZone(encFile, HdfsConstants.ReencryptAction.CANCEL);
            Assert.fail((String)"Re-encrypting on a file should fail");
        }
        catch (RemoteException expected) {
            LOG.info("Expected exception caught.", (Throwable)expected);
            GenericTestUtils.assertExceptionContains((String)"not the root of an encryption zone", (Throwable)expected);
        }
        Assert.assertEquals((long)5L, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    private void cancelFutureDuringReencryption(Path zone) throws Exception {
        final AtomicBoolean callableRunning = new AtomicBoolean(false);
        class MyInjector
        extends EncryptionFaultInjector {
            private volatile int exceptionCount = 0;

            MyInjector(int numFailures) {
                this.exceptionCount = numFailures;
            }

            public synchronized void reencryptEncryptedKeys() throws IOException {
                if (this.exceptionCount > 0) {
                    --this.exceptionCount;
                    try {
                        callableRunning.set(true);
                        Thread.sleep(Long.MAX_VALUE);
                    }
                    catch (InterruptedException ie) {
                        LOG.info("Fault injector interrupted", (Throwable)ie);
                    }
                }
            }
        }
        MyInjector injector = new MyInjector(1);
        EncryptionFaultInjector.instance = injector;
        int len = 8196;
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path("/dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForQueuedZones(1);
        this.getEzManager().pauseReencryptUpdaterForTesting();
        this.getEzManager().resumeReencryptForTesting();
        LOG.info("Waiting for re-encrypt callables to run");
        GenericTestUtils.waitFor((Supplier)new Supplier<Boolean>(){

            @Override
            public Boolean get() {
                return callableRunning.get();
            }
        }, (long)100L, (long)10000L);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
        this.getEzManager().resumeReencryptUpdaterForTesting();
        this.waitForZoneCompletes(zone.toString());
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        Assert.assertTrue((boolean)zs.isCanceled());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)0L, (long)zs.getFilesReencrypted());
        Assert.assertTrue((boolean)this.getUpdater().isRunning());
    }

    @Test
    public void testCancelFutureThenReencrypt() throws Exception {
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.cancelFutureDuringReencryption(zone);
        this.getEzManager().resumeReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForZoneCompletes(zone.toString());
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        Assert.assertFalse((boolean)zs.isCanceled());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
    }

    @Test
    public void testCancelFutureThenRestart() throws Exception {
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.cancelFutureDuringReencryption(zone);
        this.restartClusterDisableReencrypt();
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        Assert.assertTrue((boolean)zs.isCanceled());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)0L, (long)zs.getFilesReencrypted());
        this.getEzManager().resumeReencryptForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForZoneCompletes(zone.toString());
        it = this.dfsAdmin.listReencryptionStatus();
        zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        Assert.assertFalse((boolean)zs.isCanceled());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
    }

    @Test
    public void testReencryptCancelForUpdater() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        Path subdir = new Path("/dir");
        this.fsWrapper.mkdir(subdir, FsPermission.getDirDefault(), true);
        DFSTestUtil.createFile((FileSystem)this.fs, new Path(subdir, "f"), 8196L, (short)1, 65261L);
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseReencryptUpdaterForTesting();
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        Thread.sleep(3000L);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
        this.getEzManager().resumeReencryptUpdaterForTesting();
        this.waitForZoneCompletes(zone.toString());
        Thread.sleep(3000L);
        Assert.assertEquals((long)0L, (long)this.getZoneStatus(zone.toString()).getFilesReencrypted());
    }

    @Test
    public void testReencryptionWithoutProvider() throws Exception {
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        this.cluster.getConfiguration(0).unset("hadoop.security.key.provider.path");
        this.cluster.restartNameNodes();
        this.cluster.waitClusterUp();
        try {
            this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
            Assert.fail((String)"should not be able to re-encrypt");
        }
        catch (RemoteException expected) {
            GenericTestUtils.assertExceptionContains((String)"rejected", (Throwable)expected.unwrapRemoteException());
        }
        try {
            this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.CANCEL);
            Assert.fail((String)"should not be able to cancel re-encrypt");
        }
        catch (RemoteException expected) {
            GenericTestUtils.assertExceptionContains((String)"rejected", (Throwable)expected.unwrapRemoteException());
        }
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
    }

    @Test
    public void testReencryptionNNSafeMode() throws Exception {
        int i;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.getEzManager().pauseForTestingAfterNthSubmission(1);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedFiles(zone.toString(), 5);
        this.fs.setSafeMode(SafeModeAction.ENTER);
        this.getEzManager().resumeReencryptForTesting();
        for (i = 0; i < 3; ++i) {
            Thread.sleep(1000L);
            RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
            Assert.assertTrue((boolean)it.hasNext());
            ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
            Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
            Assert.assertEquals((long)0L, (long)zs.getCompletionTime());
            Assert.assertEquals((long)5L, (long)zs.getFilesReencrypted());
        }
        this.fs.setSafeMode(SafeModeAction.LEAVE);
        this.getHandler().notifyNewSubmission();
        this.waitForReencryptedFiles(zone.toString(), 10);
    }

    @Test
    public void testReencryptionKMSDown() throws Exception {
        class MyInjector
        extends EncryptionFaultInjector {
            private volatile int exceptionCount = 0;

            MyInjector(int numFailures) {
                this.exceptionCount = numFailures;
            }

            public synchronized void reencryptEncryptedKeys() throws IOException {
                if (this.exceptionCount > 0) {
                    --this.exceptionCount;
                    throw new IOException("Injected KMS failure");
                }
            }
        }
        MyInjector injector = new MyInjector(1);
        EncryptionFaultInjector.instance = injector;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        Assert.assertEquals((long)0L, (long)injector.exceptionCount);
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)5L, (long)zs.getFilesReencrypted());
        Assert.assertEquals((long)5L, (long)zs.getNumReencryptionFailures());
    }

    @Test
    public void testReencryptionUpdaterFaultOneTask() throws Exception {
        class MyInjector
        extends EncryptionFaultInjector {
            private volatile int exceptionCount = 0;

            MyInjector(int numFailures) {
                this.exceptionCount = numFailures;
            }

            public synchronized void reencryptUpdaterProcessOneTask() throws IOException {
                if (this.exceptionCount > 0) {
                    --this.exceptionCount;
                    throw new IOException("Injected process task failure");
                }
            }
        }
        MyInjector injector = new MyInjector(1);
        EncryptionFaultInjector.instance = injector;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        Assert.assertEquals((long)0L, (long)injector.exceptionCount);
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)5L, (long)zs.getFilesReencrypted());
        Assert.assertEquals((long)1L, (long)zs.getNumReencryptionFailures());
    }

    @Test
    public void testReencryptionUpdaterFaultCkpt() throws Exception {
        class MyInjector
        extends EncryptionFaultInjector {
            private volatile int exceptionCount = 0;

            MyInjector(int numFailures) {
                this.exceptionCount = numFailures;
            }

            public synchronized void reencryptUpdaterProcessCheckpoint() throws IOException {
                if (this.exceptionCount > 0) {
                    --this.exceptionCount;
                    throw new IOException("Injected process checkpoint failure");
                }
            }
        }
        MyInjector injector = new MyInjector(1);
        EncryptionFaultInjector.instance = injector;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        Assert.assertEquals((long)0L, (long)injector.exceptionCount);
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
        Assert.assertEquals((long)1L, (long)zs.getNumReencryptionFailures());
    }

    @Test
    public void testReencryptionUpdaterFaultRecover() throws Exception {
        class MyInjector
        extends EncryptionFaultInjector {
            private volatile int exceptionCount = 0;

            MyInjector(int oneTask) {
                this.exceptionCount = oneTask;
            }

            public synchronized void reencryptUpdaterProcessOneTask() throws IOException {
                if (this.exceptionCount > 0) {
                    --this.exceptionCount;
                    throw new RetriableException("Injected process task failure");
                }
            }
        }
        MyInjector injector = new MyInjector(10);
        EncryptionFaultInjector.instance = injector;
        int len = 8196;
        Path zoneParent = new Path("/zones");
        Path zone = new Path(zoneParent, "zone");
        this.fsWrapper.mkdir(zone, FsPermission.getDirDefault(), true);
        this.dfsAdmin.createEncryptionZone(zone, TEST_KEY, NO_TRASH);
        for (int i = 0; i < 10; ++i) {
            DFSTestUtil.createFile((FileSystem)this.fs, new Path(zone, Integer.toString(i)), 8196L, (short)1, 65261L);
        }
        this.rollKey(TEST_KEY);
        Whitebox.setInternalState((Object)this.getUpdater(), (String)"faultRetryInterval", (Object)50);
        this.dfsAdmin.reencryptEncryptionZone(zone, HdfsConstants.ReencryptAction.START);
        this.waitForReencryptedZones(1);
        Assert.assertEquals((long)0L, (long)injector.exceptionCount);
        RemoteIterator it = this.dfsAdmin.listReencryptionStatus();
        Assert.assertTrue((boolean)it.hasNext());
        ZoneReencryptionStatus zs = (ZoneReencryptionStatus)it.next();
        Assert.assertEquals((Object)zone.toString(), (Object)zs.getZoneName());
        Assert.assertEquals((Object)ZoneReencryptionStatus.State.Completed, (Object)zs.getState());
        this.verifyZoneCompletionTime(zs);
        Assert.assertEquals((long)10L, (long)zs.getFilesReencrypted());
        Assert.assertEquals((long)0L, (long)zs.getNumReencryptionFailures());
    }

    private ReencryptionHandler getHandler() {
        return (ReencryptionHandler)Whitebox.getInternalState((Object)this.getEzManager(), (String)"reencryptionHandler");
    }

    private ReencryptionUpdater getUpdater() {
        return (ReencryptionUpdater)Whitebox.getInternalState((Object)this.getHandler(), (String)"reencryptionUpdater");
    }

    protected void rollKey(String keyName) throws Exception {
        this.dfsAdmin.getKeyProvider().rollNewVersion(keyName);
        this.dfsAdmin.getKeyProvider().flush();
    }
}

