/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.health;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.TimerTask;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.util.Shell;
import org.apache.hadoop.util.StringUtils;
import org.apache.hadoop.yarn.server.nodemanager.health.TimedHealthReporterService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NodeHealthScriptRunner
extends TimedHealthReporterService {
    private static final Logger LOG = LoggerFactory.getLogger(NodeHealthScriptRunner.class);
    private String nodeHealthScript;
    private long scriptTimeout;
    private Shell.ShellCommandExecutor commandExecutor = null;
    private static final String ERROR_PATTERN = "ERROR";
    static final String NODE_HEALTH_SCRIPT_TIMED_OUT_MSG = "Node health script timed out";

    private NodeHealthScriptRunner(String scriptName, long checkInterval, long timeout, String[] scriptArgs, boolean runBeforeStartup) {
        super(NodeHealthScriptRunner.class.getName(), checkInterval, runBeforeStartup);
        this.nodeHealthScript = scriptName;
        this.scriptTimeout = timeout;
        this.setTimerTask(new NodeHealthMonitorExecutor(scriptArgs));
    }

    public static NodeHealthScriptRunner newInstance(String scriptName, Configuration conf) {
        String nodeHealthScriptsConfig = String.format("yarn.nodemanager.health-checker.%s.path", scriptName);
        String nodeHealthScript = conf.get(nodeHealthScriptsConfig);
        if (!NodeHealthScriptRunner.shouldRun(scriptName, nodeHealthScript)) {
            return null;
        }
        String checkIntervalMsConfig = String.format("yarn.nodemanager.health-checker.%s.interval-ms", scriptName);
        long checkIntervalMs = conf.getLong(checkIntervalMsConfig, 0L);
        if (checkIntervalMs == 0L) {
            checkIntervalMs = conf.getLong("yarn.nodemanager.health-checker.interval-ms", 600000L);
        }
        if (checkIntervalMs < 0L) {
            throw new IllegalArgumentException("The node health-checker's interval-ms can not be set to a negative number.");
        }
        boolean runBeforeStartup = conf.getBoolean("yarn.nodemanager.health-checker.run-before-startup", false);
        String scriptTimeoutConfig = String.format("yarn.nodemanager.health-checker.%s.timeout-ms", scriptName);
        long scriptTimeout = conf.getLong(scriptTimeoutConfig, 0L);
        if (scriptTimeout == 0L) {
            scriptTimeout = conf.getLong("yarn.nodemanager.health-checker.timeout-ms", 1200000L);
        }
        if (scriptTimeout <= 0L) {
            throw new IllegalArgumentException("The node health-checker's timeout can only be set to a positive number.");
        }
        String scriptArgsConfig = String.format("yarn.nodemanager.health-checker.%s.opts", scriptName);
        String[] scriptArgs = conf.getStrings(scriptArgsConfig, new String[0]);
        return new NodeHealthScriptRunner(nodeHealthScript, checkIntervalMs, scriptTimeout, scriptArgs, runBeforeStartup);
    }

    @Override
    public void serviceStop() throws Exception {
        Process p;
        if (this.commandExecutor != null && (p = this.commandExecutor.getProcess()) != null) {
            p.destroy();
        }
        super.serviceStop();
    }

    static boolean shouldRun(String script, String healthScript) {
        if (healthScript == null || healthScript.trim().isEmpty()) {
            LOG.info("Missing location for the node health check script \"{}\".", (Object)script);
            return false;
        }
        File f = new File(healthScript);
        if (!f.exists()) {
            LOG.warn("File {} for script \"{}\" does not exist.", (Object)healthScript, (Object)script);
            return false;
        }
        if (!FileUtil.canExecute((File)f)) {
            LOG.warn("File {} for script \"{}\" can not be executed.", (Object)healthScript, (Object)script);
            return false;
        }
        return true;
    }

    private class NodeHealthMonitorExecutor
    extends TimerTask {
        private String exceptionStackTrace = "";

        NodeHealthMonitorExecutor(String[] args) {
            ArrayList<String> execScript = new ArrayList<String>();
            execScript.add(NodeHealthScriptRunner.this.nodeHealthScript);
            if (args != null) {
                execScript.addAll(Arrays.asList(args));
            }
            NodeHealthScriptRunner.this.commandExecutor = new Shell.ShellCommandExecutor(execScript.toArray(new String[execScript.size()]), null, null, NodeHealthScriptRunner.this.scriptTimeout);
        }

        @Override
        public void run() {
            HealthCheckerExitStatus status = HealthCheckerExitStatus.SUCCESS;
            try {
                NodeHealthScriptRunner.this.commandExecutor.execute();
            }
            catch (Shell.ExitCodeException e) {
                status = HealthCheckerExitStatus.FAILED_WITH_EXIT_CODE;
                if (Shell.WINDOWS && NodeHealthScriptRunner.this.commandExecutor.isTimedOut()) {
                    status = HealthCheckerExitStatus.TIMED_OUT;
                }
            }
            catch (Exception e) {
                LOG.warn("Caught exception : " + e.getMessage());
                status = !NodeHealthScriptRunner.this.commandExecutor.isTimedOut() ? HealthCheckerExitStatus.FAILED_WITH_EXCEPTION : HealthCheckerExitStatus.TIMED_OUT;
                this.exceptionStackTrace = StringUtils.stringifyException((Throwable)e);
            }
            finally {
                if (status == HealthCheckerExitStatus.SUCCESS && this.hasErrors(NodeHealthScriptRunner.this.commandExecutor.getOutput())) {
                    status = HealthCheckerExitStatus.FAILED;
                }
                this.reportHealthStatus(status);
            }
        }

        void reportHealthStatus(HealthCheckerExitStatus status) {
            switch (status) {
                case SUCCESS: 
                case FAILED_WITH_EXIT_CODE: {
                    NodeHealthScriptRunner.this.setHealthyWithoutReport();
                    break;
                }
                case TIMED_OUT: {
                    NodeHealthScriptRunner.this.setUnhealthyWithReport(NodeHealthScriptRunner.NODE_HEALTH_SCRIPT_TIMED_OUT_MSG);
                    break;
                }
                case FAILED_WITH_EXCEPTION: {
                    NodeHealthScriptRunner.this.setUnhealthyWithReport(this.exceptionStackTrace);
                    break;
                }
                case FAILED: {
                    NodeHealthScriptRunner.this.setUnhealthyWithReport(NodeHealthScriptRunner.this.commandExecutor.getOutput());
                    break;
                }
                default: {
                    LOG.warn("Unknown HealthCheckerExitStatus - ignored.");
                }
            }
        }

        private boolean hasErrors(String output) {
            String[] splits;
            for (String split : splits = output.split("\n")) {
                if (!split.startsWith(NodeHealthScriptRunner.ERROR_PATTERN)) continue;
                return true;
            }
            return false;
        }
    }

    private static enum HealthCheckerExitStatus {
        SUCCESS,
        TIMED_OUT,
        FAILED_WITH_EXIT_CODE,
        FAILED_WITH_EXCEPTION,
        FAILED;

    }
}

