/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.zookeeper;

import java.io.Closeable;
import java.io.IOException;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.DelayQueue;
import java.util.concurrent.Delayed;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.DoNotRetryIOException;
import org.apache.hadoop.hbase.util.FutureUtils;
import org.apache.hadoop.hbase.util.Threads;
import org.apache.hadoop.hbase.zookeeper.ZKConfig;
import org.apache.hbase.thirdparty.io.netty.util.HashedWheelTimer;
import org.apache.hbase.thirdparty.io.netty.util.TimerTask;
import org.apache.yetus.audience.InterfaceAudience;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.client.ZKClientConfig;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public final class ReadOnlyZKClient
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(ReadOnlyZKClient.class);
    public static final String RECOVERY_RETRY = "zookeeper.recovery.retry";
    private static final int DEFAULT_RECOVERY_RETRY = 30;
    public static final String RECOVERY_RETRY_INTERVAL_MILLIS = "zookeeper.recovery.retry.intervalmill";
    private static final int DEFAULT_RECOVERY_RETRY_INTERVAL_MILLIS = 1000;
    public static final String KEEPALIVE_MILLIS = "zookeeper.keep-alive.time";
    private static final int DEFAULT_KEEPALIVE_MILLIS = 60000;
    private static final EnumSet<KeeperException.Code> FAIL_FAST_CODES = EnumSet.of(KeeperException.Code.NOAUTH, KeeperException.Code.AUTHFAILED);
    private final String connectString;
    private final int sessionTimeoutMs;
    private final int maxRetries;
    private final int retryIntervalMs;
    private final int keepAliveTimeMs;
    private HashedWheelTimer retryTimer;
    private final ZKClientConfig zkClientConfig;
    private static final Task CLOSE = new Task(){};
    private final DelayQueue<Task> tasks = new DelayQueue();
    private final AtomicBoolean closed = new AtomicBoolean(false);
    ZooKeeper zookeeper;
    private int pendingRequests = 0;

    private String getId() {
        return String.format("0x%08x", System.identityHashCode(this));
    }

    public ReadOnlyZKClient(Configuration conf, HashedWheelTimer retryTimer) {
        String clientZkQuorumServers = ZKConfig.getClientZKQuorumServersString((Configuration)conf);
        this.connectString = clientZkQuorumServers != null ? clientZkQuorumServers : ZKConfig.getZKQuorumServersString((Configuration)conf);
        this.sessionTimeoutMs = conf.getInt("zookeeper.session.timeout", 90000);
        this.maxRetries = conf.getInt(RECOVERY_RETRY, 30);
        this.retryIntervalMs = conf.getInt(RECOVERY_RETRY_INTERVAL_MILLIS, 1000);
        this.keepAliveTimeMs = conf.getInt(KEEPALIVE_MILLIS, 60000);
        this.retryTimer = retryTimer;
        this.zkClientConfig = ZKConfig.getZKClientConfig((Configuration)conf);
        LOG.debug("Connect {} to {} with session timeout={}ms, retries={}, retry interval={}ms, keepAlive={}ms, zk client config={}", new Object[]{this.getId(), this.connectString, this.sessionTimeoutMs, this.maxRetries, this.retryIntervalMs, this.keepAliveTimeMs, this.zkClientConfig});
        Threads.setDaemonThreadRunning((Thread)new Thread(this::run), (String)("ReadOnlyZKClient-" + this.connectString + "@" + this.getId()));
    }

    private static TimerTask getTimerTask(long timeoutMs, CompletableFuture<?> future, String api) {
        return timeout -> {
            if (!future.isDone()) {
                future.completeExceptionally((Throwable)((Object)new DoNotRetryIOException("Zookeeper " + api + " could not be completed in " + timeoutMs + " ms")));
            }
        };
    }

    public CompletableFuture<byte[]> get(String path, long timeoutMs) {
        CompletableFuture<byte[]> future = this.get(path);
        TimerTask timerTask = ReadOnlyZKClient.getTimerTask(timeoutMs, future, "GET");
        this.retryTimer.newTimeout(timerTask, timeoutMs + 1L, TimeUnit.MILLISECONDS);
        return future;
    }

    public CompletableFuture<byte[]> get(String path) {
        if (this.closed.get()) {
            return FutureUtils.failedFuture((Throwable)((Object)new DoNotRetryIOException("Client already closed")));
        }
        CompletableFuture<byte[]> future = new CompletableFuture<byte[]>();
        this.tasks.add(new ZKTask<byte[]>(path, future, "get"){

            @Override
            protected void doExec(ZooKeeper zk) {
                zk.getData(this.path, false, (rc, path, ctx, data, stat) -> this.onComplete(zk, rc, data, true), null);
            }
        });
        return future;
    }

    public CompletableFuture<Stat> exists(String path, long timeoutMs) {
        CompletableFuture<Stat> future = this.exists(path);
        TimerTask timerTask = ReadOnlyZKClient.getTimerTask(timeoutMs, future, "EXISTS");
        this.retryTimer.newTimeout(timerTask, timeoutMs + 1L, TimeUnit.MILLISECONDS);
        return future;
    }

    public CompletableFuture<Stat> exists(String path) {
        if (this.closed.get()) {
            return FutureUtils.failedFuture((Throwable)((Object)new DoNotRetryIOException("Client already closed")));
        }
        CompletableFuture<Stat> future = new CompletableFuture<Stat>();
        this.tasks.add(new ZKTask<Stat>(path, future, "exists"){

            @Override
            protected void doExec(ZooKeeper zk) {
                zk.exists(this.path, false, (rc, path, ctx, stat) -> this.onComplete(zk, rc, stat, false), null);
            }
        });
        return future;
    }

    public CompletableFuture<List<String>> list(String path, long timeoutMs) {
        CompletableFuture<List<String>> future = this.list(path);
        TimerTask timerTask = ReadOnlyZKClient.getTimerTask(timeoutMs, future, "LIST");
        this.retryTimer.newTimeout(timerTask, timeoutMs + 1L, TimeUnit.MILLISECONDS);
        return future;
    }

    public CompletableFuture<List<String>> list(String path) {
        if (this.closed.get()) {
            return FutureUtils.failedFuture((Throwable)((Object)new DoNotRetryIOException("Client already closed")));
        }
        CompletableFuture<List<String>> future = new CompletableFuture<List<String>>();
        this.tasks.add(new ZKTask<List<String>>(path, future, "list"){

            @Override
            protected void doExec(ZooKeeper zk) {
                zk.getChildren(this.path, false, (rc, path, ctx, children) -> this.onComplete(zk, rc, children, true), null);
            }
        });
        return future;
    }

    private void closeZk() {
        if (this.zookeeper != null) {
            try {
                this.zookeeper.close();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            this.zookeeper = null;
        }
    }

    private ZooKeeper getZk() throws IOException {
        if (this.zookeeper == null || !this.zookeeper.getState().isAlive()) {
            this.zookeeper = new ZooKeeper(this.connectString, this.sessionTimeoutMs, e -> {}, this.zkClientConfig);
        }
        return this.zookeeper;
    }

    private void run() {
        while (true) {
            ZooKeeper zk;
            Task task;
            try {
                task = (Task)this.tasks.poll(this.keepAliveTimeMs, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                continue;
            }
            if (task == CLOSE) break;
            if (task == null) {
                if (this.pendingRequests != 0) continue;
                LOG.trace("{} to {} inactive for {}ms; closing (Will reconnect when new requests)", new Object[]{this.getId(), this.connectString, this.keepAliveTimeMs});
                this.closeZk();
                continue;
            }
            if (!task.needZk()) {
                task.exec(null);
                continue;
            }
            try {
                zk = this.getZk();
            }
            catch (Exception e) {
                task.connectFailed(e);
                continue;
            }
            task.exec(zk);
        }
        this.closeZk();
        DoNotRetryIOException error = new DoNotRetryIOException("Client already closed");
        Arrays.stream(this.tasks.toArray(new Task[0])).forEach(t -> t.closed((IOException)((Object)error)));
        this.tasks.clear();
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            LOG.debug("Close zookeeper connection {} to {}", (Object)this.getId(), (Object)this.connectString);
            this.tasks.add(CLOSE);
        }
    }

    public String getConnectString() {
        return this.connectString;
    }

    private abstract class ZKTask<T>
    extends Task {
        protected final String path;
        private final CompletableFuture<T> future;
        private final String operationType;
        private int retries;

        protected ZKTask(String path, CompletableFuture<T> future, String operationType) {
            this.path = path;
            this.future = future;
            this.operationType = operationType;
        }

        protected final void onComplete(final ZooKeeper zk, final int rc, final T ret, final boolean errorIfNoNode) {
            ReadOnlyZKClient.this.tasks.add(new Task(){

                @Override
                public void exec(ZooKeeper alwaysNull) {
                    ReadOnlyZKClient.this.pendingRequests--;
                    KeeperException.Code code = KeeperException.Code.get((int)rc);
                    if (code == KeeperException.Code.OK) {
                        ZKTask.this.future.complete(ret);
                    } else if (code == KeeperException.Code.NONODE) {
                        if (errorIfNoNode) {
                            ZKTask.this.future.completeExceptionally(KeeperException.create((KeeperException.Code)code, (String)ZKTask.this.path));
                        } else {
                            ZKTask.this.future.complete(ret);
                        }
                    } else if (FAIL_FAST_CODES.contains(code)) {
                        ZKTask.this.future.completeExceptionally(KeeperException.create((KeeperException.Code)code, (String)ZKTask.this.path));
                    } else {
                        if (code == KeeperException.Code.SESSIONEXPIRED) {
                            LOG.warn("{} to {} session expired, close and reconnect", (Object)ReadOnlyZKClient.this.getId(), (Object)ReadOnlyZKClient.this.connectString);
                            try {
                                zk.close();
                            }
                            catch (InterruptedException e) {
                                Thread.currentThread().interrupt();
                            }
                        }
                        if (ZKTask.this.delay(ReadOnlyZKClient.this.retryIntervalMs, ReadOnlyZKClient.this.maxRetries)) {
                            LOG.warn("{} to {} failed for {} of {}, code = {}, retries = {}", new Object[]{ReadOnlyZKClient.this.getId(), ReadOnlyZKClient.this.connectString, ZKTask.this.operationType, ZKTask.this.path, code, ZKTask.this.retries});
                            ReadOnlyZKClient.this.tasks.add(ZKTask.this);
                        } else {
                            LOG.warn("{} to {} failed for {} of {}, code = {}, retries = {}, give up", new Object[]{ReadOnlyZKClient.this.getId(), ReadOnlyZKClient.this.connectString, ZKTask.this.operationType, ZKTask.this.path, code, ZKTask.this.retries});
                            ZKTask.this.future.completeExceptionally(KeeperException.create((KeeperException.Code)code, (String)ZKTask.this.path));
                        }
                    }
                }

                @Override
                public void closed(IOException e) {
                    ZKTask.this.future.completeExceptionally(e);
                }
            });
        }

        @Override
        public boolean needZk() {
            return true;
        }

        protected abstract void doExec(ZooKeeper var1);

        @Override
        public final void exec(ZooKeeper zk) {
            ReadOnlyZKClient.this.pendingRequests++;
            this.doExec(zk);
        }

        public boolean delay(long intervalMs, int maxRetries) {
            if (this.retries >= maxRetries) {
                return false;
            }
            ++this.retries;
            this.time = System.nanoTime() + TimeUnit.MILLISECONDS.toNanos(intervalMs);
            return true;
        }

        @Override
        public void connectFailed(Exception e) {
            if (this.delay(ReadOnlyZKClient.this.retryIntervalMs, ReadOnlyZKClient.this.maxRetries)) {
                LOG.warn("{} to {} failed to connect to zk fo {} of {}, retries = {}", new Object[]{ReadOnlyZKClient.this.getId(), ReadOnlyZKClient.this.connectString, this.operationType, this.path, this.retries, e});
                ReadOnlyZKClient.this.tasks.add(this);
            } else {
                LOG.warn("{} to {} failed to connect to zk fo {} of {}, retries = {}, give up", new Object[]{ReadOnlyZKClient.this.getId(), ReadOnlyZKClient.this.connectString, this.operationType, this.path, this.retries, e});
                this.future.completeExceptionally(e);
            }
        }

        @Override
        public void closed(IOException e) {
            this.future.completeExceptionally(e);
        }
    }

    private static abstract class Task
    implements Delayed {
        protected long time = System.nanoTime();

        private Task() {
        }

        public boolean needZk() {
            return false;
        }

        public void exec(ZooKeeper zk) {
        }

        public void connectFailed(Exception e) {
        }

        public void closed(IOException e) {
        }

        @Override
        public int compareTo(Delayed o) {
            Task that = (Task)o;
            int c = Long.compare(this.time, that.time);
            if (c != 0) {
                return c;
            }
            return Integer.compare(System.identityHashCode(this), System.identityHashCode(that));
        }

        @Override
        public long getDelay(TimeUnit unit) {
            return unit.convert(this.time - System.nanoTime(), TimeUnit.NANOSECONDS);
        }
    }
}

